/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006
 */
package org.jboss.soa.esb.actions;

import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.message.ActionProcessingPipeline;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;

/**
 * Simple action that passes the message to a logging mechanism.
 * This action is primarily used for testing the quickstarts.
 * 
 * @author Kevin Conner
 */
public class TestMessageStore extends AbstractActionPipelineProcessor
{
    /**
     * The message content location.
     */
    public static final String LOCATION = "location" ;
    /**
     * The log success flag.
     */
    public static final String LOG_SUCCESS = "logSuccess" ;
    /**
     * The log exception flag.
     */
    public static final String LOG_EXCEPTION = "logException" ;
    
    /**
     * The global test message store sink.
     */
    private static TestMessageStoreSink testMessageStoreSink ;

    /**
     * The message payload proxy.
     */
    private final MessagePayloadProxy payloadProxy ;
    /**
     * Should we log the success notification?
     */
    private final boolean logSuccess ;
    /**
     * Should we log the exception notification?
     */
    private final boolean logException ;

    /**
     * Create the action with the specified configuration.
     * 
     * @param config The action configuration.
     */
    public TestMessageStore(final ConfigTree config)
    {
        String location = config.getAttribute(LOCATION) ;
        if(location != null)
        {
            config.setAttribute(MessagePayloadProxy.GET_PAYLOAD_LOCATION, location) ;
        }
        payloadProxy = new MessagePayloadProxy(config) ;
        payloadProxy.setNullGetPayloadHandling(MessagePayloadProxy.NullPayloadHandling.LOG) ;
        logSuccess = config.getBooleanAttribute(LOG_SUCCESS, false) ;
        logException = config.getBooleanAttribute(LOG_EXCEPTION, false) ;
    }

    /**
     * Process the message.
     * 
     * @param message The current message.
     * @throws ActionProcessingPipeline for errors in processing.
     */
    public Message process(final Message message)
        throws ActionProcessingException
    {
        logContents(getContents(message)) ;
        return message ;
    }
    
    /**
     * Process the exception notification.
     * @param message The associated message.
     * @param th The throwable causing the exception.
     */
    public void processException(final Message message, final Throwable th)
    {
        if (logException)
        {
            try
            {
                logContents(th.getLocalizedMessage() + ':' + getContents(message)) ;
            }
            catch (final ActionProcessingException ape)
            {
                logContents(ape.getLocalizedMessage()) ;
            }
        }
    }
    
    /**
     * Process the success notification.
     * @param message The associated message.
     */
    public void processSuccess(final Message message)
    {
        if (logSuccess)
        {
            try
            {
                logContents("Success:" + getContents(message)) ;
            }
            catch (final ActionProcessingException ape)
            {
                logContents(ape.getLocalizedMessage()) ;
            }
        }
    }
    
    /**
     * Get the message contents.
     * @param message The message.
     * @return the message contents as a string or null if no contents.
     * @throws ActionProcessingException for errors in processing.
     */
    private String getContents(final Message message)
        throws ActionProcessingException
    {
        final Object contents;

        try
        {
            contents = payloadProxy.getPayload(message) ;
        }
        catch (final MessageDeliverException mde)
        {
            throw new ActionProcessingException(mde) ;
        }

        if (contents != null)
        {
            final String messageVal ;
            if (contents instanceof byte[])
            {
                messageVal = new String((byte[])contents) ;
            }
            else
            {
                messageVal = contents.toString() ;
            }
            return messageVal ;
        }
        return null ;
    }
    
    /**
     * Set the message store sink.
     * @param testMessageStoreSink The test message store sink.
     */
    public static synchronized void setMessageStoreSink(final TestMessageStoreSink testMessageStoreSink)
    {
        TestMessageStore.testMessageStoreSink = testMessageStoreSink ;
    }
    
    /**
     * Log the contents to the message store sink.
     * @param message The current message. 
     */
    private static synchronized void logContents(final String message)
    {
        if (testMessageStoreSink != null)
        {
            testMessageStoreSink.logMessage(message) ;
        }
    }
}
