/*
 * JBoss, Home of Professional Open Source
 * Copyright 2012, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2012,
 * @author JBoss Inc.
 */
package org.jboss.soa.esb.actions.aggregator;

import java.util.HashMap ;
import java.util.Map ;
import java.util.concurrent.locks.Lock ;
import java.util.concurrent.locks.ReentrantLock ;

import org.apache.log4j.Logger ;
import org.jboss.soa.esb.ConfigurationException ;
import org.jboss.soa.esb.actions.AbstractActionPipelineProcessor ;
import org.jboss.soa.esb.actions.ActionLifecycleException ;
import org.jboss.soa.esb.actions.ActionProcessingException ;
import org.jboss.soa.esb.helpers.ConfigTree ;
import org.jboss.soa.esb.message.Message ;
import org.jboss.soa.esb.message.format.MessageFactory ;

/**
 * Abstract action class which defines how aggregation is processed but 
 * leaves abstract how messages are stored.
 * 
 * @author Kevin Conner
 */
public abstract class AbstractAggregator extends AbstractActionPipelineProcessor
{
    private static final Logger LOGGER = Logger.getLogger(AbstractAggregator.class) ;
    
    private static final long DEFAULT_TIMEOUT = 600000L ;

    private final long timeout ;
    
    private TimeoutProcessor timeoutProcessor ;
    
    private AggregatorEntry head ;
    private Map<String, AggregatorEntry> seriesUUIDtoEntry ;

    protected AbstractAggregator(final ConfigTree config)
        throws ConfigurationException
    {
        final String timeoutValue = config.getAttribute("timeoutInMillis") ;
        if (timeoutValue != null)
        {
            long parsedTimeout ;
            try
            {
                parsedTimeout = Long.valueOf(timeoutValue) ;
            }
            catch (final NumberFormatException nfe)
            {
                parsedTimeout = DEFAULT_TIMEOUT ;
                throw new ConfigurationException("Failed to parse timeout " + timeoutValue) ;
            }
            timeout = parsedTimeout ;
        }
        else
        {
            timeout = DEFAULT_TIMEOUT ;
        }
    }
    
    @Override
    public void initialise() throws ActionLifecycleException
    {
        head = new AggregatorEntry(null) ;
        head.setNextEntry(head) ;
        head.setPreviousEntry(head) ;
        
        seriesUUIDtoEntry = new HashMap<String, AggregatorEntry>() ;
        
        timeoutProcessor = new TimeoutProcessor() ;
        timeoutProcessor.start() ;
    }

    @Override
    public void destroy() throws ActionLifecycleException
    {
        timeoutProcessor.stop() ;
        timeoutProcessor = null ;
    }
    
    @Override
    public Message process(Message message)
        throws ActionProcessingException
    {
        final Object detailObject = message.getProperties().getProperty(AggregateDetails.AGGREGATE_DETAILS) ;
        if ((detailObject == null) || !(detailObject instanceof AggregateDetails))
        {
            return message ;
        }
        
        final AggregateDetails aggregateDetails = (AggregateDetails)detailObject ;
        
        final String seriesUUID = aggregateDetails.getSeriesUUID() ;
        
        if (LOGGER.isDebugEnabled())
        {
            LOGGER.debug("Aggregating message into " + seriesUUID) ;
        }
        
        AggregatorEntry entry = retrieveEntry(seriesUUID) ;
        try
        {
            Message entryMessage = entry.getMessage() ;
            if (entryMessage == null)
            {
                entryMessage = MessageFactory.getInstance().getMessage() ;
                entryMessage.getProperties().setProperty(AggregateDetails.SERIES_UUID, seriesUUID) ;
                entry.setMessage(entryMessage) ;
            }
            final Integer messageSequence = aggregateDetails.getMessageSequence() ;
            if (aggregateMessage(entryMessage, messageSequence, message))
            {
                final int aggregateCount = entry.incAggregateCount() ;
                
                final Integer sequenceCount = aggregateDetails.getSequenceCount() ;
                if (sequenceCount != null)
                {
                    entry.setSequenceCount(sequenceCount.intValue()) ;
                    entryMessage.getProperties().setProperty(AggregateDetails.SEQUENCE_COUNT, sequenceCount) ;
                }
                final int entrySequenceCount = entry.getSequenceCount() ;
                if ((entrySequenceCount == 0) || (aggregateCount < entrySequenceCount))
                {
                    entryMessage = null ;
                }
                else
                {
                    if (LOGGER.isDebugEnabled())
                    {
                        LOGGER.debug("Aggregated message for " + seriesUUID + ", aggregateCount " + aggregateCount) ;
                    }
                    entry.setMessage(null) ;
                }
            }
            return entryMessage ;
        }
        finally
        {
            releaseEntry(entry) ;
        }
    }
    
    private AggregatorEntry retrieveEntry(final String seriesUUID)
    {
        if (LOGGER.isDebugEnabled())
        {
            LOGGER.debug("Retrieving entry for " + seriesUUID) ;
        }
        while(true)
        {
            AggregatorEntry entry ;
            synchronized(head)
            {
                entry = seriesUUIDtoEntry.get(seriesUUID) ;
                if (entry == null)
                {
                    entry = new AggregatorEntry(seriesUUID) ;
                    seriesUUIDtoEntry.put(seriesUUID, entry) ;
                    final AggregatorEntry previousEntry = head.getPreviousEntry() ;
                    entry.setNextEntry(head) ;
                    head.setPreviousEntry(entry) ;
                    entry.setPreviousEntry(previousEntry) ;
                    previousEntry.setNextEntry(entry) ;
                }
            }
            if (!entry.hasExpired())
            {
                entry.lock() ;
                return entry ;
            }
        }
    }

    private void releaseEntry(final AggregatorEntry entry)
    {
        if (LOGGER.isDebugEnabled())
        {
            LOGGER.debug("Releasing entry for " + entry.getSeriesUUID()) ;
        }
        if (entry.getMessage() == null)
        {
            removeEntry(entry) ;
        }
        entry.unlock() ;
    }

    private void removeEntry(final AggregatorEntry entry)
    {
        if (LOGGER.isDebugEnabled())
        {
            LOGGER.debug("Removing entry for " + entry.getSeriesUUID()) ;
        }
        synchronized(head)
        {
            if (!entry.hasExpired())
            {
                seriesUUIDtoEntry.remove(entry.getSeriesUUID()) ;
                final AggregatorEntry nextEntry = entry.getNextEntry() ;
                final AggregatorEntry previousEntry = entry.getPreviousEntry() ;
                entry.setNextEntry(null) ;
                entry.setPreviousEntry(null) ;
                nextEntry.setPreviousEntry(previousEntry) ;
                previousEntry.setNextEntry(nextEntry) ;
            }
        }
    }
    
    protected abstract boolean aggregateMessage(final Message aggregatedMessage, final Integer messageSequence, final Message currentMessage)
        throws ActionProcessingException ;
    
    private final class AggregatorEntry
    {
        private final long expiry ;
        private final String seriesUUID ;
        private int sequenceCount ;
        private int aggregateCount ;
        private Message message ;

        private Lock lock = new ReentrantLock() ;
        private AggregatorEntry previousEntry ;
        private AggregatorEntry nextEntry ;
        
        AggregatorEntry(final String seriesUUID)
        {
            expiry = System.currentTimeMillis() + timeout ;
            this.seriesUUID = seriesUUID ;
        }

        long getExpiry()
        {
            return expiry ;
        }
        
        String getSeriesUUID()
        {
            return seriesUUID ;
        }
        
        void setSequenceCount(final int sequenceCount)
        {
            this.sequenceCount = sequenceCount ;
        }
        
        int getSequenceCount()
        {
            return sequenceCount ;
        }
        
        void setMessage(final Message message)
        {
            this.message = message ;
        }
        
        Message getMessage()
        {
            return message ;
        }
        
        int incAggregateCount()
        {
            return ++aggregateCount ;
        }
        
        void lock()
        {
            lock.lock() ;
        }
        
        void unlock()
        {
            lock.unlock() ;
        }
        
        void setPreviousEntry(final AggregatorEntry previousEntry)
        {
            this.previousEntry = previousEntry;
        }
        
        AggregatorEntry getPreviousEntry()
        {
            return previousEntry ;
        }
        
        void setNextEntry(final AggregatorEntry nextEntry)
        {
            this.nextEntry = nextEntry ;
        }
        
        AggregatorEntry getNextEntry()
        {
            return nextEntry ;
        }
        
        boolean hasExpired()
        {
            return nextEntry == null ;
        }
    }
    
    private final class TimeoutProcessor implements Runnable
    {
        private Thread thread ;
        private volatile boolean active ;
        
        @Override
        public void run()
        {
            LOGGER.debug("Starting timeout processor") ;
            synchronized(head)
            {
                while(active)
                {
                    final long now = System.currentTimeMillis() ;
                    
                    final AggregatorEntry firstEntry = head.getNextEntry()  ;
                    try
                    {
                        if (firstEntry == head)
                        {
                            if (LOGGER.isDebugEnabled())
                            {
                                LOGGER.debug("Empty aggregator list, waiting for " + timeout) ;
                            }
                            head.wait(timeout) ;
                        }
                        else if (firstEntry.getExpiry() > now)
                        {
                            final long waitPeriod = firstEntry.getExpiry() - now ;
                            if (LOGGER.isDebugEnabled())
                            {
                                LOGGER.debug("First entry not expired, waiting for " + waitPeriod) ;
                            }
                            head.wait(waitPeriod) ;
                        }
                        else
                        {
                            if (LOGGER.isDebugEnabled())
                            {
                                LOGGER.debug("Timing out entry for " + firstEntry.getSeriesUUID()) ;
                            }
                            removeEntry(firstEntry) ;
                        }
                    }
                    catch (final InterruptedException ie) {} // ignore
                }
            }
            LOGGER.debug("Stopping timeout processor") ;
        }
        
        public void start()
        {
            if (thread == null)
            {
                thread = new Thread(this) ;
                active = true ;
                thread.start();
            }
        }
        
        public void stop()
        {
            if (thread != null)
            {
                active = false ;
                synchronized(head)
                {
                    head.notify() ;
                }
                try
                {
                    thread.join() ;
                }
                catch (final InterruptedException ie)
                {
                    LOGGER.warn("Unexpected interruption while waiting for timeout processor to stop") ;
                }
                thread = null ;
            }
        }
    }
}