/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions.converters;

import java.io.BufferedInputStream;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Map;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.actions.AbstractActionPipelineProcessor;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.message.ActionProcessingConstants;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;

/**
 * Byte Order Mark (BOM) Filtering Action.
 * <p/>
 * Only works on {@link byte} arrays and {@link InputStream InputStreams}.
 * 
 * @author <a href="mailto:tom.fennelly@gmail.com">tom.fennelly@gmail.com</a>
 */
public class BOMFilter extends AbstractActionPipelineProcessor {

	private static final Logger logger = Logger.getLogger(BOMFilter.class);

    private MessagePayloadProxy payloadProxy;
	private final String contentEncoding;
	
	private static final Map<String, byte[]> bomMap;
	
	static {
		bomMap = new HashMap<String, byte[]>();
		bomMap.put("UTF8", new byte[] {(byte)0xEF, (byte)0xBB, (byte)0xBF});
		bomMap.put("UTF16BE", new byte[] {(byte)0xFE, (byte)0xFF});
		bomMap.put("UTF16LE", new byte[] {(byte)0xFF, (byte)0xFE});
		bomMap.put("UTF32BE", new byte[] {(byte)0x00, (byte)0x00, (byte)0xFE, (byte)0xFF});
		bomMap.put("UTF32LE", new byte[] {(byte)0xFF, (byte)0xFE, (byte)0x00, (byte)0x00});
	}

	/**
	 * Public constructor.
	 * @param config Config.
	 */
	public BOMFilter(ConfigTree config) {
		contentEncoding = config.getAttribute("contentEncoding");
		payloadProxy = new MessagePayloadProxy(config);
	}

	/* (non-Javadoc)
	 * @see org.jboss.soa.esb.actions.ActionPipelineProcessor#process(org.jboss.soa.esb.message.Message)
	 */
	public Message process(Message message) throws ActionProcessingException {
		final String bomEncoding;
		
		if (contentEncoding != null) {
			bomEncoding = contentEncoding;
		} else {
			bomEncoding = (String) message.getProperties().getProperty(ActionProcessingConstants.PROPERTY_PAYLOAD_CONTENT_ENCODING);
		}
		
		if (bomEncoding != null) {
			Object payload;
			
			try {
				payload = payloadProxy.getPayload(message);
			} catch (MessageDeliverException e) {
				throw new ActionProcessingException("Error getting payload from Message instance.", e);
			}

			try {
				if (payload instanceof byte[]) {
					payloadProxy.setPayload(message, filterBOM((byte[])payload, bomEncoding));
				} else if (payload instanceof InputStream) {
					payloadProxy.setPayload(message, filterBOM((InputStream)payload, bomEncoding));
				} else {
					logger.debug("BOMFilter not applied to message.  Only supports byte[] and InputStream payload types.");
				}
			} catch (MessageDeliverException e) {
				throw new ActionProcessingException("Error setting payload on Message instance.", e);
			}
		} else {
			logger.debug("BOMFilter not applied to message.  No 'contentEncoding' was specified on the Action configuration, or on the incoming message properties (\"" + ActionProcessingConstants.PROPERTY_PAYLOAD_CONTENT_ENCODING + "\").");
		}		
		
		return message;
	}

	private byte[] filterBOM(byte[] payload, String bomEncoding) throws ActionProcessingException {
		byte[] bom = getBOM(bomEncoding);
		
		if (bom == null) {
			return payload;
		}
		if (payload.length < bom.length) {			
			return payload;
		}
		
		// Check payload for a BOM...
		if (!hasBOM(payload, bom)) {
			// No BOM on payload... return unchanged...
			return payload;
		}

		// Payload has a BOM... remove it...
		byte[]  newPayload = new byte[payload.length - bom.length];
		System.arraycopy(payload, bom.length, newPayload, 0, newPayload.length);
		
		return newPayload;
	}

	private InputStream filterBOM(InputStream payload, String bomEncoding) throws ActionProcessingException {
		byte[] bom = getBOM(bomEncoding);		

		if (bom == null) {
			return payload;
		}
		
		return new BOMFilterInputStream(new BufferedInputStream(payload), bom);
	}

	protected static byte[] getBOM(String bomEncoding) {
		if (bomEncoding == null) {
			return null;
		}
		
		return bomMap.get(bomEncoding.replace("-", "").toUpperCase());
	}

	protected static boolean hasBOM(byte[] dataBuffer, byte[] bom) {
		for (int i = 0; i < bom.length; i++) {
			if (dataBuffer[i] != bom[i]) {
				// No BOM on payload...
				return false;
			}
		}
		
		return true;
	}

	private static class BOMFilterInputStream extends FilterInputStream {

		private byte[] bom;
		private boolean bomFilteringComplete = false;

		protected BOMFilterInputStream(BufferedInputStream in, byte[] bom) {
			super(in);
			this.bom = bom;
			in.mark(bom.length);
 		}

		/* (non-Javadoc)
		 * @see java.io.FilterInputStream#read()
		 */
		@Override
		public int read() throws IOException {
			if (!bomFilteringComplete) {
				filterBOM();
			}
			return super.read();
		}

		/* (non-Javadoc)
		 * @see java.io.FilterInputStream#read(byte[])
		 */
		@Override
		public int read(byte[] b) throws IOException {
			if (!bomFilteringComplete) {
				filterBOM();
			}
			return super.read(b);
		}

		/* (non-Javadoc)
		 * @see java.io.FilterInputStream#read(byte[], int, int)
		 */
		@Override
		public int read(byte[] b, int off, int len) throws IOException {
			if (!bomFilteringComplete) {
				filterBOM();
			}
			return super.read(b, off, len);
		}

		protected void filterBOM() throws IOException {
			try {
				byte[] bomBuffer = new byte[bom.length];
				int readCount = in.read(bomBuffer);
				
				if (readCount < bom.length) {
					// There's no BOM... reset back to the start of the stream...
					in.reset();
				} else if (!hasBOM(bomBuffer, bom)) {
					// There's no BOM... reset back to the start of the stream...
					in.reset();
				} else {
					// It has a BOM... don't reset.  This filters the BOM...
				}
			} finally {
				bomFilteringComplete = true;
			}
		}
	}
}
