/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions.routing.email;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.AbstractActionPipelineProcessor;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;

/**
 * EmailWiretap will publish the ESB message payload to a configured email account.
 * <p/>
 * 
 * Example configuration:
 * <pre>{@code
 * <action name="send-email" class="org.jboss.soa.esb.actions.routing.email.EmailWiretap">
 *    <property name="host" value="localhost" />
 *    <property name="port" value="25" />
 *    <property name="username" value="danbev" />
 *    <property name="password" value="password" />
 *    <property name="from" value="jbossesb" />
 *    <property name="sendTo" value="danbev" />
 *    <property name="subject" value="Subject goes here" />
 * </action>
 * }</pre>
 * 
 * For details about the properties listed above please see {@link Emailer}s javadoc.
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 * @since 4.6
 *
 */
public class EmailWiretap extends AbstractActionPipelineProcessor 
{
    private Emailer emailer;

    public EmailWiretap(final ConfigTree config) throws ConfigurationException
    {
        emailer = new Emailer(config);
    }
    
    public Message process(final Message message) throws ActionProcessingException 
    {
        try
        {
            emailer.sendEmail(message);
        } 
        catch (final MessageDeliverException e)
        {
            throw new ActionProcessingException(e.getMessage(), e);
        }
        
        return message;
    }
}
