package org.jboss.soa.esb.addressing.eprs;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */


/**
 * This class represents the endpoint reference for services.
 */

import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;

import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.PortReference;
import org.jboss.soa.esb.addressing.XMLUtil;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * A helper class for using FTP style EPRs. Simply create instances of this
 * class instead of the base EPR. Since URLs can use FTP, we try to leverage
 * that as much as possible.
 * 
 * @author marklittle
 *
 */
public class FTPEpr extends FileEpr
{
	public static final String USERNAME_TAG = "username";
	public static final String PASSWORD_TAG = "password";
	public static final String PASSIVE_TAG = "passive";
	
	public static final String CONTROL_CHANNEL_ENCODING_TAG = "controlChannelEncoding";
	
	public FTPEpr (EPR epr)
	{
		super(epr);
		
		if (epr instanceof FTPEpr)
		{
		    FTPEpr instance = (FTPEpr) epr;
		    
		    passwordSet = instance.passwordSet;
		    userSet = instance.userSet;
		    passiveSet = instance.passiveSet;
		}
	}
	
	public FTPEpr (EPR epr, Element header)
	{
		super(epr, header);
		
		NodeList nl = header.getChildNodes();

		for (int i = 0; i < nl.getLength(); i++)
		{
			try
			{
				String prefix = nl.item(i).getPrefix();
				String tag = nl.item(i).getLocalName();
				
				if ((prefix != null) && (prefix.equals(XMLUtil.JBOSSESB_PREFIX)))
				{
					if (tag != null)
					{
						if (tag.equals(USERNAME_TAG))
						    setUserName(nl.item(i).getTextContent());
						else
						{
							if (tag.equals(PASSWORD_TAG))
							    setPassword(nl.item(i).getTextContent());
							else
							{
								if (tag.equals(PASSIVE_TAG))
								{
								    String content = nl.item(i).getTextContent();
								    
								    if ("true".equalsIgnoreCase(content))
									setPassive(true);
								    else
								    {
									if ("false".equalsIgnoreCase(content))
									    setPassive(false);
								    }
								}
							}
						}
					}
				}
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}
		}
	}
	
	/**
	 * Construct the FTP EPR with the specified URL.
	 * @param url
	 * @throws URISyntaxException
	 * @deprecated
	 */
	public FTPEpr (URL url) throws URISyntaxException
	{
		this(url.toURI());
	}

       /**
         * Construct the FTP EPR with the specified URI.
         * @param uri
         * @throws URISyntaxException
         */
        public FTPEpr (URI uri)
        {
                this(uri, type().toString());
        }

	public FTPEpr (String url) throws URISyntaxException
	{
		this(new URI(url));
	}

	protected FTPEpr(final URI uri, final String type)
	{
		super(uri, type);
	}

	protected FTPEpr(final String url, final String type) throws URISyntaxException
	{
		this(new URI(url), type);
	}

	/**
	 * Set the URL for this endpoint.
	 * 
	 * @param url the address.
	 * @deprecated
	 */
	
	public final void setURL (URL url)
	{
		super.setAddr(new PortReference(url.toString()));
		setType(type().toString()) ;
	}
	
	/**
	 * Get the URL address.
	 * 
	 * @return the address.
	 * @deprecated
	 */
	
	public final URL getURL ()
	{
            try
            {
		return new URL(super.getAddr().getAddress());
            }
            catch (Exception ex)
            {
                _logger.warn("Unexpected parsing exception!", ex);
                
                return null;
            }
	}

	/**
	 * Set the username for this FTP EPR. In case it cannot be put in the URL.
	 * 
	 * @param username the user's name.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setUserName (String username) throws URISyntaxException
	{
		if (username == null)
			throw new IllegalArgumentException();
		
		if (userSet)
			throw new IllegalStateException("Username already set.");
		
		getAddr().addExtension(USERNAME_TAG, username);
		userSet = true;
	}
	
	/**
	 * @return the user's name associated with this EPR.
	 */
	
	public final String getUserName ()
	{
		return getAddr().getExtensionValue(USERNAME_TAG);
	}
	
	/**
	 * Set the password for this FTP EPR.
	 * 
	 * @param password the user's name.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setPassword (String password) throws URISyntaxException
	{
		if (password == null)
			throw new IllegalArgumentException();
		
		if (passwordSet)
			throw new IllegalStateException("Cannot change password");
		
		getAddr().addExtension(PASSWORD_TAG, password);
		passwordSet = true;
	}
	
	/**
	 * @return the password associated with this EPR.
	 */
	
	public final String getPassword ()
	{
		return getAddr().getExtensionValue(PASSWORD_TAG);
	}
	
	/**
	 * Passive FTP?
	 * 
	 * @param passive the value.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setPassive (boolean passive) throws URISyntaxException
	{
		if (passiveSet)
			throw new IllegalStateException("Cannot change passive");
		
		if (passive)
			getAddr().addExtension(PASSIVE_TAG, "true");
		else
			getAddr().addExtension(PASSIVE_TAG, "false");
		
		passiveSet = true;
	}
	
	/**
	 * @return the passive value associated with this EPR.
	 */
	
	public final boolean getPassive ()
	{
		return "true".equals(getAddr().getExtensionValue(PASSIVE_TAG));
	}
	
	public String toString ()
	{
		return "FTPEpr [ "+super.getAddr().extendedToString()+" ]";
	}

	public EPR copy ()
	{
	    return new FTPEpr(this);
	}
	
	public static URI type ()
	{
	    return _type;
	}

	public void setControlChannelEncoding(String ctrlChannelEncoding) {
		this.ctrlChannelEncoding = ctrlChannelEncoding;
	}
	
	public String getControlChannelEncoding() {
		return this.ctrlChannelEncoding;
	}
	
	private boolean passwordSet = false;
	private boolean userSet = false;
	private boolean passiveSet = false;
	private String ctrlChannelEncoding;
	
	private static URI _type;
	
	static
	{
	    try
		{
		    _type = new URI("urn:jboss/esb/epr/type/ftp");
		}
		catch (Exception ex)
		{
		    ex.printStackTrace();
		    
		    throw new ExceptionInInitializerError(ex.toString());
		}
	}


}