/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.common;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Properties;

import javax.jms.ConnectionFactory;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.jboss.internal.soa.esb.rosetta.pooling.handlers.JBossMessagingConnectionExceptionHandler;
import org.jboss.internal.soa.esb.rosetta.pooling.handlers.WMQConnectionExceptionHandler;
import org.jboss.internal.soa.esb.services.registry.InVMRegistryInterceptor;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.helpers.NamingContextException;
import org.jboss.soa.esb.helpers.NamingContextPool;

public class Configuration
{

	private static KeyValuePair[] s_oaKV = new KeyValuePair[]
	{
			new KeyValuePair(Environment.SMTP_HOST, 					getSmtpHost()),
			new KeyValuePair(Environment.SMTP_USERNAME, 				getSmtpUsername()),
			new KeyValuePair(Environment.SMTP_PASSWORD, 				getSmtpPassword()),
			new KeyValuePair(Environment.SMTP_PORT, 					getSmtpPort()),
			new KeyValuePair(Environment.SMTP_FROM, 					getSmtpFrom()),
			new KeyValuePair(Environment.SMTP_AUTH, 					getSmtpAuth()),
			new KeyValuePair(Environment.HTTP_HOST, 					getHttpHost()),
			new KeyValuePair(Environment.HTTP_PORT, 					getHttpPort()),
			new KeyValuePair(Environment.HTTP_SECURE_PORT, 				getHttpSecurePort()),
			new KeyValuePair(Environment.HTTP_RESPONSE_STATUS_ENABLED, 	getHttpResponseStatusEnabled()),
			new KeyValuePair(Environment.FTP_LOCALDIR, 	                getFtpLocalDir()),
			new KeyValuePair(Environment.FTP_REMOTEDIR, 	            getFtpRemoteDir()),
			new KeyValuePair(Environment.FTP_RENAME_RETRY, 	            getFtpRenameRetry()),
			new KeyValuePair(Environment.FTP_TIMEOUT, 	                getFtpTimeout()),
			new KeyValuePair(Environment.FTP_TIMEOUT_DEFAULT, 	        getFtpTimeoutDefault()),
			new KeyValuePair(Environment.FTP_TIMEOUT_DATA, 	            getFtpTimeoutData()),
			new KeyValuePair(Environment.FTP_TIMEOUT_SO, 	            getFtpTimeoutSo()),
			new KeyValuePair(Environment.JNDI_SERVER_URL, 				getJndiServerURL()),
			new KeyValuePair(Environment.JNDI_SERVER_CONTEXT_FACTORY,   getJndiServerContextFactory()),
			new KeyValuePair(Environment.JNDI_SERVER_PKG_PREFIX, 	    getJndiServerPkgPrefix()),
			new KeyValuePair(Environment.REGISTRY_QUERY_MANAGER_URI, 	getRegistryQueryManageURI()),
			new KeyValuePair(Environment.REGISTRY_LIFECYCLE_MANAGER_URI,getRegistryLifecycleManagerURI()),
			new KeyValuePair(Environment.REGISTRY_SECURITY_MANAGER_URI, getRegistrySecurityManagerURI()),
			new KeyValuePair(Environment.REGISTRY_IMPEMENTATION_CLASS,  getRegistryImplementationClass()),
			new KeyValuePair(Environment.REGISTRY_FACTORY_CLASS, 		getRegistryFactoryClass()),
			new KeyValuePair(Environment.REGISTRY_USER, 				getRegistryUser()),
			new KeyValuePair(Environment.REGISTRY_PASSWORD, 			getRegistryPassword()),
			new KeyValuePair(Environment.REGISTRY_SCOUT_TRANSPORT_CLASS,getRegistryScoutTransportClass()),
			new KeyValuePair(Environment.REGISTRY_SCOUT_UDDI_VERSION,   getRegistryUDDIVersion()),
			new KeyValuePair(Environment.REGISTRY_SCOUT_UDDI_NAMESPACE, getRegistryUDDINameSpace()),
			new KeyValuePair(Environment.PARAMS_REPOS_IMPL_CLASS,		getParamRepositoryImplClass()),
			new KeyValuePair(Environment.OBJECT_STORE_CONFIG_FILE, 		getObjStoreConfigFile()),
			new KeyValuePair(Environment.ENCRYPT_FACTORY_CLASS, 		getEncryptionFactoryClass()),
            new KeyValuePair(Environment.LOAD_BALANCER_POLICY,          getLoadBalancerPolicy()),
            new KeyValuePair(Environment.REDELIVER_DLS_SERVICE_ON,      getRedeliveryDlsOn()),
            new KeyValuePair(Environment.REGISTRY_CACHE_LIFE_MILLIS,   getRegistryCacheLife()),
			new KeyValuePair(Environment.MSG_STORE_DB_CONNECTION_URL, 	getStoreUrl()),
			new KeyValuePair(Environment.MSG_STORE_DB_JDBC_DRIVER, 		getStoreDriver()),
			new KeyValuePair(Environment.MSG_STORE_DB_CONNECTION_USER, 	getStoreUser()),
			new KeyValuePair(Environment.MSG_STORE_DB_CONNECTION_PWD, 	getStorePwd()),
			new KeyValuePair(Environment.MSG_STORE_DB_POOL_MIN_SIZE, 	getStorePoolMinSize()),
			new KeyValuePair(Environment.MSG_STORE_DB_POOL_MAX_SIZE, 	getStorePoolMaxSize()),
			new KeyValuePair(Environment.MSG_STORE_DB_POOL_INITIAL_SIZE,getStorePoolInitialSize()),
			new KeyValuePair(Environment.MSG_STORE_DB_POOL_TEST_TABLE,  getStorePoolTestTable()),
			new KeyValuePair(Environment.MSG_STORE_DB_POOL_TIMEOUT_MILLIS,getStorePoolTimeoutMillis()),
			new KeyValuePair(Environment.MSG_STORE_DB_CONN_MANAGER		,getStoreDBConnectionManager()),
			new KeyValuePair(Environment.MSG_STORE_DB_DATASOURCE_NAME	,getStoreDBDatasourceName()),
			new KeyValuePair(Environment.PROCESSING_PIPELINE_FAILURE_INTERCEPTORS, getPipelineFailureInterceptors()),
			new KeyValuePair(Environment.PROCESSING_PIPELINE_END_INTERCEPTORS, getPipelineEndInterceptors()),
			new KeyValuePair(Environment.PROCESSING_PIPELINE_INSTANTIATION_INTERCEPTORS, getPipelineInstantiateInterceptors()),
			new KeyValuePair(Environment.PROCESSING_PIPELINE_START_INTERCEPTORS, getPipelineStartInterceptors()),
			new KeyValuePair(Environment.REGISTRY_INTERCEPTORS			,getRegistryInterceptors()),
			new KeyValuePair(Environment.REGISTRY_CACHE_MAX_SIZE		,getRegistryCacheMaxSize()),
			new KeyValuePair(Environment.REGISTRY_CACHE_VALIDITY_PERIOD	,getRegistryCacheValidityPeriod()),
			new KeyValuePair(Environment.ESB_BIND_ADDRESS ,getBindAddress()),
			new KeyValuePair(Environment.RULES_CONTINUE_STATE			,getRulesContinueState()),
			new KeyValuePair(Environment.RULES_RESOURCE_SCANNER_INTERVAL,getRulesResourceScannerInterval()),
			new KeyValuePair(Environment.DEPLOYMENT_SCHEMA_VALIDATION	,getDeploymentSchemaValidation()),
			new KeyValuePair(Environment.BINARY_SECURITY_TOKEN_IMPLEMENTATION_CLASS, getBinarySecurityTokenImplClass())
	};

	public static String dump()
	{
		StringBuilder sb = new StringBuilder("Dump of SystemProperties:\n");
		for (KeyValuePair oCurr : s_oaKV)
		{
			sb.append(oCurr.getKey()).append("=").append(oCurr.getValue())
					.append("\n");
		}
		return sb.append("______________________________________").toString();
	} // ________________________________

	public static String getSmtpHost()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.SMTP_HOST, Environment.DEFAULT_HOST);
	}

	public static String getSmtpUsername()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.SMTP_USERNAME,
				Environment.DEFAULT_USERNAME);
	}

	public static String getSmtpPassword()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.SMTP_PASSWORD,
				Environment.DEFAULT_PASSWORD);
	}

	public static String getSmtpPort()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.SMTP_PORT, Environment.DEFAULT_PORT);
	}

	public static String getSmtpFrom()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.SMTP_FROM);
	}

	public static String getSmtpAuth()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.SMTP_AUTH);
	}
	
	public static String getHttpHost()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.HTTP_HOST);
	}
	
	public static String getHttpPort()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.HTTP_PORT);
	}
	
	public static String getHttpSecurePort()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.HTTP_SECURE_PORT);
	}
	
	public static String getHttpResponseStatusEnabled()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.HTTP_RESPONSE_STATUS_ENABLED);
	}
	
	public static String getFtpLocalDir()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.FTP_LOCALDIR);
	}
	
	public static String getFtpRemoteDir()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.FTP_REMOTEDIR);
	}
	
	public static String getFtpRenameRetry()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.FTP_RENAME_RETRY);
	}
	
	public static String getFtpTimeout()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.FTP_TIMEOUT);
	}
	
	public static String getFtpTimeoutDefault()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.FTP_TIMEOUT_DEFAULT);
	}
	
	public static String getFtpTimeoutData()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.FTP_TIMEOUT_DATA);
	}
	
	public static String getFtpTimeoutSo()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.FTP_TIMEOUT_SO);
	}

	public static String getJndiServerURL()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).getProperty(Environment.JNDI_SERVER_URL,
				Environment.DEFAULT_HOST);
	}

	public static String getJndiServerContextFactory()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).getProperty(Environment.JNDI_SERVER_CONTEXT_FACTORY,
				Environment.DEFAULT_JNDI_CONTEXT_FACTORY);
	}

	public static String getJndiServerPkgPrefix()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).getProperty(Environment.JNDI_SERVER_PKG_PREFIX,
				Environment.DEFAULT_JNDI_PKG_PREFIX);
	}

    public static String getLoadBalancerPolicy()
    {
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).getProperty(Environment.LOAD_BALANCER_POLICY,
                Environment.DEFAULT_LOAD_BALANCER_POLICY);
    }

    public static String getRedeliveryDlsOn()
    {
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).getProperty(Environment.REDELIVER_DLS_SERVICE_ON,
                    Environment.DEFAULT_REDELIVER_DLS_ON);
    }

    public static String getRegistryCacheLife()
    {
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).getProperty(Environment.REGISTRY_CACHE_LIFE_MILLIS,
                    Environment.DEFAULT_REGISTRY_CACHE_LIFE_MILLIS);
    }
	/**
	 * The Registry Query Manager URI can be used to obtain information about services and their endPoints.
	 * @return Registry Query Manager URI String
	 */
	public static String getRegistryQueryManageURI()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_QUERY_MANAGER_URI);
	}
	/**
	 * The Registry Lifecycle Manager URI can be use to publish services.
	 * @return Registry Lifecycle Manager URI String
	 */
	public static String getRegistryLifecycleManagerURI()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_LIFECYCLE_MANAGER_URI);
	}
	/**
	 * The Registry Security Manager URI can be use to publish services.
	 * @return Registry Lifecycle Manager URI String
	 */
	public static String getRegistrySecurityManagerURI()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_SECURITY_MANAGER_URI);
	}
	/**
	 * The Registry Factory Class setting specifies which JAXR implementation should be used.
	 * @return Registry Factory Class String
	 */
	public static String getRegistryFactoryClass()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_FACTORY_CLASS);
	}
	/**
	 * The Registry Implememtation Class defines wrapper class
	 * @return Registry Implementation Class String
	 */
	public static String getRegistryImplementationClass()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_IMPEMENTATION_CLASS);
	}
	/**
	 * The Registry Scout Transport Class defines which communication protocol Scout should use to communicate
	 * with the UDDI registry. Note that this parameter is Scout specific.
	 * @return Registry Scout Transport Class String
	 */
	public static String getRegistryScoutTransportClass()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_SCOUT_TRANSPORT_CLASS);
	}
	/**
	 * The Registry Scout UDDI version. Note that this parameter is Scout specific.
	 * @return Registry Scout UDDI version
	 */
	public static String getRegistryUDDIVersion()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_SCOUT_UDDI_VERSION);
	}	
	/**
	 * The Registry Scout UDDI namespace. Note that this parameter is Scout specific.
	 * @return Registry Scout UDDI namespace
	 */
	public static String getRegistryUDDINameSpace()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_SCOUT_UDDI_NAMESPACE);
	}	
	/**
	 * The Registry User is used to inquire and publish to the Registry.
	 * @return Registry User String
	 */
	public static String getRegistryUser()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_USER);
	}
	/**
	 * The Registry Password that goes with the Registry User.
	 * @return Registry Password String
	 */
	public static String getRegistryPassword()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_PASSWORD);
	}

    public static String getRegistrySemanticEquivalences()
    {
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_SEMANTIC_EQUIVALENCES);
    }

    public static String getRegistryPostalAddressScheme()
    {
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_POSTAL_ADDRESS_SCHEME);
    }

    public static String getRegistrySecurityAuthenticationMethod()
    {
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_SECURITY_AUTHENTICATION_METHOD);
    }

    public static String getRegistryUDDIMaxRows()
    {
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_UDDI_MAX_ROWS);
    }

	public static String getParamRepositoryImplClass()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).getProperty(Environment.PARAMS_REPOS_IMPL_CLASS);
	}

	public static String getObjStoreConfigFile()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).getProperty(Environment.OBJECT_STORE_CONFIG_FILE);
		return property;
	}

	public static String getEncryptionFactoryClass()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).getProperty(Environment.ENCRYPT_FACTORY_CLASS,
				org.jboss.soa.esb.services.DefaultEncryptionFactory.class
						.getName());
	}

	public static String getStoreUrl()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_DB_CONNECTION_URL);
		return property;
	}

	public static String getStoreDriver()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_DB_JDBC_DRIVER);
		return property;
	}

	public static String getStorePwd()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_DB_CONNECTION_PWD);
		return property;
	}

	public static String getStoreUser()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_DB_CONNECTION_USER);
		return property;
	}

	public static String getStorePoolInitialSize()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_DB_POOL_INITIAL_SIZE);
		return property;
	}

	public static String getStorePoolMinSize()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_DB_POOL_MIN_SIZE);
		return property;
	}

	public static String getStorePoolMaxSize()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_DB_POOL_MAX_SIZE);
		return property;
	}

	public static String getStorePoolTestTable()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_DB_POOL_TEST_TABLE);
		return property;
	}

	public static String getStorePoolTimeoutMillis()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_DB_POOL_TIMEOUT_MILLIS);
		return property;
	}

	public static String getStoreDBConnectionManager()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_DB_CONN_MANAGER);
		return property;
	}

	public static String getStoreDBDatasourceName()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_DB_DATASOURCE_NAME);
		return property;
	}

	public static String getJcrStoreJNDIPath()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_JCR_JNDI_PATH);
	}

	public static String getJcrStoreUsername()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_JCR_USERNAME);
	}

	public static String getJcrStorePassword()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_JCR_PASSWORD);
	}

	public static String getJcrStoreRootNodePath()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_JCR_ROOT_NODE_PATH);
	}

    /**
     * Get the pipeline failure event interceptor class names
     * @return Pipeline failure event interceptor Class names.
     */	
	public static String getPipelineFailureInterceptors()
	{
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.INTERCEPTOR_MODULE).getProperty(Environment.PROCESSING_PIPELINE_FAILURE_INTERCEPTORS,
                null);

	}
	
	/**
     * Get the pipeline end event interceptor class names
     * @return Pipeline end event interceptor Class names.
     */	
	public static String getPipelineEndInterceptors()
	{
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.INTERCEPTOR_MODULE).getProperty(Environment.PROCESSING_PIPELINE_END_INTERCEPTORS,
                null);

	}

    /**
     * Get the pipeline start event interceptor class names
     * @return Pipeline start event interceptor Class names.
     */	
	public static String getPipelineStartInterceptors()
	{
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.INTERCEPTOR_MODULE).getProperty(Environment.PROCESSING_PIPELINE_START_INTERCEPTORS,
                null);
	}

    /**
     * Get the pipeline instantiate event interceptor class names
     * @return Pipeline instantiate event interceptor Class names.
     */	
	public static String getPipelineInstantiateInterceptors()
	{
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.INTERCEPTOR_MODULE).getProperty(Environment.PROCESSING_PIPELINE_INSTANTIATION_INTERCEPTORS,
                null);		
	}

	
    /**
     * Get the registry interceptor class names
     * @return Registry interceptor Class names.
     */
    public static String getRegistryInterceptors()
    {
        // If nothing is specified then we default to the InVM interceptor
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_INTERCEPTORS,
            InVMRegistryInterceptor.class.getName());
    }

    /**
     * Get the maximum cache size for the caching registry.
     * @return The maximum cache size.
     */
    public static String getRegistryCacheMaxSize()
    {
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_CACHE_MAX_SIZE) ;
    }

    /**
     * Get the validity period for the caching registry.
     * @return The cache validity period.
     */
    public static String getRegistryCacheValidityPeriod()
    {
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_CACHE_VALIDITY_PERIOD,
            getRegistryCacheLife()) ;
    }

	/**
	 * Construct a naming context based on the connection details outlined
	 * in the named configuration module/section.
	 * <p/>
	 * The module/section parameters are:
	 * <ol>
	 * 	<li><b>{@link Context#PROVIDER_URL}</b>: Value defaults to "{@link Environment#JBOSS_PROVIDER_URL}".</li>
	 * 	<li><b>{@link Context#INITIAL_CONTEXT_FACTORY}</b>: Value defaults to "{@link Environment#JBOSS_INITIAL_CONTEXT_FACTORY}".</li>
	 * 	<li><b>{@link Context#URL_PKG_PREFIXES}</b>: Value defaults to "{@link Environment#JBOSS_URL_PKG_PREFIX}".</li>
	 * </ol>
	 *
	 * @param contextModuleName Conficuration module/section name.
	 * @return The context instance.
	 * @throws ConfigurationException Unable to connect to context.
	 *
	 * @deprecated {@link org.jboss.soa.esb.helpers.NamingContextPool}
	 */
	@Deprecated
    public static Context getNamingContext(String contextModuleName) throws ConfigurationException {
	    final Properties environment = getProperties(contextModuleName) ;

        try {
            return new InitialContext(environment) ;
        } catch (final NamingException ne) {
            throw new ConfigurationException("Failed to create JNDI context [" + contextModuleName + "].");
        }
	}

	private static Properties getProperties(final String contextModuleName)
	{
            final String providerUrl = ModulePropertyManager.getPropertyManager(contextModuleName).getProperty(Context.PROVIDER_URL, Environment.JBOSS_PROVIDER_URL);
            final String initialContextFactory = ModulePropertyManager.getPropertyManager(contextModuleName).getProperty(Context.INITIAL_CONTEXT_FACTORY, Environment.JBOSS_INITIAL_CONTEXT_FACTORY);
            final String urlPackagePrefix = ModulePropertyManager.getPropertyManager(contextModuleName).getProperty(Context.URL_PKG_PREFIXES, Environment.JBOSS_URL_PKG_PREFIX);
            final Properties environment = new Properties();
            environment.setProperty(Context.PROVIDER_URL, providerUrl);
            environment.setProperty(Context.INITIAL_CONTEXT_FACTORY, initialContextFactory);
            environment.setProperty(Context.URL_PKG_PREFIXES, urlPackagePrefix);
            return environment ;
	}

	/**
	 * Lookup the JMS Connection Factory based on the connection details outlined
	 * in the named configuration module/section.
	 * <p/>
	 * The module/section parameters are:
	 * <ol>
	 * 	<li><b>{@link Context#PROVIDER_URL}</b>: Value defaults to "{@link Environment#JBOSS_PROVIDER_URL}".</li>
	 * 	<li><b>{@link Context#INITIAL_CONTEXT_FACTORY}</b>: Value defaults to "{@link Environment#JBOSS_INITIAL_CONTEXT_FACTORY}".</li>
	 * 	<li><b>{@link Context#URL_PKG_PREFIXES}</b>: Value defaults to "{@link Environment#JBOSS_URL_PKG_PREFIX}".</li>
	 * 	<li><b>javax.jms.ConnectionFactory</b>: Value defaults to "".</li>
	 * </ol>
	 *
	 * @param jmsConnectionFactoryModuleName Configuration module/section name.
	 * @return The JMS ConnectionFactory instance.
	 * @throws ConfigurationException Lookup failed either because it was unable to
	 * lookup the context, or the ConnectionFactory lookup failed.
	 */
	public static ConnectionFactory getJmsConnectionFactory(String jmsConnectionFactoryModuleName) throws ConfigurationException {
	        final Properties environment = getProperties(jmsConnectionFactoryModuleName) ;
	        try {
	            Context context = NamingContextPool.getNamingContext(environment);
	            try {
                        String connectionFactoryRuntime = ModulePropertyManager.getPropertyManager(jmsConnectionFactoryModuleName).getProperty(ConnectionFactory.class.getName(), "ConnectionFactory");
                        ConnectionFactory factory = null;

                        try {
                                factory = (ConnectionFactory) context.lookup(connectionFactoryRuntime);
                        } catch (NamingException e) {
                                throw new ConfigurationException("JNDI lookup of JMS Connection Factory [" + connectionFactoryRuntime + "] failed.", e);
                        } catch (ClassCastException e) {
                                throw new ConfigurationException("JNDI lookup of JMS Connection Factory failed.  Class [" + connectionFactoryRuntime + "] is not an instance of [" + ConnectionFactory.class.getName() + "].", e);
                        }
                        return factory;
	            } finally {
	                NamingContextPool.releaseNamingContext(context) ;
	            }
	        } catch (final NamingContextException nce) {
	            throw new ConfigurationException("Unexpected exception while accessing naming context pool", nce) ;
	        }
        }

    /**
     * Get the JMS Connection Exception Handler class names
     * @return JMS Connection Exception Handler class names.
     */
    public static String getJMSConnectionExceptionHandlers()
    {
        // If nothing is specified then we default to the InVM interceptor
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.JMS_CONNECTION_EXCEPTION_HANDLERS,
            JBossMessagingConnectionExceptionHandler.class.getName() + ", " + WMQConnectionExceptionHandler.class.getName());
    }

	/**
	 * The SecurityService Implememtation Class
	 *
	 * @return String the fully qualified name of the ServiceService implementation class
	 */
	public static String getSecurityServiceImplementationClass()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.SECURITY_MODULE).getProperty(Environment.SECURITY_SERVICE_IMPEMENTATION_CLASS);
	}

	/*
	 *  Public Keystore getters
	 */

	public static String getSecurityServicePublicKeystore()
    {
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.SECURITY_MODULE).getProperty(Environment.SECURITY_SERVICE_PUBLIC_KEYSTORE);
    }

    public static String getSecurityServicePublicKeyAlias()
    {
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.SECURITY_MODULE).getProperty(Environment.SECURITY_SERVICE_PUBLIC_KEY_ALIAS);
    }

    public static String getSecurityServicePublicKeyPassword()
    {
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.SECURITY_MODULE).getProperty(Environment.SECURITY_SERVICE_PUBLIC_KEY_PASS);
    }

    public static String getSecurityServicePublicKeyTransformation()
    {
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.SECURITY_MODULE).getProperty(Environment.SECURITY_SERVICE_PUBLIC_KEY_TRANSFORMATION);
    }

    public static String getSecurityServicePublicKeystorePassword()
    {
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.SECURITY_MODULE).getProperty(Environment.SECURITY_SERVICE_PUBLIC_KEYSTORE_PASS);
    }

    public static String getSecurityServicePublicKeystoreType()
    {
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.SECURITY_MODULE).getProperty(Environment.SECURITY_SERVICE_PUBLIC_KEYSTORE_TYPE);
    }

	public static String getSecurityServiceCallbackHandlerImplClass()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.SECURITY_MODULE).getProperty(Environment.SECURITY_SERVICE_CALLBACK_HANLDER_CLASS);
	}

	public static String getSecurityServiceSealAlgorithm()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.SECURITY_MODULE).getProperty(Environment.SECURITY_SERVICE_SEAL_ALGORITHM);
	}

	public static String getSecurityServiceSealKeySize()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.SECURITY_MODULE).getProperty(Environment.SECURITY_SERVICE_SEAL_KEYSIZE);
	}

	public static String getSecurityServiceContextTimeout()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.SECURITY_MODULE).getProperty(Environment.SECURITY_SERVICE_CONTEXT_TIMEOUT);
	}

	public static Properties getSecurityServiceProperies()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.SECURITY_MODULE).getProperties();
	}

	/**
	 * The SecurityContextPropagator implememtation class to use.
	 *
	 * @return String The fully qualified name of the SecurityContextPropagator implementation class
	 */
	public static String getSecurityContextPropagatorImplementationClass()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.SECURITY_MODULE).getProperty(Environment.SECURITY_SERVICE_CONTEXT_PROPAGATOR_CLASS);
	}

	/**
	 * The JBossASSecurityContextOperations implementation class to use.
	 *
	 * @return String The fully qualified name of the JBossASSecurityContextOperations implementation class
	 */
	public static String getJBossASSecurityContextOperationsImplementationClass()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.SECURITY_MODULE).getProperty(Environment.SECURITY_JBOSSAS_SECURITY_CONTEXT_OPERATIONS_CLASS);
	}

    /**
	 * Get the bind address.
	 * @return the bind address.
	 */
	public static String getBindAddress()
	{
		String address = System.getProperty(Environment.ESB_BIND_ADDRESS) ;
		if (address == null)
		{
			// This property is always set in the server, but may not be set if executing outside.
			try
			{
				address = InetAddress.getLocalHost().getHostName();
			}
			catch (final UnknownHostException uhe)
			{
				address = "127.0.0.1" ;
			}
		}
		return address ;
	}

	public static String getRulesContinueState()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.RULES_MODULE).getProperty(Environment.RULES_CONTINUE_STATE, "false");
	}

	public static String getRulesResourceScannerInterval()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.RULES_MODULE).getProperty(Environment.RULES_RESOURCE_SCANNER_INTERVAL, "60");
	}
	
	/**
	 * Should we validate the jboss-esb.xml?
	 * @return true if enabled, false if disabled
	 */
	public static String getDeploymentSchemaValidation()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).getProperty(Environment.DEPLOYMENT_SCHEMA_VALIDATION, "true") ;
	}
	
	public static String getBinarySecurityTokenImplClass()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.SECURITY_MODULE).getProperty(Environment.BINARY_SECURITY_TOKEN_IMPLEMENTATION_CLASS,
				"org.jboss.soa.esb.services.security.auth.ws.BinarySecurityTokenImpl");
	}
}