/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.helpers.persist;

import java.io.PrintWriter;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;

import javax.sql.DataSource;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.util.ClassUtil;

/**
 * <p>
 * Obtain a connection that implements the DataSource interface
 * </p>
 * <p>
 * Description: Enable compatible use of JdbcCleanConn from outside a J2EE app
 * container
 * </p>
 * 
 * @version 1.0
 * @see JdbcCleanConn
 * @see ConfigTree
 */
public class SimpleDataSource implements DataSource
{
	private PrintWriter m_oPW = new PrintWriter(System.out);

	private int m_iTO = 10;

	private String m_sUrl, m_sUsr, m_sPwd;

	private Connection m_oConn;
	
	public static final String DRIVER = "driver-class";

	public static final String URL = "connection-url";

	public static final String USER = "user-name";

	public static final String PASSWORD = "password";

	private static final Logger _logger = Logger
			.getLogger(SimpleDataSource.class);

	// Disable default constructor
	private SimpleDataSource ()
	{
	}

	/**
	 * Obtain a DataSource by providing connection parameters. ConfigTree
	 * argument must contain the following attributes
	 * <p>
	 * <li> driver-class</li>
	 * <li> connection-url</li>
	 * <li> user-name</li>
	 * <li> password</li>
	 * </p>
	 * 
	 * @param p_oP
	 *            ConfigTree - Parameter tree that contains the 4 attributes
	 *            needed
	 * @see ConfigTree
	 * @see SimpleDataSource#SimpleDataSource(String,String,String,String)
	 * @see javax.sql.DataSource
	 */
	public SimpleDataSource (ConfigTree p_oP)
	{
		this(p_oP.getAttribute(DRIVER), p_oP.getAttribute(URL), p_oP
				.getAttribute(USER), p_oP.getAttribute(PASSWORD));
	} // ________________________________

	/**
	 * Obtain a connection to a DB using the provided arguments to call
	 * 
	 * @param p_sDriver
	 *            String - Java class of the JDBC driver to use (e.g.
	 *            org.postgresql.Driver)
	 * @param p_sDbURL
	 *            String - The URL of the database (e.g.
	 *            jdbc:postgresql://chodedb1:5432/cq3)
	 * @param p_sUsr
	 *            String - Username
	 * @param p_sPwd
	 *            String - Password
	 * @see DriverManager#getConnection(String,String,String)
	 * @see javax.sql.DataSource
	 */
	public SimpleDataSource (String p_sDriver, String p_sDbURL, String p_sUsr,
			String p_sPwd)
	{
		try
		{
			ClassUtil.forName(p_sDriver, getClass());
			m_sUrl = p_sDbURL;
			getConnection(p_sUsr, p_sPwd);
		}
		catch (ClassNotFoundException ex)
		{
			throw new IllegalArgumentException(ex);
		}
	} // ________________________________
	
	public Connection getConnection() throws SQLException
	{
		if ((m_oConn == null) || (m_oConn.isClosed())) {
			m_oConn = getConnection(m_sUsr, m_sPwd);
		}
		
		if (m_oConn == null) {
			throw new SQLException();
		}
		
		return m_oConn;
	}

	public Connection getConnection (String username, String password)
	{
		m_sUsr = username;
		m_sPwd = password;
		try
		{
			m_oConn = DriverManager.getConnection(m_sUrl, m_sUsr, m_sPwd);
		}
		catch (Exception e)
		{
			_logger.error("Could not create connection for " + m_sUrl + " - datasource may be unavailable.  "
					+ "See server log for details.");
			_logger.debug("Can't obtain datasource", e);
			m_oConn = null;
		}
		return m_oConn;
	} // ________________________________

	public int getLoginTimeout ()
	{
		return m_iTO;
	}

	public PrintWriter getLogWriter ()
	{
		return m_oPW;
	}

	public void setLoginTimeout (int seconds)
	{
		m_iTO = seconds;
	}

	public void setLogWriter (PrintWriter out)
	{
		m_oPW = out;
	}
	
	public boolean isWrapperFor(final Class<?> iface)
		throws SQLException
	{
		return false ;
	}
	
	public <T> T unwrap(final Class<T> iface) throws SQLException
	{
		throw new SQLException("unwrap method not supported by this implementation") ;
	}
} // ______________________________________________________
