/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.http;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.http.configurators.HttpProtocol;
import org.jboss.soa.esb.http.configurators.Connection;
import org.jboss.soa.esb.util.ClassUtil;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpConnectionManager;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;

import java.util.Properties;
import java.io.IOException;

/**
 * {@link org.apache.commons.httpclient.HttpClient} factory class.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public abstract class HttpClientFactory {

    /**
     * Target host url.
     */
    public static final String TARGET_HOST_URL = "target-host-url";
    /**
     * http-client-property.
     */
    public static final String HTTP_CLIENT_PROPERTY = "http-client-property";

    /**
     * Factory method for creating the {@link org.apache.commons.httpclient.HttpClient}
     * instance.
     *
     * @param properties {@link org.apache.commons.httpclient.HttpClient} creation properties.
     * @return The {@link org.apache.commons.httpclient.HttpClient} instance.
     * @throws ConfigurationException Invalid configuration properties set.
     */
    public static HttpClient createHttpClient(Properties properties) throws ConfigurationException {
        AssertArgument.isNotNull(properties, "properties");

        HttpClient httpClient = new HttpClient();
        String fileConfig = (String) properties.remove("file");
        String[] configurators;

        if(fileConfig != null) {
            try {
                properties.load(StreamUtils.getResource(fileConfig));
            } catch (IOException e) {
                throw new ConfigurationException("Failed to load HttpClient config '" + fileConfig + "'.");
            }
        }

        // Apply the mandatory Configurators...
        new Connection().configure(httpClient, properties);

        // Apply the optional (configured) Configurators...
        configurators = properties.getProperty("configurators", HttpProtocol.class.getName()).split(",");
        for(String configuratorClass : configurators) {
        	configuratorClass = configuratorClass.trim();
        	boolean skip = Boolean.valueOf(properties.getProperty(configuratorClass + ".skip", "false")).booleanValue();
        	if (!skip) {
	            Configurator configurator = createConfigurator(configuratorClass);
	            configurator.configure(httpClient, properties);
        	}
        }

        final HttpConnectionManager connectionManager = httpClient.getHttpConnectionManager();
        if (connectionManager instanceof ESBMultiThreadedHttpConnectionManager) {
            final ESBMultiThreadedHttpConnectionManager esbMultiThreadedHttpConnectionManager = (ESBMultiThreadedHttpConnectionManager)connectionManager;
            esbMultiThreadedHttpConnectionManager.setHostConfiguration(httpClient.getHostConfiguration()) ;
        }
        
        return httpClient;
    }

    private static Configurator createConfigurator(String configuratorClass) throws ConfigurationException {
        try {
             return (Configurator) ClassUtil.forName(configuratorClass, HttpClientFactory.class).newInstance();
        } catch (ClassCastException e) {
            throw new ConfigurationException("Class [" + configuratorClass + "] must extend [" + Configurator.class.getName() + "].", e);
        } catch (ClassNotFoundException e) {
            if(!configuratorClass.startsWith(HttpProtocol.class.getPackage().getName())) {
                return createConfigurator(HttpProtocol.class.getPackage().getName() + "." + configuratorClass);
            }
            throw new ConfigurationException("Configurator implementation class [" + configuratorClass + "] not found in classpath.", e);
        } catch (InstantiationException e) {
            throw new ConfigurationException("Failed to instantiate Configurator implementation class [" + configuratorClass + "].", e);
        } catch (IllegalAccessException e) {
            throw new ConfigurationException("Failed to instantiate Configurator implementation class [" + configuratorClass + "].", e);
        }
    }

    /**
     * Shutdown http client and associated connections.
     * @param httpclient The http client
     */
    public static void shutdown(final HttpClient httpclient) {
        if (httpclient != null) {
            final HttpConnectionManager connectionManager = httpclient.getHttpConnectionManager();
            if (connectionManager instanceof MultiThreadedHttpConnectionManager) {
                final MultiThreadedHttpConnectionManager multiThreadedHttpConnectionManager = (MultiThreadedHttpConnectionManager)connectionManager;
                multiThreadedHttpConnectionManager.shutdown();
            }
        }
    }
}
