/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.listeners.config.mappers110;

import java.util.HashMap;
import java.util.List;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.FsListenerDocument.FsListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.FtpListenerDocument.FtpListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.GroovyListenerDocument.GroovyListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.HibernateListenerDocument.HibernateListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.JbrListenerDocument.JbrListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.HttpListenerDocument.HttpListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.JmsListenerDocument.JmsListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.ServiceDocument.Service;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.SqlListenerDocument.SqlListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.UdpListenerDocument.UdpListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.Listener;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.JcaGatewayDocument;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.AbstractScheduledListener;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * Gateway Listeners Configuration Generator.
 * <p/>
 * Generates a "ConfigTree" style configuration for the Gateways Listeners.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class GatewayGenerator {
	
	private static final Logger logger = Logger.getLogger(GatewayGenerator.class);

    /**
     * XMLBeans based configuration model instance.
     */
    private XMLBeansModel model;

    /**
     * Public constructor.
     *
     * @param model XMLBeans based configuration model instance.
     */
    public GatewayGenerator(XMLBeansModel model) {
        this.model = model;
    }

    /**
     * Generate the configuration DOM for the target Server.
     *
     * @return The configuration DOM.
     * @throws ConfigurationException Error creating configuration.
     */
    public Document generate() throws ConfigurationException {
        Document doc = YADOMUtil.createDocument();
        Element root;

        doc.appendChild(doc.createComment("\n\tNOTE: DO NOT MODIFY\n\tThis file was auto-generated.\n"));
        root = YADOMUtil.addElement(doc, "jbossesb-gateways");
        root.setAttribute("parameterReloadSecs", model.getParameterReloadSecs());

        List<Service> services = model.getServices();
        HashMap<String, HashMap<String, String>> hm = new HashMap<String, HashMap<String, String>>();
        for (Service service : services) {
        	if (hm.containsKey(service.getCategory())) {
        		HashMap<String, String> hc = hm.get(service.getCategory());
        		
        		if (hc.containsKey(service.getName())) {
        			throw new ConfigurationException("Service names must be unique within a category.  "
        					+ "The service name " + service.getName()
        					+ " is already defined within the " + service.getCategory() + " category.");
        		} else {
        			hc.put(service.getName(), service.getName());
        		}
        	} else {
        		HashMap <String,String> hc = new HashMap<String,String>();
        		hc.put(service.getName(), service.getName());
        		hm.put(service.getCategory(), hc);
        	}
        }
        
        // Iterate over all the gateway configs and map them to the gateway configuration...
        List<Listener> gateways = model.getGatewayListeners();
        for (Listener gateway : gateways) {
            addGatewayConfig(root, gateway);
        }

        return doc;
    }

    /**
     * Add a single gateway configuration node to configuration root.
     *
     * @param root    Configuration root node.
     * @param gateway The Gateway Listener configuration to be added.
     * @throws ConfigurationException Invalid listener configuration.
     */
    private void addGatewayConfig(Element root, Listener gateway) throws ConfigurationException {
        Element listenerConfig;

        // Of course we could do the following reflectively if we really want to - TODO perhaps!!
        if (gateway instanceof JmsListener) {
            listenerConfig = JmsListenerMapper.map(root, (JmsListener) gateway, model);
        } else if (gateway instanceof FsListener) {
            listenerConfig = FsListenerMapper.map(root, (FsListener) gateway, model);
        } else if (gateway instanceof JcaGatewayDocument.JcaGateway) {
            listenerConfig = JcaGatewayMapper.map(root, (JcaGatewayDocument.JcaGateway) gateway, model);
        } else if (gateway instanceof FtpListener) {
            listenerConfig = FtpListenerMapper.map(root, (FtpListener) gateway, model);
        } else if (gateway instanceof SqlListener) {
            listenerConfig = SqlListenerMapper.map(root, (SqlListener) gateway, model);
        } else if (gateway instanceof JbrListener) {
            listenerConfig = JbrListenerMapper.map(root, (JbrListener) gateway, model);
        } else if (gateway instanceof HttpListener) {
        	listenerConfig = HttpListenerMapper.map(root, (HttpListener) gateway, model);            
        } else if (gateway instanceof GroovyListener) {
            listenerConfig = GroovyListenerMapper.map(root, (GroovyListener) gateway, model);
        } else if (gateway instanceof HibernateListener) {
            listenerConfig = HibernateListenerMapper.map(root, (HibernateListener) gateway, model);
        } else if (gateway instanceof UdpListener) {
            listenerConfig = UdpListenerMapper.map(root, (UdpListener) gateway, model);
        } else {
            listenerConfig = UntypedListenerMapper.map(root, gateway, model);
        }

        if (gateway instanceof AbstractScheduledListener) {
            ScheduleMapper.map(listenerConfig, (AbstractScheduledListener) gateway, model);
        }
    }
}
