/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.config.mappers130;

import java.util.List;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.addressing.eprs.HibernateEpr;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.config.xbeanmodel130.HibernateBusDocument.HibernateBus;
import org.jboss.soa.esb.listeners.config.xbeanmodel130.HibernateListenerDocument.HibernateListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel130.HibernateMessageFilterDocument.HibernateMessageFilter;
import org.jboss.soa.esb.listeners.config.xbeanmodel130.HibernateProviderDocument.HibernateProvider;
import org.jboss.soa.esb.listeners.gateway.HibernateGatewayListener;
import org.w3c.dom.Element;

/**
 * Performs the mapping of a &lt;fs-listener&gt; XSD based configuration to the "ConfigTree"
 * style configuration, adding the "ConfigTree" listener config to the "root" node.
 *
 * @author <a href="mailto:tcunning@redhat.com">tcunning@redhat.com</a>
 */
public class HibernateListenerMapper {

	/**
	 * Perform the mapping.
	 * @param root The "ConfigTree" configuration root node.
	 * @param listener The Jmslistener to be mapped into the ConfigTree.
	 * @param model The configuration model from which the mapping is being performed.
	 * @return The ConfigTree listener configuration node.
	 * @throws org.jboss.soa.esb.ConfigurationException Invalid listener configuration.
	 */
	public static Element map(Element root, HibernateListener listener, XMLBeansModel model) throws ConfigurationException {
		Element listenerNode = YADOMUtil.addElement(root, "listener");
		HibernateBus bus;
		HibernateProvider provider;

        listenerNode.setAttribute("name", listener.getName());

		try {
			bus = (HibernateBus) model.getBus(listener.getBusidref());
		} catch (ClassCastException e) {
			throw new ConfigurationException("Invalid busid reference [" + listener.getBusidref() + "] on listener [" + listener.getName() + "].  A <hibernate-listener> must reference a <hibernate-bus>.");
		}
		try {
			provider = (HibernateProvider) model.getProvider(bus);
		} catch (ClassCastException e) {
			throw new ConfigurationException("Invalid bus config [" + listener.getBusidref() + "].  Should be contained within a <hibernate-provider> instance.  Unexpected exception - this should have caused a validation error!");
		}

		List<HibernateMessageFilter> messageFilters = null;
		messageFilters = listener.getHibernateMessageFilterList();

		if(messageFilters.size() == 0) {
			messageFilters = bus.getHibernateMessageFilterList();
			if(messageFilters == null) {
				throw new ConfigurationException("No <hibernate-message-filter> defined on either <hibernate-listener> [" + listener.getName() + "] or <hibernate-bus> [" + bus.getBusid() + "].");
			}
		}
		// Map the standard listener attributes - common across all listener types...
		MapperUtil.mapDefaultAttributes(listener, listenerNode, model);
		// Map the <property> elements targeted at the listener - from the listener itself.
		MapperUtil.mapProperties(listener.getPropertyList(), listenerNode);
		if(listener.getIsGateway()) {
			listenerNode.setAttribute("gatewayClass", HibernateGatewayListener.class.getName());
			// Map EPR related attributes onto the listener - from the bus and provider and listener.
			// Note: This will change - the Gateways will also support the EPR element...
			mapHibernateEprProperties(listenerNode, provider, messageFilters);
			MapperUtil.mapEPRProperties(listener, listenerNode, model);
		} else {
			Element eprNode = YADOMUtil.addElement(listenerNode, ListenerTagNames.EPR_TAG);
			// Map EPR related attributes onto the EPR - from the bus and provider and listener...
			mapHibernateEprProperties(eprNode, provider, messageFilters);
			MapperUtil.mapEPRProperties(listener, eprNode, model);
			// Remove any empty attributes set on the EPR config...
			YADOMUtil.removeEmptyAttributes(eprNode);
		}

		// Remove any empty attributes set on the listener config...
		YADOMUtil.removeEmptyAttributes(listenerNode);

		return listenerNode;
	}

	/**
	 * mapHibernateEprProperties creates a ConfigTree with the details we need
	 * from the HibernateProvider and MessageFilters.
	 * @param toElement the element we are passing on
	 * @param provider the provider object
	 * @param messageFilters collection of MessageFilters we need to implement
	 */
	private static void mapHibernateEprProperties(Element toElement, HibernateProvider provider, List<HibernateMessageFilter> messageFilters) {
		toElement.setAttribute(HibernateEpr.HIBERNATE_CFG_TAG, provider.getHibernateCfgFile());
		for (HibernateMessageFilter messageFilter: messageFilters) {
			Element temp = YADOMUtil.addElement(toElement, "messagefilter");
			temp.setAttribute(HibernateEpr.CLASS_NAME_TAG, messageFilter.getClassname());
			temp.setAttribute(HibernateEpr.EVENT_TAG, messageFilter.getEvent());
			toElement.appendChild(temp);
		}
	}
}
