/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 */

package org.jboss.soa.esb.listeners.config.mappers131;

import java.net.URI;
import java.net.URISyntaxException;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.addressing.eprs.FTPEpr;
import org.jboss.soa.esb.addressing.eprs.FTPSEpr;
import org.jboss.soa.esb.addressing.eprs.SFTPEpr;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.config.xbeanmodel131.FtpBusDocument.FtpBus;
import org.jboss.soa.esb.listeners.config.xbeanmodel131.FtpListenerDocument.FtpListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel131.FtpMessageFilterDocument.FtpMessageFilter;
import org.jboss.soa.esb.listeners.config.xbeanmodel131.FtpMessageFilterDocument.FtpMessageFilter.Protocol;
import org.jboss.soa.esb.listeners.config.xbeanmodel131.FtpProviderDocument.FtpProvider;
import org.jboss.soa.esb.listeners.gateway.ReadOnlyRemoteGatewayListener;
import org.jboss.soa.esb.listeners.gateway.RemoteGatewayListener;
import org.w3c.dom.Element;

/**
 * Performs the mapping of a &lt;fs-listener&gt; XSD based configuration to the "ConfigTree"
 * style configuration, adding the "ConfigTree" listener config to the "root" node.
 *
 * @author <a href="mailto:kurt.stam@jboss.com">kurt.stam@jboss.com</a>
 */
public class FtpListenerMapper {

	/**
	 * Perform the mapping.
	 * @param root The "ConfigTree" configuration root node.
	 * @param listener The Jmslistener to be mapped into the ConfigTree.
	 * @param model The configuration model from which the mapping is being performed.
	 * @return The ConfigTree listener configuration node.
	 * @throws org.jboss.soa.esb.ConfigurationException Invalid listener configuration.
	 */
	public static Element map(Element root, FtpListener listener, XMLBeansModel model) throws ConfigurationException {
		Element listenerNode = YADOMUtil.addElement(root, "listener");
		FtpBus bus;
		FtpProvider provider;

        listenerNode.setAttribute("name", listener.getName());

		try {
			bus = (FtpBus) model.getBus(listener.getBusidref());
		} catch (ClassCastException e) {
			throw new ConfigurationException("Invalid busid reference [" + listener.getBusidref() + "] on listener [" + listener.getName() + "].  A <ftp-listener> must reference a <ftp-bus>.");
		}
		try {
			provider = (FtpProvider) model.getProvider(bus);
		} catch (ClassCastException e) {
			throw new ConfigurationException("Invalid bus config [" + listener.getBusidref() + "].  Should be contained within a <ftp-provider> instance.  Unexpected exception - this should have caused a validation error!");
		}

		FtpMessageFilter messageFilter = listener.getFtpMessageFilter();
		if(messageFilter == null) {
			messageFilter = bus.getFtpMessageFilter();
			if(messageFilter == null) {
				throw new ConfigurationException("No <ftp-detination> defined on either <ftp-listener> [" + listener.getName() + "] or <ftp-bus> [" + bus.getBusid() + "].");
			}
		}
		// Map the standard listener attributes - common across all listener types...
		MapperUtil.mapDefaultAttributes(listener, listenerNode, model);
		// Map the <property> elements targeted at the listener - from the listener itself.
		MapperUtil.mapProperties(listener.getPropertyList(), listenerNode);
		if(listener.getIsGateway()) {
			if ( messageFilter.getReadOnly() )
				listenerNode.setAttribute("gatewayClass", ReadOnlyRemoteGatewayListener.class.getName());
			else
				listenerNode.setAttribute("gatewayClass", RemoteGatewayListener.class.getName());
			// Map EPR related attributes onto the listener - from the bus and provider and listener.
			// Note: This will change - the Gateways will also support the EPR element...
			mapFtpEprProperties(listenerNode, provider, messageFilter);
			MapperUtil.mapEPRProperties(listener, listenerNode, model);
		} else {
			Element eprNode = YADOMUtil.addElement(listenerNode, ListenerTagNames.EPR_TAG);

			// Map EPR related attributes onto the EPR - from the bus and provider and listener...
			mapFtpEprProperties(eprNode, provider, messageFilter);
			MapperUtil.mapEPRProperties(listener, eprNode, model);
			// Remove any empty attributes set on the EPR config...
			YADOMUtil.removeEmptyAttributes(eprNode);
		}

		// Remove any empty attributes set on the listener config...
		YADOMUtil.removeEmptyAttributes(listenerNode);

		return listenerNode;
	}

	/**
	 *  The separator used by the ftp protocol to separate the
	 *  host:port section from the path
	 */
	private static final String FTP_SEPARATOR = "/";

	private static void mapFtpEprProperties(Element toElement, FtpProvider provider, FtpMessageFilter messageFilter)
		throws ConfigurationException
	{
		String inputDir = messageFilter.getDirectory();
		if ( ! inputDir.startsWith( FTP_SEPARATOR ) )
			inputDir = FTP_SEPARATOR + inputDir;

		final String filterPassword = messageFilter.getPassword() ;
		final String username = encodeUsername(messageFilter.getUsername()) ;
		final String userInfo = (filterPassword == null ? username : username + ":" + filterPassword) ;
		final String origHostname = provider.getHostname() ;
		final int colonLocation = origHostname.indexOf(':') ;
		final String hostname ;
		final int port ;
		if (colonLocation > -1)
		{
			hostname = origHostname.substring(0, colonLocation) ;
			final int portLocation = colonLocation+1 ;
			if (portLocation < origHostname.length())
			{
				try
				{
					port = Integer.parseInt(origHostname.substring(portLocation)) ;
				}
				catch (final NumberFormatException nfe)
				{
					throw new ConfigurationException("Failed to parse port value") ;
				}
			}
			else
			{
				port = -1 ;
			}
		}
		else
		{
			hostname = origHostname ;
			port = -1 ;
		}
		final URI uri ;
		try
		{
			uri = new URI(messageFilter.getProtocol().toString(),
				userInfo, hostname, port, inputDir, null, null) ;
		}
		catch (final URISyntaxException urise)
		{
			throw new ConfigurationException("Unexpected exception creating FTP URI, " + urise.getMessage()) ;
		}
		toElement.setAttribute(ListenerTagNames.URL_TAG, uri.toASCIIString());
		toElement.setAttribute(FTPEpr.INPUT_SUFFIX_TAG, messageFilter.getInputSuffix());
		toElement.setAttribute(FTPEpr.WORK_SUFFIX_TAG, messageFilter.getWorkSuffix());
		toElement.setAttribute(FTPEpr.POST_DEL_TAG, String.valueOf(messageFilter.getPostDelete()));
		toElement.setAttribute(FTPEpr.POST_RENAME_TAG, String.valueOf(messageFilter.getPostRename()));

		if (!messageFilter.getPostDelete()) {
			toElement.setAttribute(FTPEpr.POST_DIR_TAG, messageFilter.getPostDirectory());
			toElement.setAttribute(FTPEpr.POST_SUFFIX_TAG, messageFilter.getPostSuffix());
		}
		toElement.setAttribute(FTPEpr.ERROR_DEL_TAG, String.valueOf(messageFilter.getErrorDelete()));
		if (!messageFilter.getErrorDelete()) {
			toElement.setAttribute(FTPEpr.ERROR_DIR_TAG, messageFilter.getErrorDirectory());
			toElement.setAttribute(FTPEpr.ERROR_SUFFIX_TAG, messageFilter.getErrorSuffix());
		}
		toElement.setAttribute(FTPEpr.PASSIVE_TAG, String.valueOf(messageFilter.getPassive()));
		if (Protocol.SFTP.equals(messageFilter.getProtocol())) {
			toElement.setAttribute(SFTPEpr.CERTIFICATE_TAG, messageFilter.getCertificateUrl());
			toElement.setAttribute(SFTPEpr.PASSPHRASE_TAG, messageFilter.getCertificatePassphrase()) ;
		}
		if (Protocol.FTPS.equals(messageFilter.getProtocol())) {
			toElement.setAttribute(FTPSEpr.CERTIFICATE_TAG, messageFilter.getCertificateUrl());
			toElement.setAttribute(FTPSEpr.CERTIFICATE_NAME_TAG, messageFilter.getCertificateName());
		}
	}
	
	private static String encodeUsername(final String username)
	{
		if (username != null)
		{
			int colonPosn = username.indexOf(':') ;
			if (colonPosn > -1)
			{
				int currentIndex = 0 ;
				final StringBuilder sb = new StringBuilder() ;
				while(colonPosn > 0)
				{
					sb.append(username.substring(currentIndex, colonPosn)) ;
					sb.append("%3A") ;
					currentIndex = colonPosn+1 ;
					colonPosn = username.indexOf(':', currentIndex) ;
				}
				if (currentIndex < username.length())
				{
					sb.append(username.substring(currentIndex)) ;
				}
				return sb.toString() ;
			}
		}
		return username ;
	}
}
