/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.listeners.config.model;

import java.io.Reader;
import java.util.HashMap;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.validation.Schema;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.util.XMLHelper;
import org.jboss.internal.soa.esb.util.stax.StreamHelper;
import org.jboss.soa.esb.common.Configuration;
import org.xml.sax.ErrorHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

/**
 * Handle parsing of different schemas into model adapter.
 * 
 * <a href='mailto:Kevin.Conner@jboss.com'>Kevin Conner</a>
 */
public class ModelParser
{
    /**
     * Map of namespaces to schema parsers.
     */
    private static final Map<String, SchemaParser> SCHEMA_PARSERS = new HashMap<String, SchemaParser>() ;
    
    /**
     * The schema to validate against.
     */
    private static Schema SCHEMA ;
    
    /**
     * The logger for this class.
     */
    private static Logger LOG = Logger.getLogger(XmlErrorHandler.class) ;
    
    static
    {
        SCHEMA_PARSERS.put(Model101SchemaParser.NAMESPACE, new Model101SchemaParser()) ;
        SCHEMA_PARSERS.put(Model110SchemaParser.NAMESPACE, new Model110SchemaParser()) ;
        SCHEMA_PARSERS.put(Model120SchemaParser.NAMESPACE, new Model120SchemaParser()) ;
        SCHEMA_PARSERS.put(Model130SchemaParser.NAMESPACE, new Model130SchemaParser()) ;
        SCHEMA_PARSERS.put(Model131SchemaParser.NAMESPACE, new Model131SchemaParser()) ;

        Schema schema = null ;
        try
        {
            schema = XMLHelper.getSchema(new String[] {Model101SchemaParser.SCHEMA,
                Model110SchemaParser.SCHEMA, Model120SchemaParser.SCHEMA, Model130SchemaParser.SCHEMA,
                Model131SchemaParser.SCHEMA}) ;
        }
        catch (SAXException saxe)
        {
            LOG.warn("Unexpected exception loading schemas", saxe) ;
        }
        SCHEMA = schema ;
    }
    
    /**
     * The parser singleton.
     */
    private static final ModelParser PARSER = new ModelParser() ;
    
    private final DocumentBuilderFactory documentBuilderFactory = createDocumentBuilderFactory();
    
    /**
     * Get the model parser singleton.
     * @return The model parser singleton.
     */
    public static ModelParser getParser()
    {
        return PARSER ;
    }
    
    /**
     * Get a model adapter from the input.
     * @param config The configuration input.
     * @return The appropriate model adapter.
     */
    public ModelAdapter parse(final Reader config)
        throws ModelException
    {
        final XMLStreamReader reader ;
        try
        {
            reader = XMLHelper.getXMLStreamReader(config) ;
            StreamHelper.skipToStartElement(reader) ;
        }
        catch (final XMLStreamException xmlse)
        {
            throw new ModelException("Error parsing schema", xmlse) ;
        }
        
        final QName name = reader.getName() ;
        final String namespace = name.getNamespaceURI() ;
        final SchemaParser parser = SCHEMA_PARSERS.get(namespace) ;
        if (parser != null)
        {
            return parser.parse(reader) ;
        }
        else
        {
            throw new ModelException("Unsupported schema namespace: " + namespace) ;
        }
    }

    /**
     * Validate a model from the input.
     * @param config The configuration input.
     * @return true if valid, false otherwise.
     */
    public boolean validate(Reader config)
    {
        if (documentBuilderFactory == null)
        {
            return true ;
        }
        
        try
        {
            final DocumentBuilder parser = documentBuilderFactory.newDocumentBuilder() ;
            final XmlErrorHandler errorHandler = new XmlErrorHandler() ;
            parser.setErrorHandler(errorHandler) ;
            parser.parse(new InputSource(config)); 
            return errorHandler.isValid() ;
        }
        catch (final Exception ex)
        {
            LOG.error("Unexpected exception", ex);
            return false ;
        }
    }

    private static DocumentBuilderFactory createDocumentBuilderFactory()
    {
        final boolean deploymentSchemaValidation = Boolean.parseBoolean(Configuration.getDeploymentSchemaValidation()) ;
        if (deploymentSchemaValidation)
        {
            final DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            factory.setNamespaceAware(true) ;
            factory.setSchema(SCHEMA) ;
            return factory;
        }
        else
        {
            return null ;
        }
    }

    /**
     * The XML Error handler for validation.
     * @author <href="mailto:kevin.conner@jboss.com">Kevin Conner</a>
     *
     */
    private static class XmlErrorHandler implements ErrorHandler 
    {
        /**
         * The valid flag.
         */
        private boolean valid = true ;
        
        public void warning(final SAXParseException exception)
            throws SAXException
        {
            LOG.warn("WARNING-"+exception.getSystemId()+"-"+exception.getLineNumber()+":"+exception.getMessage()) ;
        }

        public void error(final SAXParseException exception)
            throws SAXException
        {
            LOG.error("ERROR-"+exception.getSystemId()+"-"+exception.getLineNumber()+":"+exception.getMessage()) ;
            valid = false ;
        }

        public void fatalError(final SAXParseException exception)
            throws SAXException
        {
            LOG.fatal("FATAL-"+exception.getSystemId()+"-"+exception.getLineNumber()+":"+exception.getMessage()) ;
            valid = false ;
        }
        
        boolean isValid()
        {
            return valid ;
        }
    }
}
