/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners.gateway.mina;

import java.io.IOException;
import java.net.InetSocketAddress;
import org.apache.log4j.Logger;
import org.apache.mina.core.service.IoAcceptor;
import org.apache.mina.core.service.IoHandler;
import org.apache.mina.core.session.IdleStatus;
import org.apache.mina.core.session.IoSession;
import org.apache.mina.transport.socket.DatagramAcceptor;
import org.apache.mina.transport.socket.nio.NioDatagramAcceptor;
import org.apache.mina.transport.socket.nio.NioSocketAcceptor;
import org.jboss.internal.soa.esb.util.MessageFlowContext;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.lifecycle.AbstractManagedLifecycle;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;

/**
 * UDP gateway listener implementation for receiving ESB unaware messages.
 * <p/>
 * This gateway will set up a netty server and listen for incoming connection request. The 
 * requests will be passed to the Netty channel pipeline configured and the last handler will
 * invoke the target ESB service.
 * 
 * Example configuration:
 * <pre>{@code
 * <listeners>
 *    <udp-listener name="udp-listener" host="localhost" port="9999" handlerClass="org.jboss.soa.esb.listeners.gateway.mina.DefaultMessageHandler" is-gateway="true"/>
 * </listeners>
 * }</pre>
 * <lu>
 * <li>{@code host} the host to listen to.</li>
 * <li>{@code port} the port to listen on.</li>
 * <li>{@code handlerClass} A class that implemtents {@link MessageHandler}. If not specified will default to {@link DefaultMessageHandler}.</li>
 * <li>{@code is-gateway} is always true as udp cannot be used as a bus transport at the moment.</li>
 * </lu>
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 */
public class UdpGatewayListener extends AbstractManagedLifecycle
{
    /**
     * Class Logger instance.
     */
    private Logger log = Logger.getLogger(UdpGatewayListener.class);
    
    /**
     * The local address this gateway listeners to.
     */
    private InetSocketAddress socketAddress;
    
    /**
     * MINA IoAcceptor.
     */
    private IoAcceptor datagramAcceptor;
    
    /**
     * The target service category that this gateway should dispatch to.
     */
    private String serviceCategory;

    /**
     * The target service name that this gateway should dispatch to.
     */
    private String serviceName;
    
    /**
     * The {@link Channel} that this gateway is bound to.
     */
    private MessageHandler messageHandler;
    /**
     * Message flow priority.
     */
    private final Integer messageFlowPriority ;
    
    /**
     * Sole constructor.
     * 
     * @param config The configuration associated with this instance. Must not be null.
     * @throws ConfigurationException if a configuration errors during initialisation.
     */
    public UdpGatewayListener(final ConfigTree config) throws ConfigurationException 
    {
        super(config);
        final UdpGatewayConfig udpConfig = new UdpGatewayConfig(config);
        socketAddress = udpConfig.getSocketAddress();
        serviceCategory = udpConfig.getServiceCategory();
        serviceName = udpConfig.getServiceName();
        messageHandler = udpConfig.getHandler();
        messageFlowPriority = MessageFlowContext.parseMessageFlowPriority(config) ;
    }

    /**
     * Initialize will bind to the configured host and port. 
     * This is done in initialize so that the time this takes is done upon 
     * deployment of the gateway instead of during the first call to the 
     * gateway.
     */
    @Override
    protected void doInitialise() throws ManagedLifecycleException 
    {
        final ServiceInvoker serviceInvoker = createServiceInvoker(serviceCategory, serviceName);
        messageHandler.setServiceInvoker(serviceInvoker);
        
        datagramAcceptor = new NioDatagramAcceptor(); 
        
        try
        {
        	datagramAcceptor.setHandler(new MessageFlowPriorityIoAdapter(messageHandler, messageFlowPriority));
            datagramAcceptor.bind(socketAddress);
        } 
        catch (final IOException e)
        {
            throw new ManagedLifecycleException("IOException while trying to bind UdpListerner to '" + socketAddress + "'. Exception was:" + e);
        }
        log.info("Started " + toString());
    }

    /**
     * Currently does nothing.
     */
    @Override
    protected void doStart() throws ManagedLifecycleException
    {
        // NoOp
    }
    
    /**
     * Unbinds the udp listener.
     */
    @Override
    protected void doStop() throws ManagedLifecycleException 
    {
        if(socketAddress != null) 
        {
            datagramAcceptor.unbind(socketAddress);
        }
        log.info("Stopped " + toString());
    }

    /**
     * Currently does nothing.
     */
    @Override
    protected void doDestroy() throws ManagedLifecycleException 
    {
        // NoOp
    }

    /**
     * Returns a string representation in the form:
     * 
     * [UdpGatewayListener] Started UdpGatewayListener [address=localhost/127.0.0.1:9999, 
     * targetService=UdpServiceCategory:Service1, 
     * messageHandler=org.jboss.soa.esb.listeners.gateway.mina.DefaultMessageHandler@3c205c]
     * 
     * 
     * @return String   A string representation of this instance.
     */
    @Override
    public String toString()
    {
        return "UdpGatewayListener [address=" + socketAddress + ", targetService=" + serviceCategory + ":" + serviceName + ", messageHandler=" + messageHandler + "]";
    }

    /**
     * Creates a new ServiceInvoker using the serviceCategory and serviceName.
     * 
     * @param serviceCategory   The service category.
     * @param serviceName       The service name.
     * @return ServiceInvoker   The newly created {@link ServiceInvoker}. 
     * 
     * @throws ManagedLifecycleException If the service invoker could not be created.
     */
    private ServiceInvoker createServiceInvoker(final String serviceCategory, final String serviceName) throws ManagedLifecycleException
    {
        try
        {
            return new ServiceInvoker(serviceCategory, serviceName);
        } 
        catch (MessageDeliverException e)
        {
            throw new ManagedLifecycleException(e.getMessage(), e);
        }
    }
    
    private static final class MessageFlowPriorityIoAdapter implements IoHandler
    {
        private final IoHandler handler ;
        private final Integer messageFlowPriority ;
        
        MessageFlowPriorityIoAdapter(final IoHandler handler, final Integer messageFlowPriority)
        {
            this.handler = handler ;
            this.messageFlowPriority = messageFlowPriority ;
        }
        
        @Override
        public void messageReceived(final IoSession session, final Object message)
            throws Exception
        {
            MessageFlowContext.setMessageFlowPriority(messageFlowPriority) ;
            try
            {
                handler.messageReceived(session, message) ;
            }
            finally
            {
                MessageFlowContext.setMessageFlowPriority(null) ;
            }
        }

        public void exceptionCaught(final IoSession session, final Throwable throwable)
                throws Exception
        {
            handler.exceptionCaught(session, throwable);
        }

        public void messageSent(final IoSession session, final Object message) throws Exception
        {
            handler.messageSent(session, message);
        }

        public void sessionClosed(final IoSession session) throws Exception
        {
            handler.sessionClosed(session);
        }

        public void sessionCreated(final IoSession session) throws Exception
        {
            handler.sessionCreated(session);
        }

        public void sessionIdle(final IoSession session, final IdleStatus idleStatus)
                throws Exception
        {
            handler.sessionIdle(session, idleStatus);
        }

        public void sessionOpened(final IoSession session) throws Exception
        {
            handler.sessionOpened(session);
        }
    }
}
