/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.soa.esb.listeners.jca;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;

import javax.resource.ResourceException;
import javax.resource.spi.endpoint.MessageEndpointFactory;
import javax.transaction.Status;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;
import javax.transaction.xa.XAResource;

import org.apache.log4j.Logger;
import org.jboss.tm.TransactionManagerLocator;

import EDU.oswego.cs.dl.util.concurrent.SynchronizedBoolean;

/**
 * comment
 *
 * @author <a href="bill@jboss.com">Bill Burke</a>
 * @version $Revision: 1.1 $
 */
public class EndpointProxy implements InvocationHandler
{
   private static final Logger log = Logger.getLogger(EndpointProxy.class);

   /**
    * Whether trace is enabled
    */
   private boolean trace = log.isTraceEnabled();

   /**
    * Whether this proxy has been released
    */
   protected SynchronizedBoolean released = new SynchronizedBoolean(false);

   /**
    * Whether we have delivered a message
    */
   protected boolean delivered = false;

   /**
    * The in use thread
    */
   protected Thread inUseThread = null;

   /**
    * The old classloader of the thread
    */
   protected ClassLoader oldClassLoader = null;
   
   /**
    * The flag indicating that beforeDelivery was invoked.
    */
   protected boolean beforeDeliveryInvoked ;

   /**
    * Any transaction we started
    */
   protected Transaction transaction = null;

   /**
    * Any suspended transaction
    */
   protected Transaction suspended = null;

   protected ClassLoader loader;

   private XAResource resource;
   private MessageEndpointFactory messageEndpointFactory;
   private EndpointContainer container;


   public void setContainer(EndpointContainer container)
   {
      this.container = container;
   }

   public void setMessageEndpointFactory(MessageEndpointFactory messageEndpointFactory)
   {
      this.messageEndpointFactory = messageEndpointFactory;
   }

   public void setXaResource(XAResource resource)
   {
      this.resource = resource;
   }


   public void setLoader(ClassLoader loader)
   {
      this.loader = loader;
   }

   public Object invoke(Object proxy, Method method, Object[] args)
           throws Throwable
   {
      // Are we still useable?
      if (released.get())
         throw new IllegalStateException("This message endpoint + " + getProxyString(proxy) + " has been released");

      // Concurrent invocation?
      Thread currentThread = Thread.currentThread();
      if (inUseThread != null && inUseThread.equals(currentThread) == false)
         throw new IllegalStateException("This message endpoint + " + getProxyString(proxy) + " is already in use by another thread " + inUseThread);
      inUseThread = currentThread;

      if (trace)
         log.trace("MessageEndpoint " + getProxyString(proxy) + " in use by " + method + " " + inUseThread);

      // Which operation?
      final String name = method.getName() ;
      if ("release".equals(name))
      {
         release(proxy);
         return null;
      }
      else if ("beforeDelivery".equals(name))
      {
         before(proxy, method, args);
         return null;
      }
      else if ("afterDelivery".equals(name))
      {
         after(proxy);
         return null;
      }
      else if ("toString".equals(name))
      {
          return getProxyString(proxy) ;
      }
      else if ("equals".equals(name))
      {
          return (args.length == 1) && (args[0] == this) ;
      }
      else if ("hashCode".equals(name))
      {
          return hashCode() ;
      }
      else
         return delivery(proxy, method, args);
   }

   // -----------------------------------------------------------

   protected void release(Object proxy) throws Throwable
   {
      // We are now released
      released.set(true);

      if (trace)
         log.trace("MessageEndpoint " + getProxyString(proxy) + " released");

      // Tidyup any outstanding delivery
      if (beforeDeliveryInvoked)
      {
         try
         {
            finish("release", proxy, false);
         }
         catch (Throwable t)
         {
            log.warn("Error in release ", t);
         }
      }
   }

   protected void before(Object proxy, Method method, Object[] args) throws Throwable
   {
      // Called out of sequence
      if (beforeDeliveryInvoked)
         throw new IllegalStateException("Missing afterDelivery from the previous beforeDelivery for message endpoint " + getProxyString(proxy));
      beforeDeliveryInvoked = true ;

      if (trace)
         log.trace("MessageEndpoint " + getProxyString(proxy) + " released");

      // Set the classloader
      oldClassLoader = inUseThread.getContextClassLoader();
      inUseThread.setContextClassLoader(loader);
      if (trace)
         log.trace("MessageEndpoint " + getProxyString(proxy) + " set context classloader to " + loader);

      // start any transaction
      try
      {
         // Is the delivery transacted?
         Method beforeMethod = (Method) args[0];
         boolean isTransacted = messageEndpointFactory.isDeliveryTransacted(beforeMethod);

         startTransaction("beforeDelivery", proxy, method, args, isTransacted);
      }
      catch (Throwable t)
      {
         resetContextClassLoader(proxy);
         throw new ResourceException(t);
      }
   }

   protected void after(Object proxy) throws Throwable
   {
      // Called out of sequence
      if (!beforeDeliveryInvoked)
         throw new IllegalStateException("afterDelivery without a previous beforeDelivery for message endpoint " + getProxyString(proxy));
      beforeDeliveryInvoked = false ;

      // Finish this delivery committing if we can
      try
      {
         finish("afterDelivery", proxy, true);
      }
      catch (Throwable t)
      {
         throw new ResourceException(t);
      }
   }

   protected Object delivery(Object proxy, Method method, Object[] args) throws Throwable
   {
      // Have we already delivered a message?
      if (delivered)
         throw new IllegalStateException("Multiple message delivery between before and after delivery is not allowed for message endpoint " + getProxyString(proxy));

      if (trace)
         log.trace("MessageEndpoint " + getProxyString(proxy) + " delivering");

      final Thread currentThread = Thread.currentThread() ;
      final ClassLoader contextClassLoader = currentThread.getContextClassLoader() ;

      // Mark delivery if beforeDelivery was invoked
      if (beforeDeliveryInvoked)
      {
         delivered = true;
      }
      else
      {
         currentThread.setContextClassLoader(loader) ;
      }

      boolean commit = true;
      try
      {
         // Check for starting a transaction
         if (!beforeDeliveryInvoked)
         {
            boolean isTransacted = messageEndpointFactory.isDeliveryTransacted(method);
            startTransaction("delivery", proxy, method, args, isTransacted);
         }
         return container.invoke(method, args);
      }
      catch (Throwable t)
      {
         if (trace)
            log.trace("MessageEndpoint " + getProxyString(proxy) + " delivery error", t);
         if (t instanceof Error || t instanceof RuntimeException)
         {
            if (transaction != null)
               transaction.setRollbackOnly();
            commit = false;
         }
         throw t;
      }
      finally
      {
         // No before/after delivery, end any transaction and release the lock
         if (!beforeDeliveryInvoked)
         {
            currentThread.setContextClassLoader(contextClassLoader) ;
            try
            {
               // Finish any transaction we started
               endTransaction(proxy, commit);
            }
            finally
            {
               releaseThreadLock(proxy);
            }
         }
      }
   }

   protected void finish(String context, Object proxy, boolean commit) throws Throwable
   {
      try
      {
         endTransaction(proxy, commit);
      }
      finally
      {
         // Reset delivered flag
         delivered = false;
         // Change back to the original context classloader
         resetContextClassLoader(proxy);
         // We no longer hold the lock
         releaseThreadLock(proxy);
      }
   }

   protected void startTransaction(String context, Object proxy, Method m, Object[] args, boolean isTransacted) throws Throwable
   {
      Method method;

      // Normal delivery
      if ("delivery".equals(context))
         method = m;
         // Before delivery
      else
         method = (Method) args[0];

      if (trace)
         log.trace("MessageEndpoint " + getProxyString(proxy) + " " + context + " method=" + method + " xaResource=" + resource + " transacted=" + isTransacted);

      // Get the transaction status
      TransactionManager tm = TransactionManagerLocator.getInstance().locate();
      suspended = tm.suspend();

      if (trace)
         log.trace("MessageEndpoint " + getProxyString(proxy) + " " + context + " currentTx=" + suspended);

      // Delivery is transacted
      if (isTransacted)
      {
         // No transaction means we start a new transaction and enlist the resource
         if (suspended == null)
         {
            tm.begin();
            transaction = tm.getTransaction();
            if (trace)
               log.trace("MessageEndpoint " + getProxyString(proxy) + " started transaction=" + transaction);

            // Enlist the XAResource in the transaction
            if (resource != null)
            {
               transaction.enlistResource(resource);
               if (trace)
                  log.trace("MessageEndpoint " + getProxyString(proxy) + " enlisted=" + resource);
            }
         }
         else
         {
            // If there is already a transaction we ignore the XAResource (by spec 12.5.9)
            try
            {
               tm.resume(suspended);
            }
            finally
            {
               suspended = null;
               if (trace)
                  log.trace("MessageEndpoint " + getProxyString(proxy) + " transaction=" + suspended + " already active, IGNORED=" + resource);
            }
         }
      }
   }

   protected void endTransaction(Object proxy, boolean commit) throws Throwable
   {
      TransactionManager tm = null;
      Transaction currentTx = null;
      try
      {
         // If we started the transaction, commit it
         if (transaction != null)
         {
            tm = TransactionManagerLocator.getInstance().locate();
            currentTx = tm.getTransaction();

            // Suspend any bad transaction - there is bug somewhere, but we will try to tidy things up
            if (currentTx != null && currentTx.equals(transaction) == false)
            {
               log.warn("Current transaction " + currentTx + " is not the expected transaction.");
               tm.suspend();
               tm.resume(transaction);
            }
            else
            {
               // We have the correct transaction
               currentTx = null;
            }

            // Commit or rollback depending on the status
            if (commit == false || transaction.getStatus() == Status.STATUS_MARKED_ROLLBACK)
            {
               if (trace)
                  log.trace("MessageEndpoint " + getProxyString(proxy) + " rollback");
               tm.rollback();
            }
            else
            {
               if (trace)
                  log.trace("MessageEndpoint " + getProxyString(proxy) + " commit");
               tm.commit();
            }
         }

         // If we suspended the incoming transaction, resume it
         if (suspended != null)
         {
            try
            {
               tm = TransactionManagerLocator.getInstance().locate();
               tm.resume(suspended);
            }
            finally
            {
               suspended = null;
            }
         }
      }
      finally
      {
         // Resume any suspended transaction
         if (currentTx != null)
         {
            try
            {
               tm.resume(currentTx);
            }
            catch (Throwable t)
            {
               log.warn("MessageEndpoint " + getProxyString(proxy) + " failed to resume old transaction " + currentTx);

            }
         }
      }
   }

   protected void resetContextClassLoader(Object proxy)
   {
      if (trace)
         log.trace("MessageEndpoint " + getProxyString(proxy) + " reset classloader " + oldClassLoader);
      inUseThread.setContextClassLoader(oldClassLoader);
      oldClassLoader = null;
   }

   protected void releaseThreadLock(Object proxy)
   {
      if (trace)
         log.trace("MessageEndpoint " + getProxyString(proxy) + " no longer in use by " + inUseThread);
      inUseThread = null;
   }

   protected String getProxyString(Object proxy)
   {
      return container.getDescription();
   }

}
