/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.lifecycle;

/**
 * This enumeration represents the lifecycle state of a managed instance.
 * <p/>
 * Allowable transitions are as follows
 * <table border="1">
 * <thead>
 * <tr><th>Originating state</th><th>New states</th></tr>
 * </thead>
 * <tbody>
 * <tr><td>CONSTRUCTED</td><td>INITIALISING</td></tr>
 * <tr><td>INITIALISING</td><td>INITIALISED, DESTROYED</td></tr>
 * <tr><td>INITIALISED</td><td>STARTING, DESTROYED</td></tr>
 * <tr><td>STARTING</td><td>STARTED, STOPPED</td></tr>
 * <tr><td>STARTED</td><td>RUNNING, STOPPING</td></tr>
 * <tr><td>RUNNING</td><td>STOPPING</td></tr>
 * <tr><td>STOPPING</td><td>STOPPED</td></tr>
 * <tr><td>STOPPED</td><td>STARTING, DESTROYING</td></tr>
 * <tr><td>DESTROYING</td><td>DESTROYED</td></tr>
 * </tbody>
 * </table>
 * 
 * @author kevin
 */
public enum ManagedLifecycleState
{
    CONSTRUCTED { public boolean canTransition(final ManagedLifecycleState newState) { return (newState == INITIALISING) ;}} ,
    INITIALISING { public boolean canTransition(final ManagedLifecycleState newState) { return ((newState == INITIALISED) || (newState == DESTROYED)) ;}} ,
    INITIALISED { public boolean canTransition(final ManagedLifecycleState newState) { return ((newState == STARTING) || (newState == DESTROYING)) ;}} ,
    STARTING { public boolean canTransition(final ManagedLifecycleState newState) { return ((newState == STARTED) || (newState == STOPPED)) ;}} ,
    STARTED { public boolean canTransition(final ManagedLifecycleState newState) { return ((newState == RUNNING) || (newState == STOPPING)) ;}} ,
    RUNNING { public boolean canTransition(final ManagedLifecycleState newState) { return (newState == STOPPING) ;}} ,
    STOPPING { public boolean canTransition(final ManagedLifecycleState newState) { return (newState == STOPPED) ;}} ,
    STOPPED { public boolean canTransition(final ManagedLifecycleState newState) { return ((newState == STARTING) || (newState == DESTROYING)) ;}} ,
    DESTROYING { public boolean canTransition(final ManagedLifecycleState newState) { return (newState == DESTROYED) ;}} ,
    DESTROYED  { public boolean canTransition(final ManagedLifecycleState newState) { return false ;}} ;
    
    public abstract boolean canTransition(final ManagedLifecycleState newState) ;
}
