/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.message;

import java.lang.reflect.Constructor;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionLifecycleException;
import org.jboss.soa.esb.actions.ActionPipelineProcessor;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;

/**
 * Instance of ActionPipelineProcessor to allow dynamic specification of the methods
 * on the action processor.
 * @author kevin
 */
final class OverriddenActionProcessor implements ActionPipelineProcessor
{
    /**
     * The method info.
     */
    private final ActionProcessorMethodInfo methodInfo ;
    /**
     * The action constructor.
     */
    private final Constructor constructor ;
    /**
     * The action config.
     */
    private final ConfigTree config ;
    
    /**
     * The map of messages to associated instances.
     */
    private Map<Message, Object> messageInstanceMap = Collections.synchronizedMap(new HashMap<Message, Object>()) ;

    /**
     * Construct the overridden action processor.
     * @param config The action config.
     * @param actionClass The action class.
     * @throws ConfigurationException for errors during configuration.
     */
    OverriddenActionProcessor(final ConfigTree config, final Class actionClass)
        throws ConfigurationException
    {
        this.config = config ;
        constructor = ActionProcessorMethodInfo.getActionClassConstructor(actionClass) ;
        methodInfo = ActionProcessorMethodInfo.getMethodInfo(config, actionClass) ;
    }
    
    /**
     * Perform the action processing on the specified message.
     * This method is invoked for each message passing throught the pipeline.
     * 
     * @param message The current message being processed.
     * @return The message to be passed to the next stage of the pipeline.
     * @throws ActionProcessingException for errors during processing.
     */
    public Message process(final Message message)
        throws ActionProcessingException
    {
        if (methodInfo.hasProcessMethods())
        {
            final Object instance ;
            try
            {
                instance = ActionProcessorMethodInfo.getActionClassInstance(config, constructor) ;
            }
            catch (final ConfigurationException ce)
            {
                throw new ActionProcessingException("Unexpected exception creating action class instance", ce) ;
            }
            
            messageInstanceMap.put(message, instance) ;
            return methodInfo.processMethods(instance, message) ;
        }
        else
        {
            return message ;
        }
    }
    
    /**
     * Process an exception generated by the pipeline processing.
     * Invoked when the processing of a subsequent stage of the
     * pipeline generates an exception.
     * 
     * @param message The original message.
     * @param th The throwable raised by the pipeline processing
     */
    public void processException(final Message message, final Throwable th)
    {
        final Object instance = messageInstanceMap.remove(message) ;
        if (instance != null)
        {
            methodInfo.processException(instance, message, th) ;
        }
    }
    
    /**
     * Process a successful pipeline process. 
     * Invoked when the pipeline processing completes successfully.
     * 
     * @param message The original message.
     */
    public void processSuccess(final Message message)
    {
        final Object instance = messageInstanceMap.remove(message) ;
        if (instance != null)
        {
            methodInfo.processSuccess(instance, message) ;
        }
    }
    
    /**
     * Initialise the action instance.
     * <p/>
     * This method is called after the action instance has been instantiated so that
     * configuration options can be validated.
     * 
     * @throws ActionLifecycleException for errors during initialisation.
     */
    public void initialise()
        throws ActionLifecycleException
    {
        messageInstanceMap.clear() ;
    }

    /**
     * Destroy the action instance.
     * <p/>
     * This method is called prior to the release of the action instance.  All
     * resources associated with this action instance should be released as the
     * instance will no longer be used.
     */
    public void destroy()
        throws ActionLifecycleException
    {
        messageInstanceMap.clear() ;
    }
}