/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.listeners.message;

import java.util.HashMap;
import java.util.Map;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.lifecycle.LifecyclePriorities;
import org.jboss.soa.esb.lifecycle.LifecycleResource;
import org.jboss.soa.esb.lifecycle.LifecycleResourceException;
import org.jboss.soa.esb.lifecycle.LifecycleResourceFactory;
import org.jboss.soa.esb.lifecycle.LifecycleResourceManager;
import org.jboss.soa.esb.listeners.ListenerTagNames;

/**
 */
public class ServiceMessageCounterLifecycleResource
{
    /**
     * The lifecycle resource factory for service message counters.
     */
    private static final LifecycleResourceFactory<Map<String, ServiceMessageCounter>> lifecycleServiceMessageCounterFactory = new LifecycleServiceMessageCounterFactory();
    /**
     * The lifecycle resources for service message counters.
     */
    private static final LifecycleResource<Map<String, ServiceMessageCounter>> lifecycleServiceMessageCounters =
        new LifecycleResource<Map<String, ServiceMessageCounter>>(lifecycleServiceMessageCounterFactory, LifecyclePriorities.SERVICE_MESSAGE_COUNTER_PRIORITY);
    
    /**
     * The logger for the service message counter lifecycle resource.
     */
    private static final Logger LOGGER = Logger.getLogger(ServiceMessageCounterLifecycleResource.class) ;
    
    /**
     * Get the service message counter for the specified config tree.
     * @param configTree The configTree 
     * @return The service message counter.
     * throws LifecycleResourceException for errors accessing the lifecycle context.
     */
    public static ServiceMessageCounter getServiceMessageCounter(final ConfigTree configTree)
        throws LifecycleResourceException
    {
        final String objectName = getObjectName(configTree) ;
        final Map<String, ServiceMessageCounter> contextCounters = lifecycleServiceMessageCounters.getLifecycleResource() ;
        
        synchronized(contextCounters)
        {
            final ServiceMessageCounter serviceMessageCounter = contextCounters.get(objectName) ;
            if (serviceMessageCounter != null)
            {
                return serviceMessageCounter ;
            }
            final ServiceMessageCounter newServiceMessageCounter = new ServiceMessageCounter(configTree, objectName) ;
            newServiceMessageCounter.registerMBean() ;
            contextCounters.put(objectName, newServiceMessageCounter) ;
            return newServiceMessageCounter ;
        }
    }
    
    /**
     * Create the object name associated with the configuration.
     * @param configTree The current configuration.
     * @return The object name.
     */
    private static String getObjectName(final ConfigTree configTree)
    {
        final String categoryName = configTree.getAttribute(ListenerTagNames.SERVICE_CATEGORY_NAME_TAG);
        final String serviceName = configTree.getAttribute(ListenerTagNames.SERVICE_NAME_TAG);
        final String targetCategoryName = configTree.getAttribute(ListenerTagNames.TARGET_SERVICE_CATEGORY_TAG);
        final String targetServiceName = configTree.getAttribute(ListenerTagNames.TARGET_SERVICE_NAME_TAG);
        final LifecycleResourceManager lifecycleResourceManager = LifecycleResourceManager.getSingleton() ;
        final String[] associatedDeployments = lifecycleResourceManager.getAssociatedDeployments() ;
        final String deployment ;
        if ((associatedDeployments != null) && (associatedDeployments.length == 1))
        {
            deployment = associatedDeployments[0] ;
        }
        else
        {
            deployment = lifecycleResourceManager.getIdentity() ;
        }

        final StringBuffer objectName = new StringBuffer("category=MessageCounter");
        append(objectName, "deployment", deployment) ;
        append(objectName, ListenerTagNames.SERVICE_CATEGORY_NAME_TAG, categoryName) ;
        append(objectName, ListenerTagNames.SERVICE_NAME_TAG, serviceName);
        append(objectName, ListenerTagNames.TARGET_SERVICE_CATEGORY_TAG, targetCategoryName);
        append(objectName, ListenerTagNames.TARGET_SERVICE_NAME_TAG, targetServiceName);
        
        return objectName.toString();
    }
    
    /**
     * Append the name/value to the object name.
     * @param objectName The current object name.
     * @param name The property name.
     * @param value The property value.
     */
    private static void append(final StringBuffer objectName, final String name, final String value)
    {
        if ((value != null) && (value.length() > 0)) {
            objectName.append(",").append(name).append("=").append(value) ;
        }
    }
    
    /**
     * The lifecycle resource factory
     * @author kevin
     */
    private static class LifecycleServiceMessageCounterFactory implements LifecycleResourceFactory<Map<String, ServiceMessageCounter>>
    {
        /**
         * Create a resource object which will be associated with the specified lifecycle identity.
         * @param lifecycleIdentity The associated lifecycle identity.
         * @return The lifecycle resource
         * @throws LifecycleResourceException for errors during construction.
         */
        public Map<String, ServiceMessageCounter> createLifecycleResource(final String lifecycleIdentity)
            throws LifecycleResourceException
        {
            return new HashMap<String, ServiceMessageCounter>() ;
        }

        /**
         * Destroy a resource object which is associated with the specified lifecycle identity.
         * @param resource The lifecycle resource.
         * @param lifecycleIdentity The associated lifecycle identity.
         * @return The lifecycle resource.
         * @throws LifecycleResourceException for errors during destroy.
         */
        public void destroyLifecycleResource(final Map<String, ServiceMessageCounter> resource,
            final String lifecycleIdentity)
            throws LifecycleResourceException
        {
            if (resource.size() > 0)
            {
                LOGGER.warn("Calling cleanup on existing service message counters for identity " + lifecycleIdentity) ;
                for (final ServiceMessageCounter counter: resource.values())
                {
                    counter.unregisterMBean() ;
                }
            }
        }
    }
}
