/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.security;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * This class is indended to security related configuration properties.
 *
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 * @since 4.4
 */
public class SecurityConfig
{
	private final String runAs;
	private final List<String> rolesAllowed;
	private final String useCallerIdentity;
	private final String moduleName;
	private final String callbackHandler;
	private Map<String,String> properties = new HashMap<String,String>();

	private SecurityConfig(
			final String runAs,
			final List<String> rolesAllowed,
			final String useCallerIdentity,
			final String moduleName,
			final String callbackHandler,
			final Map<String,String> properties)
	{
		this.runAs = runAs;
		this.rolesAllowed = rolesAllowed;
		this.useCallerIdentity = useCallerIdentity;
		this.moduleName = moduleName;
		this.callbackHandler= callbackHandler;
		this.properties = properties;
	}

	public String getCallbackHandler()
	{
		return callbackHandler;
	}

	public String getRunAs()
	{
		return runAs;
	}
	public String getUseCallerIdentity()
	{
		return useCallerIdentity;
	}

	public String getModuleName()
	{
		return moduleName;
	}

	public boolean hasRunAs()
	{
		return runAs != null;
	}

	public Map<String, String> getProperties()
	{
		return properties;
	}

	public List<String> getRolesAllowed()
    {
        return rolesAllowed;
    }

    @Override
    public String toString()
    {
    	return "[SecurityConfigInfo : runAs=" + runAs + ", useCallerIdentity=" + useCallerIdentity + ", moduleName=" + moduleName + ", callbackHandler=" + callbackHandler +", properties=" + properties +", rolesAllowed=" + rolesAllowed + "]";
    }

    public static class Builder
	{
    	private String runAs;
    	private String rolesAllowed;
    	private String useCallerIdentity;
    	private final String moduleName;
    	private String callbackHandler;
    	private final Map<String,String> properties = new HashMap<String,String>();

    	public Builder(final String moduleName)
    	{
    	    this.moduleName = moduleName;
    	}

    	public Builder runAs(final String runAs)
    	{
    	    this.runAs = runAs;
    	    return this;
    	}

    	public Builder rolesAllowed(final String rolesAllowed)
    	{
    	    this.rolesAllowed = rolesAllowed;
    	    return this;
    	}

    	public Builder useCallerIdentity(final String useCallerIdentity)
    	{
    	    this.useCallerIdentity = useCallerIdentity;
    	    return this;
    	}

    	public Builder callBackhandler(final String callbackHandler)
    	{
    	    this.callbackHandler = callbackHandler;
    	    return this;
    	}

    	public Builder property(final String name, final String value)
    	{
    	   properties.put(name, value);
    	   return this;
    	}

    	public SecurityConfig build()
    	{
    	    final List<String> rolesAllowedList = new ArrayList<String>();
    	    if ( rolesAllowed != null )
    	    {
    	       String[] split = rolesAllowed.split(",");
    	       List<String> asList = Arrays.asList(split);
    	       for (String roleName : asList)
               {
    	           if ( !"".equals(roleName))
            	       rolesAllowedList.add(roleName.trim());
               }
    	    }

    	    final List<String> unmodifiableRoles = Collections.unmodifiableList(rolesAllowedList);
    	    final Map<String,String> unmodifiableProperties = Collections.unmodifiableMap(properties);
    	    SecurityConfig securityConfig = new SecurityConfig(
    	            runAs,
    	            unmodifiableRoles,
    	            useCallerIdentity,
    	            moduleName,
    	            callbackHandler,
    	            unmodifiableProperties);

    	    return securityConfig;
    	}

	}

}
