/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.actions.converters;

import java.util.ArrayList;
import java.util.List;

import junit.framework.TestCase;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.actions.ActionUtils;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;

/**
 * ObjectToCSVString unit tests.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 * @since Version 4.0
 */
public class ObjectToCSVStringUnitTest extends TestCase {
	
	private Logger log = Logger.getLogger( ObjectToCSVStringUnitTest.class );

    public void test() throws ConfigurationException, ActionProcessingException {
        ConfigTree config;
        Message oMsg = MessageFactory.getInstance().getMessage();
        
        // Single value - property exist on bean...
        config = new ConfigTree("config");
        config.setAttribute(ObjectToCSVString.BEAN_PROPERTIES_PROP, "name");
        ObjectToCSVString processor = new ObjectToCSVString(config);
        
        oMsg.getBody().add(new TestBean());
        oMsg = processor.process(oMsg);
        assertEquals("Tom Fennelly", oMsg.getBody().get());

        // Multi value - properties all exist on bean...
        config = new ConfigTree("config");
        config.setAttribute(ObjectToCSVString.BEAN_PROPERTIES_PROP, "name,age");
        processor = new ObjectToCSVString(config);
        oMsg.getBody().add(new TestBean());
        processor.process(oMsg);
        assertEquals("Tom Fennelly,21", oMsg.getBody().get());

        // Multi value - some properties exist on bean and some don't...
        config = new ConfigTree("config");
        config.setAttribute(ObjectToCSVString.BEAN_PROPERTIES_PROP, "id,name,phone,age");
        processor = new ObjectToCSVString(config);
        oMsg.getBody().add(new TestBean());
        log.warn( "The following two error messages are intentional" );
        processor.process(oMsg);
        assertEquals("<no-such-property>,Tom Fennelly,<no-such-property>,21", oMsg.getBody().get());

        // Single value - property doesn't exist on bean...
        config = new ConfigTree("config");
        config.setAttribute(ObjectToCSVString.BEAN_PROPERTIES_PROP, "id");
        processor = new ObjectToCSVString(config);
        oMsg.getBody().add(new TestBean());
        log.warn( "The following error message is intentional" );
        processor.process(oMsg);
        assertEquals("<no-such-property>", oMsg.getBody().get());

        // Single value - property doesn't exist on bean and "fail-on-missing-property" action config is set on...
        config = new ConfigTree("config");
        config.setAttribute(ObjectToCSVString.BEAN_PROPERTIES_PROP, "id");
        config.setAttribute(ObjectToCSVString.FAIL_ON_MISSING_PROPERTY, "true");
        processor = new ObjectToCSVString(config);
        oMsg.getBody().add(new TestBean());
        
        try {
	        log.warn( "The following error message is intentional" );
        	processor.process(oMsg);
        	fail("Expected ActionProcessingException");
        } catch(ActionProcessingException e) {
        	assertEquals("Bean method: getId not found/accessible on message object org.jboss.soa.esb.actions.converters.ObjectToCSVStringUnitTest$TestBean", e.getMessage());
        }
    }
    
    public static class TestBean  implements java.io.Serializable
    {
		private static final long serialVersionUID = 1L;
		private String name = "Tom Fennelly";
        private int age = 21; // hehehehe
        
        public int getAge() {
            return age;
        }
        public String getName() {
            return name;
        }
    }
}
