/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.configure;

import java.util.List;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionLifecycleException;
import org.jboss.soa.esb.helpers.ConfigTree;

import junit.framework.TestCase;

/**
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class ConfiguratorUnitTest extends TestCase {

	public void test_config_no_lifecycle_no_errors() throws ConfigurationException {
		ConfigTree configTree = createConfig();
		
		ComponentWithNoLifecycle component = new ComponentWithNoLifecycle();
		
		Configurator.configure(component, configTree);
		
		assertComponentConfigured(component);
	}

	public void test_config_no_lifecycle_with_errors_01() throws ConfigurationException {
		ConfigTree configTree = createConfig();

		// intVal1 not configured
		configTree.setAttribute("intVal1", null);
		
		assertFailure(configTree, "Property 'intVal1' not specified on configuration");
	}

	public void test_config_no_lifecycle_with_errors_02() throws ConfigurationException {
		ConfigTree configTree = createConfig();

		// enumVal1 set to an invalid choice...
		configTree.setAttribute("enumVal1", "3");
		
		assertFailure(configTree, "Value '3' for property 'enumVal1' is invalid.  Valid choices for this property are: [1, 2]");
	}

	public void test_config_no_lifecycle_with_errors_03() throws ConfigurationException {
		ConfigTree configTree = createConfig();

		// enumVal2 set to an invalid enum constant value...
		configTree.setAttribute("enumVal2", "X");
		
		assertFailure(configTree, "Exception decoding configuration value for property 'enumVal2' on class 'org.jboss.soa.esb.configure.ComponentWithNoLifecycle': Failed to decode 'X' as a valid Enum constant of type 'org.jboss.soa.esb.configure.AlphaChar'.");
	}

	public void test_config_with_lifecycle_but_no_lifecycle_methods() throws ConfigurationException, ActionLifecycleException {
		ConfigTree configTree = createConfig();
		
		ComponentWithNoLifecycle component = new ComponentWithNoLifecycle();
		
		Configurator.configure(component, configTree);
		
		Configurator.initialise(component, configTree);
		Configurator.destroy(component);
	}

	public void test_config_with_lifecycle_and_lifecycle_methods() throws ConfigurationException, ActionLifecycleException {
		ConfigTree configTree = createConfig();
		
		ComponentWithLifecycle component = new ComponentWithLifecycle();
		
		Configurator.configure(component, configTree);
		
		Configurator.initialise(component, configTree);
		Configurator.destroy(component);
		
		assertTrue(component.compInit1Called);
		assertNotNull(component.configTree);
		assertTrue(component.compDestroyCalled);
	}

	public void test_config_with_lifecycle_and_super_lifecycle_methods() throws ConfigurationException, ActionLifecycleException {
		ConfigTree configTree = createConfig();
		
		ComponentWithLifecycleAndSuperLifecycle component = new ComponentWithLifecycleAndSuperLifecycle();
		
		Configurator.configure(component, configTree);
		
		Configurator.initialise(component, configTree);
		Configurator.destroy(component);
		
		assertTrue(component.compInit1Called);
		assertNotNull(component.configTree);
		assertTrue(component.compDestroyCalled);
		assertTrue(component.compInit3Called);
		assertNotNull(component.configTree4);
	}

	public void test_config_with_lifecycle_and_lifecycle_exception() throws ConfigurationException, ActionLifecycleException {
		ConfigTree configTree = createConfig();
		
		ComponentWithLifecycle component = new ComponentWithLifecycle();
		
		// Set the exception...
		component.lifecycleException = new ActionLifecycleException("Xception");
		
		Configurator.configure(component, configTree);
		
		try {
			Configurator.initialise(component, configTree);
			fail("Expected ActionLifecycleException.");
		} catch(ActionLifecycleException e) {
			assertEquals(component.lifecycleException, e);
		}
	}

	private ConfigTree createConfig() {
		ConfigTree configTree = new ConfigTree("config");
		
		configTree.setAttribute("intVal1", "1");
		configTree.setAttribute("enumVal1", "1");
		configTree.setAttribute("enumVal2", "B");
		configTree.setAttribute("classVal1", List.class.getName());
		
		return configTree;
	}

	private void assertComponentConfigured(ComponentWithNoLifecycle component) {
		assertEquals(1, component.getIntVal1());
		assertEquals(2, component.getIntVal2());
		assertEquals(null, component.getIntVal3());
		assertEquals("1", component.getEnumVal1());
		assertEquals(AlphaChar.B, component.getEnumVal2());
	}

	private void assertFailure(ConfigTree configTree, String message) {
		try {
			Configurator.configure(new ComponentWithNoLifecycle(), configTree);
			fail("Expected ConfigurationException");
		} catch (ConfigurationException e) {
//			e.printStackTrace();
			if(!e.getMessage().startsWith(message)) {
				fail("Expected exception starting with: [" +  message+ "].  Actual: [" + e.getMessage() + "]");
			}
		}
	}
}
