/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.message.tests;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.net.URI;
import java.util.Set;

import junit.framework.TestCase;

import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.message.ByReferenceMessage;
import org.jboss.soa.esb.message.Context;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;
import org.jboss.soa.esb.util.Util;

/**
 * Unit tests for the Context class.
 * 
 * @author <a href='mailto:kevin.conner@jboss.com'>Kevin Conner</a>
 */

public class ContextUnitTest extends TestCase
{
    public void testSerializeMessage()
        throws Exception
    {
        execute(MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED)) ;
    }
    
    public void testXMLMessage()
        throws Exception
    {
        execute(MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML)) ;
    }
    
    private void execute(final Message message)
        throws Exception
    {
        assertNotNull("Created message", message) ;
        message.getHeader().getCall().setTo(new EPR(new URI("test:to_uri"))) ;
        
        final Context context = message.getContext() ;
        assertNotNull("Context", context) ;
        
        final Object result1 = context.setContext("foo", "bar") ;
        assertNull("First setter result", result1) ;
        
        assertEquals("Getting context", "bar", context.getContext("foo")) ;
        
        final Set<String> keys = context.getContextKeys() ;
        assertNotNull("Context keys", keys) ;
        assertEquals("Context key count", 1, keys.size()) ;
        assertTrue("Contains key", keys.contains("foo")) ;
        
        final Object result2 = context.setContext("foo", "bar2") ;
        assertEquals("Second setter result", "bar", result2) ;
        
        assertEquals("Getting context", "bar2", context.getContext("foo")) ;
        
        final Set<String> keys2 = context.getContextKeys() ;
        assertNotNull("Context keys2", keys2) ;
        assertEquals("Context key count", 1, keys2.size()) ;
        assertTrue("Contains key", keys2.contains("foo")) ;
        
        context.clearContext() ;
        
        final Set<String> keys3 = context.getContextKeys() ;
        assertNotNull("Context keys3", keys3) ;
        assertEquals("Context key count", 0, keys3.size()) ;
        
        try
        {
            context.setContext("non-serializable", new Object()) ;
            fail("Non serializable object set on context");
        }
        catch (IllegalArgumentException ex) {} // expected
        
        context.setContext("skey", "svalue") ;
        final Serializable serializable = Util.serialize(message) ;
        final ByteArrayOutputStream baos = new ByteArrayOutputStream() ;
        final ObjectOutputStream oos = new ObjectOutputStream(baos) ;
        
        oos.writeObject(serializable);
        oos.close();
        
        final ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray()) ;
        final ObjectInputStream ois = new ObjectInputStream(bais) ;
        
        final Serializable newSerializable = (Serializable)ois.readObject() ;
        final Message newMessage = Util.deserialize(newSerializable) ;
        assertEquals("Serialized context", "svalue", newMessage.getContext().getContext("skey")) ;
        
        assertTrue("Implements ByReferenceMessage", (message instanceof ByReferenceMessage)) ;
        
        final Message referencedMessage = ((ByReferenceMessage)message).reference() ;
        assertSame("Attachment is same", message.getAttachment(), referencedMessage.getAttachment()) ;
        assertSame("Body is same", message.getBody(), referencedMessage.getBody()) ;
        assertSame("Fault is same", message.getFault(), referencedMessage.getFault()) ;
        assertSame("Properties are same", message.getProperties(), referencedMessage.getProperties()) ;
        assertSame("type is same", message.getType(), referencedMessage.getType()) ;
        
        assertNotSame("header is not same", message.getHeader(), referencedMessage.getHeader()) ;
        assertNotSame("Context is not same", context, referencedMessage.getContext()) ;
        
        assertEquals("Header equals", message.getHeader().getCall().getTo(), referencedMessage.getHeader().getCall().getTo()) ;
        message.getHeader().getCall().setTo(new EPR(new URI("test:to_uri2"))) ;
        assertFalse("Header not equals", message.getHeader().getCall().getTo().equals(referencedMessage.getHeader().getCall().getTo())) ;
        
        assertEquals("Context equals", context.getContext("skey"), referencedMessage.getContext().getContext("skey")) ;
        context.setContext("skey", "other value") ;
        assertFalse("Context not equals", context.getContext("skey").equals(referencedMessage.getContext().getContext("skey"))) ;
        assertEquals("Referenced Context unchanged", "svalue", referencedMessage.getContext().getContext("skey")) ;
    }
}
