/*
 * JBoss, Home of Professional Open Source
 * Copyright 2011, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
package org.jboss.soa.esb.services.security;

import java.io.IOException;
import java.net.URI;

import junit.framework.JUnit4TestAdapter;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.common.Environment;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;


/**
 * Unit tests for password plugin functionality.
 * 
 * @author <a href="mailto:kevin.conner@jboss.com">Kevin Conner</a>
 */
public final class PasswordPluginManagerUnitTest
{
    @Test
    public void checkOrder()
    	throws Exception
    {
    	final PasswordPluginManager manager = new PasswordPluginManager() ;
    	
    	BasePasswordPlugin.checked = new boolean[3] ;
    	BasePasswordPlugin.currentId = 0 ;
    	
    	Assert.assertTrue("First plugin isPasswordSupported", manager.isPasswordSupported("dummy")) ;
    	Assert.assertTrue("First plugin checked", BasePasswordPlugin.checked[0]) ;
    	Assert.assertFalse("Second plugin called", BasePasswordPlugin.checked[1]) ;
    	Assert.assertFalse("Third plugin called", BasePasswordPlugin.checked[2]) ;
    	
    	BasePasswordPlugin.checked = new boolean[3] ;
    	BasePasswordPlugin.currentId = 0 ;
    	
    	PasswordHandler passwordHandler = manager.passwordHandler("dummy") ;
    	
    	Assert.assertNotNull("First plugin passwordHandler", passwordHandler) ;
    	Assert.assertEquals("First plugin password", "password0", passwordHandler.getPassword()) ;
    	Assert.assertTrue("First plugin checked", BasePasswordPlugin.checked[0]) ;
    	Assert.assertFalse("Second plugin called", BasePasswordPlugin.checked[1]) ;
    	Assert.assertFalse("Third plugin called", BasePasswordPlugin.checked[2]) ;
    	
    	BasePasswordPlugin.checked = new boolean[3] ;
    	BasePasswordPlugin.currentId = 1 ;
    	
    	Assert.assertTrue("Second plugin isPasswordSupported", manager.isPasswordSupported("dummy")) ;
    	Assert.assertTrue("First plugin checked", BasePasswordPlugin.checked[0]) ;
    	Assert.assertTrue("Second plugin called", BasePasswordPlugin.checked[1]) ;
    	Assert.assertFalse("Third plugin called", BasePasswordPlugin.checked[2]) ;
    	
    	BasePasswordPlugin.checked = new boolean[3] ;
    	BasePasswordPlugin.currentId = 1 ;
    	
    	passwordHandler = manager.passwordHandler("dummy") ;
    	
    	Assert.assertNotNull("Second plugin passwordHandler", passwordHandler) ;
    	Assert.assertEquals("Second plugin password", "password1", passwordHandler.getPassword()) ;
    	Assert.assertTrue("First plugin checked", BasePasswordPlugin.checked[0]) ;
    	Assert.assertTrue("Second plugin called", BasePasswordPlugin.checked[1]) ;
    	Assert.assertFalse("Third plugin called", BasePasswordPlugin.checked[2]) ;
    	
    	BasePasswordPlugin.checked = new boolean[3] ;
    	BasePasswordPlugin.currentId = 2 ;
    	
    	Assert.assertTrue("Third plugin isPasswordSupported", manager.isPasswordSupported("dummy")) ;
    	Assert.assertTrue("First plugin checked", BasePasswordPlugin.checked[0]) ;
    	Assert.assertTrue("Second plugin called", BasePasswordPlugin.checked[1]) ;
    	Assert.assertTrue("Third plugin called", BasePasswordPlugin.checked[2]) ;
    	
    	BasePasswordPlugin.checked = new boolean[3] ;
    	BasePasswordPlugin.currentId = 2 ;
    	
    	passwordHandler = manager.passwordHandler("dummy") ;
    	
    	Assert.assertNotNull("Third plugin passwordHandler", passwordHandler) ;
    	Assert.assertEquals("Third plugin password", "password2", passwordHandler.getPassword()) ;
    	Assert.assertTrue("First plugin checked", BasePasswordPlugin.checked[0]) ;
    	Assert.assertTrue("Second plugin called", BasePasswordPlugin.checked[1]) ;
    	Assert.assertTrue("Third plugin called", BasePasswordPlugin.checked[2]) ;
    	
    	BasePasswordPlugin.checked = new boolean[3] ;
    	BasePasswordPlugin.currentId = -1 ;
    	
    	Assert.assertFalse("No plugin isPasswordSupported", manager.isPasswordSupported("dummy")) ;
    	Assert.assertTrue("First plugin checked", BasePasswordPlugin.checked[0]) ;
    	Assert.assertTrue("Second plugin called", BasePasswordPlugin.checked[1]) ;
    	Assert.assertTrue("Third plugin called", BasePasswordPlugin.checked[2]) ;
    	
    	BasePasswordPlugin.checked = new boolean[3] ;
    	BasePasswordPlugin.currentId = -1 ;
    	
    	passwordHandler = manager.passwordHandler("dummy") ;
    	
    	Assert.assertNull("No plugin passwordHandler", passwordHandler) ;
    	Assert.assertTrue("First plugin checked", BasePasswordPlugin.checked[0]) ;
    	Assert.assertTrue("Second plugin called", BasePasswordPlugin.checked[1]) ;
    	Assert.assertTrue("Third plugin called", BasePasswordPlugin.checked[2]) ;
    }
    
	@Before
    public void setup() throws ConfigurationException
    {
		System.setProperty(Environment.SECURITY_SERVICE_PASSWORD_PLUGINS + "1", FirstPasswordPlugin.class.getName()) ;
		System.setProperty(Environment.SECURITY_SERVICE_PASSWORD_PLUGINS + "2", SecondPasswordPlugin.class.getName()) ;
		System.setProperty(Environment.SECURITY_SERVICE_PASSWORD_PLUGINS + "3", ThirdPasswordPlugin.class.getName()) ;
    }

    @After
    public void tearDown()
    {
		System.clearProperty(Environment.SECURITY_SERVICE_PASSWORD_PLUGINS + "1") ;
		System.clearProperty(Environment.SECURITY_SERVICE_PASSWORD_PLUGINS + "2") ;
		System.clearProperty(Environment.SECURITY_SERVICE_PASSWORD_PLUGINS + "3") ;
    }
    
    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(PasswordPluginManagerUnitTest.class);
    }

    static class BasePasswordPlugin implements PasswordPlugin
    {
    	static int currentId ;
    	static boolean[] checked ;
    	
    	private final int id ;
    	
    	protected BasePasswordPlugin(final int id)
    	{
    		this.id = id ;
    	}

		@Override
		public boolean supportsPasswordFile(final URI uri)
		{
			checked[id] = true ;
			return (currentId == id) ;
		}

		@Override
		public PasswordHandler pluginHandler(final URI uri)
		{
			return new PasswordHandler() {
				public String getPassword() throws IOException {
					return "password" + id ;
				}
			} ;
		}
    }

    public static class FirstPasswordPlugin extends BasePasswordPlugin
    {
    	FirstPasswordPlugin()
    	{
    		super(0) ;
    	}
    }
    
    public static class SecondPasswordPlugin extends BasePasswordPlugin
    {
    	SecondPasswordPlugin()
    	{
    		super(1) ;
    	}
    }

    public static class ThirdPasswordPlugin extends BasePasswordPlugin
    {
    	ThirdPasswordPlugin()
    	{
    		super(2) ;
    	}
    }
}
