/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.security.auth.ws;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.security.cert.X509Certificate;

import javax.xml.soap.SOAPMessage;

import junit.framework.JUnit4TestAdapter;

import org.jboss.soa.esb.services.security.auth.AuthenticationRequest;
import org.junit.Test;


/**
 * Unit test for {@link WSSecuritySoapExtractor}
 * <p/>
 * 
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 *
 */
public class WSSecuritySoapExtractorUnitTest
{
	@Test
	public void extractSecurityInfoBinarySecurityToken() throws Exception 
	{
		WSSecuritySoapExtractor extractor = new WSSecuritySoapExtractor();
		SOAPMessage soap = WSTestUtil.createWithBinarySecurityToken("wsse:Base64Binary", "wsse:X509v3", WSTestUtil.getStringFromFile("cert-example.xml", getClass()));
		AuthenticationRequest authRequest = extractor.extractSecurityInfo(soap);
		
		assertNotNull(authRequest);
		assertTrue(authRequest.getCredentials().size() == 1 );
		assertTrue(authRequest.getCredentials().iterator().next() instanceof X509Certificate );
	}
	
	@Test
	public void extractSecurityInfoBinarySecurityTokenNoNSPrifix() throws Exception
	{
		WSSecuritySoapExtractor extractor = new WSSecuritySoapExtractor();
		//	create the SAOPMessage with out namespace prefixes for ValueType and EncodingType
		SOAPMessage soap = WSTestUtil.createWithBinarySecurityToken("Base64Binary", "X509v3", WSTestUtil.getStringFromFile("cert-example.xml", getClass()));
		AuthenticationRequest authRequest = extractor.extractSecurityInfo(soap);
		
		assertNotNull(authRequest);
		assertTrue(authRequest.getCredentials().size() == 1 );
		assertTrue(authRequest.getCredentials().iterator().next() instanceof X509Certificate );
	}
	
	@Test
	public void extractSecurityInfoBinarySecurityTokenFromFile() throws Exception
	{
		WSSecuritySoapExtractor extractor = new WSSecuritySoapExtractor();
		SOAPMessage soap = WSTestUtil.createMessage("soap-keys-example.xml", getClass());
		AuthenticationRequest authRequest = extractor.extractSecurityInfo(soap);
		
		assertNotNull(authRequest);
		assertTrue(authRequest.getCredentials().size() == 1 );
		assertTrue(authRequest.getCredentials().iterator().next() instanceof X509Certificate );
	}
	
	@Test
	public void extractSecurityInfoUsernameToken() throws Exception
	{
		final String username = "Bubbles";
		final String password = "228833dkd0";
		WSSecuritySoapExtractor extractor = new WSSecuritySoapExtractor();
		SOAPMessage soap = WSTestUtil.createWithUsernameToken(username, password);
		AuthenticationRequest authRequest = extractor.extractSecurityInfo(soap);
		
		assertNotNull(authRequest);
		assertEquals(username, authRequest.getPrincipal().getName());
		assertTrue(authRequest.getCredentials().size() == 1 );
		assertTrue(authRequest.getCredentials().iterator().next() instanceof char[] );
	}
	
	@Test
	public void extractSecurityInfoUsernameTokenNoUsername() throws Exception
	{
		final String password = "228833dkd0";
		WSSecuritySoapExtractor extractor = new WSSecuritySoapExtractor();
		SOAPMessage soap = WSTestUtil.createWithUsernameToken(null, password);
		AuthenticationRequest authRequest = extractor.extractSecurityInfo(soap);
		
		assertNotNull(authRequest);
		assertEquals(null, authRequest.getPrincipal() );
		assertTrue(authRequest.getCredentials().size() == 1 );
		assertTrue(authRequest.getCredentials().iterator().next() instanceof char[] );
	}
	
	@Test
	public void extractSecurityInfoUsernameTokenNoPassword() throws Exception
	{
		final String username = "Bubbles";
		WSSecuritySoapExtractor extractor = new WSSecuritySoapExtractor();
		SOAPMessage soap = WSTestUtil.createWithUsernameToken(username, null);
		AuthenticationRequest authRequest = extractor.extractSecurityInfo(soap);
		
		assertNotNull(authRequest);
		assertEquals(username, authRequest.getPrincipal().getName());
		assertTrue(authRequest.getCredentials().size() == 0 );
	}
	
	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter(WSSecuritySoapExtractorUnitTest.class);
	}
	
}
