/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.services.rules;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.drools.runtime.Channel;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.services.rules.RuleInfo;

/**
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 * @since 4.6
 *
 */
public class RuleInfoBuilder
{
    String ruleSource;
    String ruleType;
    boolean reload;
    String dslSource;
    String auditType;
    String auditFile;
    Integer auditInterval;
    String clockType;
    String eventProcessingType;
    Boolean multithreadEvaluation;
    Integer maxThreads;
    Map<String,Channel> channels;
    String ruleFireMethod;
    Boolean defaultContinue;
        
    Map<String, Object> globals;
    Map<String, List<Object>> factsMap;
    List<Object> defaultFacts;
    
    public RuleInfoBuilder()
    {
    	ruleSource(null);
    }
    
    public RuleInfoBuilder(final String ruleSource)
    {
        AssertArgument.isNotNullAndNotEmpty(ruleSource, "ruleSource");
        ruleSource(ruleSource);
    }
    
    public RuleInfoBuilder ruleSource(final String ruleSource)
    {
        this.ruleSource = ruleSource;
        return this;
    }

    public RuleInfoBuilder ruleType(final String type)
    {
        ruleType = type;
        return this;
    }

    public RuleInfoBuilder dslSource(final String dsl)
    {
        if (dslSource != null)
        {
            throw new IllegalArgumentException("dslSource has already been set to '" + dslSource + "'");
        }
        dslSource = dsl;
        return this;
    }
    
    public RuleInfoBuilder auditType(final String auditType)
    {
    	this.auditType = auditType;
        return this;
    }
    
    public RuleInfoBuilder auditFile(final String auditFile)
    {
    	this.auditFile = auditFile;
        return this;
    }
    
    public RuleInfoBuilder auditInterval(final Integer auditInterval)
    {
    	this.auditInterval = auditInterval;
        return this;
    }
    
    public RuleInfoBuilder clockType(final String clockType)
    {
    	this.clockType = clockType;
        return this;
    }
    
    public RuleInfoBuilder eventProcessingType(final String eventProcessingType)
    {
    	this.eventProcessingType = eventProcessingType;
        return this;
    }
    
    public RuleInfoBuilder multithreadEvaluation(final Boolean multithreadEvaluation)
    {
    	this.multithreadEvaluation = multithreadEvaluation;
        return this;
    }
    
    public RuleInfoBuilder maxThreads(final Integer maxThreads)
    {
    	this.maxThreads = maxThreads;
        return this;
    }
    
    public RuleInfoBuilder channels(final Map<String,Channel> channels)
    {
    	this.channels = channels;
    	return this;
    }
    
    public RuleInfoBuilder ruleFireMethod(final String ruleFireMethod)
    {
    	this.ruleFireMethod = ruleFireMethod;
    	return this;
    }

    public RuleInfoBuilder reload(final boolean reload)
    {
        this.reload = reload;
        return this;
    }

    public RuleInfoBuilder global(final String name, final Object object)
    {
        if (this.globals == null)
        {
            this.globals = new HashMap<String, Object>();
        }
        this.globals.put(name, object);
        return this;
    }
    
    public RuleInfoBuilder globals(Map<String, Object> globals)
    {
        if (globals != null)
        {
        	for (Entry<String, Object> entry : globals.entrySet())
        	{
        		global(entry.getKey(), entry.getValue());
        	}
        }
        return this;
    }
    
    public RuleInfoBuilder defaultFact(final Object object)
    {
        if (object != null)
        {
            if (this.defaultFacts == null)
            {
                this.defaultFacts = new ArrayList<Object>();
            }
            this.defaultFacts.add(object);
        }
        return this;
    }
    
    public RuleInfoBuilder defaultFacts(List<Object> facts)
    {
        if (facts != null)
        {
            for (Object object : facts)
            {
                defaultFact(object);
            }
        }
        return this;
    }

    public RuleInfoBuilder fact(final String name, final Object object)
    {
        if (this.factsMap == null)
        {
            this.factsMap = new HashMap<String, List<Object>>();
        }
        List<Object> list = this.factsMap.get(name);
        if (list == null)
        {
            list = new ArrayList<Object>();
            this.factsMap.put(name, list);
        }
        list.add(object);
        return this;
    }

    public RuleInfoBuilder facts(Map<String, List<Object>> facts)
    {
        if (facts != null)
        {
        	for (Entry<String, List<Object>> entry : facts.entrySet())
        	{
        		for (Object fact : entry.getValue())
        		{
        			fact(entry.getKey(), fact);
        		}
        	}
        }
        return this;
    }
    
    public RuleInfoBuilder defaultContinue(final Boolean defaultContinue)
    {
        this.defaultContinue = defaultContinue;
        return this;
    }

    public RuleInfo build()
    {
        if (factsMap == null)
            factsMap = Collections.emptyMap();
        if (defaultFacts == null)
            defaultFacts = Collections.emptyList();
        
        return new RuleInfoImpl(this);
        
    }

}