/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.services.rules;

import static org.jboss.soa.esb.services.rules.RuleServicePropertiesNames.StringValue.FIRE_UNTIL_HALT;
import static org.jboss.soa.esb.services.rules.RuleServicePropertiesNames.StringValue.REALTIME;
import static org.jboss.soa.esb.services.rules.RuleServicePropertiesNames.StringValue.STREAM;
import static org.jboss.soa.esb.services.rules.RuleServicePropertiesNames.StringValue.THREADED_FILE;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import junit.framework.JUnit4TestAdapter;

import org.jboss.soa.esb.services.rules.RuleInfo;
import org.junit.Test;

/**
 * Unit test for {@link RuleInfoImpl}.
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 *
 */
public class RuleInfoImplUnitTest
{
    @Test
    public void ruleSet()
    {
        final RuleInfoBuilder builder = new RuleInfoBuilder("dummy.drl");
        final RuleInfo info = builder.ruleType("DRL").build();;
        assertEquals("dummy.drl", info.getRuleSource());
        assertEquals("DRL", info.getRuleType());
    }
    
    @Test (expected = IllegalArgumentException.class)
    public void ruleSetNull()
    {
        new RuleInfoBuilder(null);
    }
    
    @Test
    public void fact()
    {
        final RuleInfoBuilder builder = new RuleInfoBuilder("dummy.drl");
        final RuleInfo info = builder.fact("one", "first").build();;
        Map<String, List<Object>> facts = info.getFacts();
        assertTrue(facts.get("one").contains("first"));
    }
    
    @Test
    public void factEmpty()
    {
        RuleInfo info = new RuleInfoBuilder("dummy.drl").build();
        Map<String, List<Object>> facts = info.getFacts();
        assertTrue(facts.isEmpty());
    }
    
    @Test
    public void facts()
    {
        final RuleInfoBuilder builder = new RuleInfoBuilder("dummy.drl");
        Map<String, List<Object>> facts = new HashMap<String, List<Object>>();
        facts.put("one", Arrays.asList(new Object[] {"first"}));
        builder.facts(facts);
        
        final RuleInfo info = builder.build();
        facts = info.getFacts();
        assertTrue(facts.get("one").contains("first"));
    }
    
    @Test
    public void globals()
    {
        final RuleInfoBuilder builder = new RuleInfoBuilder("dummy.drl");
        Map<String, Object> globals = new HashMap<String, Object>();
        globals.put("message", "msgObject");
        builder.globals(globals);
        
        final RuleInfo info = builder.build();
        globals = info.getGlobals();
        assertTrue(globals.containsKey("message"));
        assertTrue(globals.containsValue("msgObject"));
    }
    
    @Test
    public void audit()
    {
    	final RuleInfo ruleInfo = new RuleInfoBuilder("dummy.drl")
    		.auditType(THREADED_FILE.name())
    		.auditFile("tracker")
    		.auditInterval(5000)
    		.build();
    	assertEquals(THREADED_FILE.name(), ruleInfo.getAuditType());
    	assertEquals("tracker", ruleInfo.getAuditFile());
    	assertEquals(5000, ruleInfo.getAuditInterval());
    }
    
    @Test
    public void cep()
    {
    	final RuleInfo ruleInfo = new RuleInfoBuilder("dummy.drl")
    		.clockType(REALTIME.name())
    		.eventProcessingType(STREAM.name())
    		.multithreadEvaluation(Boolean.TRUE)
    		.maxThreads(Integer.valueOf(14))
    		.ruleFireMethod(FIRE_UNTIL_HALT.name())
    		.build();
    	assertEquals(REALTIME.name(), ruleInfo.getClockType());
    	assertEquals(STREAM.name(), ruleInfo.getEventProcessingType());
    	assertEquals(Boolean.TRUE, ruleInfo.getMultithreadEvaluation());
    	assertEquals(Integer.valueOf(14), ruleInfo.getMaxThreads());
    	assertEquals(FIRE_UNTIL_HALT.name(), ruleInfo.getRuleFireMethod());
    }
    
    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(RuleInfoImplUnitTest.class);
    }

}
