/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.smooks.splitting;

import org.jboss.soa.esb.actions.aggregator.AggregateDetails;
import org.jboss.soa.esb.listeners.message.MessageComposer;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.internal.soa.esb.assertion.AssertArgument;

import org.milyn.container.ExecutionContext;

import java.io.File;
import java.io.IOException;
import java.io.FileInputStream;
import java.util.UUID;

/**
 * File stream splitting message composer.
 * <p/>
 * This composer only sets the file name as the message payload.  It splits the message out,
 * so the source message is transformed into n+ messages that are consumed by other service
 * pipelines (i.e. not this service)...
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class FileStreamSplitter<T extends File> extends AbstractStreamSplitter implements MessageComposer<T> {

    public Message compose(T inputFile) throws MessageDeliverException {
        AssertArgument.isNotNull(inputFile, "inputFile");

        if(!inputFile.exists()) {
            throw new MessageDeliverException("Invalid File payload.  File '" + inputFile.getAbsolutePath() + "' doesn't exist.");
        }

        ExecutionContext execContext = smooks.createExecutionContext();            
        final String seriesUUID = UUID.randomUUID().toString();
        execContext.setAttribute(AggregateDetails.SERIES_UUID, seriesUUID);
        
        // Split the file input stream...
        try {
            split(new FileInputStream(inputFile), execContext);
        } catch (IOException e) {
            throw new MessageDeliverException("Exception while splitting file input stream for file '" + inputFile.getAbsolutePath() + "'.", e);
        }

        Message message = MessageFactory.getInstance().getMessage();
        message.getBody().add(inputFile.getAbsolutePath());
            
        // If the executionContext contains a sequence count, add it to the message
        if (execContext.getAttribute(AggregateDetails.SEQUENCE_COUNT) != null) {
        	Integer sequenceCount = (Integer) execContext.getAttribute(AggregateDetails.SEQUENCE_COUNT);            
   
        	if (sequenceCount.intValue() > 0) {
        		AggregateDetails aggDetails = new AggregateDetails(seriesUUID, new Integer(0));
        		aggDetails.setSequenceCount(sequenceCount);
        		message.getProperties().setProperty(AggregateDetails.AGGREGATE_DETAILS, 
        				aggDetails);
        	}
        }        
        return message;
    }

    public Object decompose(Message message, T inputFile) throws MessageDeliverException {
        return null;
    }
}