package org.jboss.soa.esb.actions.soap;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.Map;

import javax.xml.parsers.ParserConfigurationException;

import junit.framework.TestCase;

import org.jboss.internal.soa.esb.util.StreamUtils;
import org.jboss.internal.soa.esb.util.XMLHelper;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.actions.soap.SOAPClient.Response;
import org.jboss.soa.esb.actions.soap.beans.Order;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.testutils.ESBConfigUtil;
import org.xml.sax.SAXException;

/**
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class SOAPClient_Response_UnitTest extends TestCase {

    private ESBConfigUtil configUtil;

    public SOAPClient_Response_UnitTest() throws IOException, SAXException, ConfigurationException {
        configUtil = new ESBConfigUtil(getClass().getResourceAsStream("soapclient-config-01.xml"));
    }

    public void test_XStream_ResponseProcessing() throws ConfigurationException, ActionProcessingException {
        ConfigTree actionConfig = configUtil.getActionConfig("OrderNotificationService", "soapui-client-action-01");
        SOAPClient soapClient = new SOAPClient(actionConfig);
        Message message = MessageFactory.getInstance().getMessage();

        soapClient.processResponse(message, response_01);
        assertEquals(true, message.getBody().get());

        soapClient.processResponse(message, response_02);
        assertEquals(false, message.getBody().get());
    }

    @SuppressWarnings("unchecked")
    public void test_OgnlMap_ResponseProcessing_1() throws ConfigurationException, ActionProcessingException {
        ConfigTree actionConfig = configUtil.getActionConfig("OrderNotificationService", "soapui-client-action-02");
        SOAPClient soapClient = new SOAPClient(actionConfig);
        Message message = MessageFactory.getInstance().getMessage();

        soapClient.processResponse(message, response_01);
        Map<String, String> response = (Map<String, String>) message.getBody().get();
        assertEquals("true", response.get("customerOrderAck"));

        soapClient.processResponse(message, response_02);
        response = (Map<String, String>) message.getBody().get();
        assertEquals("false", response.get("customerOrderAck"));
    }

    public void test_OgnlMap_ResponseProcessing_2() throws ConfigurationException, ActionProcessingException {
        ConfigTree actionConfig = configUtil.getActionConfig("OrderNotificationService", "soapui-client-action-05");
        SOAPClient soapClient = new SOAPClient(actionConfig);
        String responseString = new String(StreamUtils.readStream(getClass().getResourceAsStream("soap-message-01.xml")));
        Message message = MessageFactory.getInstance().getMessage();

        soapClient.processResponse(message, new Response(responseString));
        @SuppressWarnings("unused")
		Order order = (Order) message.getBody().get();
        //assertEquals();
    }

    @SuppressWarnings("unchecked")
    public void test_OgnlMap_ResponseProcessing_3() throws ConfigurationException, ActionProcessingException {
        ConfigTree actionConfig = configUtil.getActionConfig("OrderNotificationService", "soapui-client-action-02");
        SOAPClient soapClient = new SOAPClient(actionConfig);
        String responseString = new String(StreamUtils.readStream(getClass().getResourceAsStream("soap-message-01.xml")));
        Message message = MessageFactory.getInstance().getMessage();

        soapClient.processResponse(message, new Response(responseString));
        Map<String, String> responseMap = (Map<String, String>) message.getBody().get();

        assertEquals("{customerOrder.orderheader.customerName=Tom Fennelly, customerOrder.items[0].partNumber=1, customerOrder.items[0].description=desc-1, customerOrder.items[0].quantity=1, customerOrder.items[0].price=1.1, customerOrder.items[0].extensionAmount=1, customerOrder.items[1].partNumber=2, customerOrder.items[1].description=desc-2, customerOrder.items[1].quantity=2, customerOrder.items[1].price=2.2, customerOrder.items[1].extensionAmount=2}", responseMap.toString());
    }

    public void test_OgnlMap_ResponseProcessing_4() throws ConfigurationException, ActionProcessingException {
        ConfigTree actionConfig = configUtil.getActionConfig("OrderNotificationService", "soapui-client-action-02");
        SOAPClient soapClient = new SOAPClient(actionConfig);
        String responseString = new String(StreamUtils.readStream(getClass().getResourceAsStream("soap-message-02.xml")));
        String expected = "{processOrderResponse.OrderStatus.id=1," +
                          " processOrderResponse.OrderStatus.comment=order processed," +
                          " processOrderResponse.OrderStatus.returnCode=1," +
                          " processOrderResponse.OrderStatus.lineItems[0].id=1," +
                          " processOrderResponse.OrderStatus.lineItems[0].name=Item1," +
                          " processOrderResponse.OrderStatus.lineItems[0].price=10," +
                          " processOrderResponse.OrderStatus.lineItems[0].comments[0]=Item1 Comments1," +
                          " processOrderResponse.OrderStatus.lineItems[0].comments[1]=Item1 Comments2," +
                          " processOrderResponse.OrderStatus.lineItems[1].id=2," +
                          " processOrderResponse.OrderStatus.lineItems[1].name=Item2," +
                          " processOrderResponse.OrderStatus.lineItems[1].price=15," +
                          " processOrderResponse.OrderStatus.lineItems[1].comments[0]=Item2 Comments1," +
                          " processOrderResponse.OrderStatus.lineItems[1].comments[1]=Item2 Comments2," +
                          " processOrderResponse.OrderStatus.lineItems[1].comments[2]=Item2 Comments3," +
                          " processOrderResponse.OrderStatus.lineItems[1].comments[3]=Item2 Comments4}";

        Message message = MessageFactory.getInstance().getMessage();

        soapClient.processResponse(message, new Response(responseString));
        Map<String, String> responseMap = (Map<String, String>) message.getBody().get();

        assertEquals(expected, responseMap.toString());
    }

    public void test_OgnlMap_ResponseProcessing_JBESB_1946() throws ConfigurationException, ActionProcessingException {
        ConfigTree actionConfig = configUtil.getActionConfig("OrderNotificationService", "soapui-client-action-02");
        SOAPClient soapClient = new SOAPClient(actionConfig);
        String responseString = new String(StreamUtils.readStream(getClass().getResourceAsStream("soap-message-03.xml")));
        String expected = "{orderResponse.orderStatus[0].id=1," +
                          " orderResponse.orderStatus[0].comment=order 1 processed," +
                          " orderResponse.orderStatus[1].id=2," +
                          " orderResponse.orderStatus[1].comment=order 2 processed}";

        Message message = MessageFactory.getInstance().getMessage();

        soapClient.processResponse(message, new Response(responseString));
        Map<String, String> responseMap = (Map<String, String>) message.getBody().get();

        assertEquals(expected, responseMap.toString());
    }

    public void test_No_ResponseProcessing() throws ConfigurationException, ActionProcessingException {
        ConfigTree actionConfig = configUtil.getActionConfig("OrderNotificationService", "soapui-client-action-03");
        SOAPClient soapClient = new SOAPClient(actionConfig);
        Message message = MessageFactory.getInstance().getMessage();

        soapClient.processResponse(message, response_01);
        assertEquals(response_01.getBody(), message.getBody().get());
    }

    public void test_ResponseLocation() throws ConfigurationException, ActionProcessingException {
        ConfigTree actionConfig = configUtil.getActionConfig("OrderNotificationService", "soapui-client-action-04");
        SOAPClient soapClient = new SOAPClient(actionConfig);
        Message message = MessageFactory.getInstance().getMessage();

        soapClient.processResponse(message, response_01);
        assertEquals(response_01.getBody(), message.getBody().get("myTestLocation"));
    }

    public void test_getEndpointOperation() throws ConfigurationException, ActionProcessingException {
        ConfigTree actionConfig = configUtil.getActionConfig("OrderNotificationService", "soapui-client-action-07");
        SOAPClient soapClient = new SOAPClient(actionConfig);
        assertEquals("SendSalesOrderNotification", soapClient.getEndpointOperation());
    }
    
    public void test_soapSetSOAPNameSpace() throws ConfigurationException, ActionProcessingException {
    	String expectedSOAPNS = "http://www.w3.org/2003/05/soap-envelope";
        ConfigTree actionConfig = configUtil.getActionConfig("OrderNotificationService", "soapui-client-action-07");
        actionConfig.setAttribute( "SOAPNS", expectedSOAPNS );
        SOAPClient soapClient = new SOAPClient(actionConfig);
        assertEquals( expectedSOAPNS, soapClient.getSoapNS());
    }

    public void test_anyType_OGNL() throws ConfigurationException, ActionProcessingException, ParserConfigurationException, SAXException, IOException {
        ConfigTree actionConfig = configUtil.getActionConfig("OrderNotificationService", "soapui-client-action-08");
        SOAPClient soapClient = new SOAPClient(actionConfig);
        String mergedResponse = new String(StreamUtils.readStream(getClass().getResourceAsStream("soap-message-08.xml")));
        String expected = 
                        "<books>\n" +
                        "   <book>\n" +
                        "      <name>JBoss ESB Beginners Guide</name>\n" +
                        "      <authors>\n" +
                        "         <author>Len DiMaggio</author>\n" +
                        "         <author>Kevin Conner</author>\n" +
                        "         <author>Tom Cunningham</author>\n" +
                        "         <author>Magesh Kumar B</author>\n" +
                        "      </authors>\n" +
                        "   </book>\n" +
                        "</books>\n";
        Message message = MessageFactory.getInstance().getMessage();

        soapClient.processResponse(message, new Response(mergedResponse));
        Map<String, String> responseMap = (Map<String, String>) message.getBody().get();
        assertTrue("OGNL parameter not as expected. See " + expected + ".  Generated param: \n" + 
                    message, XMLHelper.compareXMLContent(new ByteArrayInputStream(expected.getBytes()), new ByteArrayInputStream(responseMap.get("sayHiResponse").getBytes())));
    }

    public void test_anyType_SOAP() throws ConfigurationException, ActionProcessingException, ParserConfigurationException, SAXException, IOException {
        ConfigTree actionConfig = configUtil.getActionConfig("OrderNotificationService", "soapui-client-action-09");
        SOAPClient soapClient = new SOAPClient(actionConfig);
        String expected = "soap-message-09.xml";
        String mergedResponse = new String(StreamUtils.readStream(getClass().getResourceAsStream(expected)));
        Message message = MessageFactory.getInstance().getMessage();

        soapClient.processResponse(message, new Response(mergedResponse));
        String response = (String) message.getBody().get();
        assertTrue("OGNL parameter not as expected. See " + expected + ".  Generated param: \n" + 
                    message, XMLHelper.compareXMLContent(getClass().getResourceAsStream(expected), new ByteArrayInputStream(response.getBytes())));
    }

    private static Response response_01 = new Response(
            "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\">\n" +
            "\t<soapenv:Header/>\n" +
            "\t<soapenv:Body>\n" +
            "\t\t<aetgt:customerOrderAck xmlns:aetgt=\"http://schemas.active-endpoints.com/sample/customerorder/2006/04/CustomerOrder.xsd\">true</aetgt:customerOrderAck>\n" +
            "\t</soapenv:Body>\n" +
            "</soapenv:Envelope>");
    private static Response response_02 = new Response(
            "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\">\n" +
            "\t<soapenv:Body>\n" +
            "\t\t<aetgt:customerOrderAck xmlns:aetgt=\"http://schemas.active-endpoints.com/sample/customerorder/2006/04/CustomerOrder.xsd\">false</aetgt:customerOrderAck>\n" +
            "\t</soapenv:Body>\n" +
            "</soapenv:Envelope>");
}
