/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions.soap.wise;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import it.javalinux.wise.core.client.WSDynamicClient;
import it.javalinux.wise.core.client.WSEndpoint;
import it.javalinux.wise.core.exceptions.WiseException;
import it.javalinux.wise.core.utils.WiseProperties;

import java.io.File;
import java.util.HashMap;
import java.util.Map;

import junit.framework.JUnit4TestAdapter;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionLifecycleException;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Unit test for {@link SOAPClient}.
 * <p/>
 */
public class SOAPClientUnitTest
{
    private static File workingDir;
    private String wsdl = "http://127.0.0.1:8080/Quickstart_webservice_consumer_wise/HelloWorldWS?wsdl";
    private String soapAction = "sayHello";
    private String soapActionURI = "http://example.com/opnamespace/sayHello";
    private String endPointName = "HelloWorldPort";
    private String serviceName = "HelloWorldService";

    @BeforeClass
    public static void setup()
    {
        workingDir = new File("working");
        workingDir.mkdir();
    }
    
    @AfterClass
    public static void teardown()
    {
        workingDir.delete();
    }
    
    @Test
    public void configNoOperationName() throws ConfigurationException, ActionProcessingException, ActionLifecycleException
    {
        final ConfigTree config = createConfig(null, soapAction);
        final SOAPClient client = new MockSOAPClient(config);
        
        assertNotNull(client.getOperationName());
        assertEquals(soapAction, client.getOperationName());
    }
    
    @Test
    public void configNoOperationNameURI() throws ConfigurationException, ActionProcessingException, ActionLifecycleException
    {
        final ConfigTree config = createConfig(null, soapActionURI);
        final SOAPClient client = new MockSOAPClient(config);
        
        assertNotNull(client.getOperationName());
        assertEquals(soapAction, client.getOperationName());
    }
    
    @Test
    public void configOperationName() throws ConfigurationException, ActionProcessingException, ActionLifecycleException
    {
        final String operationName = "someOperation";
        final ConfigTree config = createConfig(operationName, soapAction);
        final SOAPClient client = new MockSOAPClient(config);
        
        assertNotNull(client.getOperationName());
        assertEquals(operationName, client.getOperationName());
    }
    
    @Test
    public void configNoSoapAction() throws ConfigurationException, ActionProcessingException, ActionLifecycleException
    {
        final String operationName = "someOperation";
        final ConfigTree config = createConfig(operationName, null);
        final SOAPClient client = new MockSOAPClient(config);
        
        assertNotNull(client.getOperationName());
        assertEquals(operationName, client.getOperationName());
    }

    @Test
    public void configPropSub() throws ConfigurationException, ActionProcessingException, ActionLifecycleException
    {
        final String operationName = "someOperation";
        final ConfigTree config = createPropSubstitutionConfig(operationName, null);
        final SOAPClient client = new MockSOAPClient(config);
        
        assertTrue(client.toString().indexOf("wiseProps={foo=bar}") > 0);
    }

    
    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(SOAPClientUnitTest.class);
    }
    
    private ConfigTree createConfig(final String operationName, final String soapAction)
    {
        final ConfigTree configTree = new ConfigTree("wise-soap-client");
        configTree.setAttribute("wsdl", wsdl);
        configTree.setAttribute("SOAPAction", soapAction);
        configTree.setAttribute("EndPointName", endPointName);
        configTree.setAttribute("serviceName", serviceName);
        configTree.setAttribute("operationName", operationName);
        return configTree;
    }

    private ConfigTree createPropSubstitutionConfig(final String operationName, final String soapAction)
    {
        ConfigTree configTree = new ConfigTree("wise-soap-client");
        configTree.setAttribute("wsdl", wsdl);
        configTree.setAttribute("SOAPAction", soapAction);
        configTree.setAttribute("EndPointName", endPointName);
        configTree.setAttribute("serviceName", serviceName);
        configTree.setAttribute("operationName", operationName);
        
        ConfigTree wiseConfig = new ConfigTree("wise-config", configTree);
        ConfigTree wiseProp = new ConfigTree("wise-property", wiseConfig);
        wiseProp.setAttribute("name", "foo");
        wiseProp.setAttribute("value", "bar");
        
        return configTree;
    }
    
    private class MockSOAPClient extends SOAPClient
    {
        public MockSOAPClient(ConfigTree config) throws ConfigurationException
        {
            super(config);
        }
        
        @Override
        WSDynamicClient createClient(final String wsdl, final String serviceName, final String username, final String password) throws ActionProcessingException
        {
            try
            {
                return new MockWSDynamicClient(new WiseProperties("wise-core.properties"));
            } 
            catch (WiseException e)
            {
                throw new ActionProcessingException(e.getMessage(), e);
            }
        }
        
        @Override
        Map<String, WSEndpoint> getEndpoints(final WSDynamicClient client)
        {
            HashMap<String, WSEndpoint> map = new HashMap<String, WSEndpoint>();
            map.put(endPointName, new MockWSEndpoint(soapAction));
            return map;
        }
    }
    
}
