/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.bpel.uddi;

import java.net.URL;
import java.rmi.RemoteException;
import java.util.Map;
import java.util.Properties;

import javax.wsdl.Definition;
import javax.xml.namespace.QName;

import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.juddi.v3.annotations.AnnotationProcessor;
import org.apache.juddi.v3.client.config.UDDIClerk;
import org.apache.juddi.v3.client.config.UDDIClerkManager;
import org.apache.juddi.v3.client.mapping.BPEL2UDDI;
import org.apache.juddi.v3.client.mapping.AsyncRegistration;
import org.apache.juddi.v3.client.mapping.RegistrationInfo;
import org.apache.juddi.v3.client.mapping.RegistrationType;
import org.apache.juddi.v3.client.mapping.URLLocalizer;
import org.apache.juddi.v3.client.transport.TransportException;
import org.jboss.soa.bpel.runtime.JBossDSPFactory;
import org.jboss.soa.bpel.runtime.engine.ode.UDDIRegistration;
import org.jboss.soa.dsp.ws.WSDLReference;

/**
 * 
 * * @author Kurt T Stam <kurt.stam@jboss.com>
 *
 */
public class UDDIRegistrationImpl extends AnnotationProcessor implements UDDIRegistration {

	protected static final Log log = LogFactory.getLog(UDDIRegistrationImpl.class);
	private final static String BPEL_UDDI_CLERK = "bpel.uddi.clerk";
	private final static String BPEL_UDDI_CONFIG = "bpel.uddi.config";
	private final static String DEFAULT_BPEL_UDDI_CLERK = "BPELClerk";
	
	UDDIClerk bpelClerk = null;
	boolean selfRegister = false;
	URLLocalizer urlLocalizer = null;
	Properties properties = null;
	AsyncRegistration registration = null;
	
	public UDDIRegistrationImpl(Properties properties) {
		super();
		
		properties.put("nodeName", JBossDSPFactory.getServerConfig().getUddiNodeName());
		String bpelUDDIConfig = properties.getProperty(BPEL_UDDI_CONFIG);
		if (bpelUDDIConfig!=null) {
			try {
				UDDIClerkManager clerkManager = new UDDIClerkManager(bpelUDDIConfig, properties);
		        clerkManager.start();
		        selfRegister = clerkManager.getClientConfig().isRegisterOnStartup();
		        Map<String,UDDIClerk> clerks = clerkManager.getClientConfig().getUDDIClerks();
		        String clerkName = properties.getProperty(BPEL_UDDI_CLERK, DEFAULT_BPEL_UDDI_CLERK);
				bpelClerk = clerks.get(clerkName);
				//Add the properties from the uddi.xml
				properties.putAll(bpelClerk.getUDDINode().getProperties());
				if (bpelClerk==null) {
					throw new ConfigurationException("Could not find UDDI Clerk named "+ bpelClerk.getName());
				}
				this.properties = properties;
				urlLocalizer = new JBossURLLocalizer(properties);
			} catch (Exception e) {
				log.error(e.getMessage(),e);
			}
		} else {
			log.error("The 'bpel.uddi.config' property must be defined in the bpel.properties file");
		}
	}
	
	/**
	 * Releasing the resources we are holding (such as the subscriptionListener in the UDDIServiceCache).
	 * @throws TransportException 
	 * @throws ConfigurationException 
	 * @throws RemoteException 
	 */
	public void shutdown() throws Exception {
		registration.getServiceLocator(bpelClerk.getName()).shutdown();
	}
	/**
	 * Registers a BPEL ServiceEndpointReference (EPR) into a UDDI registry using the jUDDI client code.
	 * If the serviceKey does not already exist we register the service along with the EPR.
	 * 
	 * @param EPR
	 * @param metaData
	 * @param wsdlRef
	 */
	public void registerBPELProcess(QName serviceQName, String version, String portName, URL serviceUrl,
			URL wsdlURL, Definition wsdlDefinition) {
		if (selfRegister) {
			
			try {
				RegistrationInfo registrationInfo = new RegistrationInfo();
				registrationInfo.setServiceQName(serviceQName);
				registrationInfo.setVersion(version);
				registrationInfo.setPortName(portName);
				registrationInfo.setServiceUrl(serviceUrl);
				registrationInfo.setWsdlUrl(wsdlURL);
				registrationInfo.setWsdlDefinition(wsdlDefinition);
				registrationInfo.setRegistrationType(RegistrationType.BPEL);
				registration = new AsyncRegistration(bpelClerk, urlLocalizer, properties, registrationInfo);
				Thread thread = new Thread(registration);
				thread.start();
			} catch (Exception e) {
				log.error("Unable to register service " + serviceQName
						+ " ." + e.getMessage(),e);
			} catch (Throwable t) {
				log.error("Unable to register service " + serviceQName
						+ " ." + t.getMessage(),t);
			}
			
		}
		
	}
	/**
	 * UnRegisters the binding from the UDDI Registry. 
	 * @param EPR
	 * @param metaData
	 * @param wsdlRef
	 */
	public void unRegisterBPELEPR(QName serviceName, String portName, URL serviceURL) {
		if (selfRegister) {
			try {
				BPEL2UDDI bpel2UDDI = new BPEL2UDDI(bpelClerk, urlLocalizer, properties);
				String serviceKey = bpel2UDDI.unRegister(serviceName, portName, serviceURL);
				if (registration.getServiceLocator(bpelClerk.getName())!=null) {
					registration.getServiceLocator(bpelClerk.getName()).removeService(serviceKey);
				}
			} catch (Exception e) {
				log.error("Unable to unRegister EPR for " + serviceName
						+ " ." + e.getMessage(),e);
			} catch (Throwable t) {
				log.error("Unable to unRegister EPR for  " + serviceName
						+ " ." + t.getMessage(),t);
			}
		}
	}
	
	public String lookupEndpoint(QName serviceQName, String portName) {
		return registration.getServiceLocator(bpelClerk.getName()).lookupEndpoint(serviceQName, portName);
	}
	
	/**
	 * Looks up the WSDL for the requested service and portName. 
	 */
    public WSDLReference lookupWSDL(QName serviceQName, String portName) {
	
		log.info("Not yet supported.");
		return null;
	}
	
	
}
