/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.qa.teargas.message.generator;

import java.io.Serializable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.log4j.Logger;
import org.jboss.soa.esb.qa.teargas.message.sender.MessageSender;

/**
 *
 * @author mvecera
 */
public class ImmediateMessageGenerator extends AbstractMessageGenerator {
   private static final Logger log = Logger.getLogger(ImmediateMessageGenerator.class);
   private AtomicInteger counter = new AtomicInteger(0);
   protected long start = -1;
   protected long stop = -1;

   protected void setStartTime() {
      if (start == -1) {
         start = System.currentTimeMillis();
      }
   }

   protected void setStopTime() {
      if (stop == -1) {
         stop = System.currentTimeMillis();
      }
   }

   protected float getSpeed(int count) {
      long now = (stop == -1) ? System.currentTimeMillis() : stop;
      return 1000f * count / (now - start);
   }

   @Override
   public void generate(Serializable message, int count) throws Exception {
      setStartTime();

      log.info("Preparing senders");
      ExecutorService es = Executors.newFixedThreadPool(threads);
      for (int i = 0; i < count; i++) {
         es.submit(new SenderTask(counter, sender, message));
      }

      es.shutdown();
      boolean terminated = false;
      int lastValue = 0;
      while (!terminated) {
         try {
            terminated = es.awaitTermination(1, TimeUnit.SECONDS);

            // should we log a change?
            int cnt = counter.get();
            if (cnt != lastValue) {
               lastValue = cnt;
               log.info("Messages sent: " + cnt + " (" + getSpeed(cnt) + " msgs/s)");
            }
         } catch (InterruptedException ie) {
            // "Shit happens!", Forrest Gump
            }
      }

      setStopTime();
   }

   private static class SenderTask implements Runnable {
      private MessageSender sender;
      private Serializable message;
      private AtomicInteger counter;

      public SenderTask(AtomicInteger counter, MessageSender sender, Serializable message) {
         this.sender = sender;
         this.message = message;
         this.counter = counter;
      }

      public void run() {
         try {
            sender.send(message, null);
            counter.incrementAndGet();
         } catch (Exception e) {
            e.printStackTrace();
         }
      }
   }
}
