/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with this
 * work for additional information regarding copyright ownership. The ASF
 * licenses this file to You under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.apache.camel.component.infinispan.blueprint;

import java.util.List;
import org.apache.camel.Exchange;
import org.apache.camel.Message;
import org.apache.camel.Processor;
import static org.apache.camel.component.infinispan.InfinispanConstants.KEY;
import static org.apache.camel.component.infinispan.InfinispanConstants.RESULT;
import static org.apache.camel.component.infinispan.InfinispanConstants.VALUE;
import static org.apache.camel.component.infinispan.UserUtils.USERS;
import static org.apache.camel.component.infinispan.UserUtils.createKey;
import static org.apache.camel.component.infinispan.UserUtils.hasUser;
import org.apache.camel.test.blueprint.CamelBlueprintTestSupport;
import org.infinispan.protostream.sampledomain.User;
import org.junit.Before;
import org.junit.Test;

/**
 * @author mgencur
 */
public class InfinispanRemoteQueryProducerBlueprintIT extends CamelBlueprintTestSupport {

    @Override
    protected String getBlueprintDescriptor() {
        return "remote-query-blueprint.xml";
    }

    @Override
    @Before
    public void setUp() throws Exception {
        super.setUp();
        for (final User user : USERS) {
            Exchange request = template.request("direct:put",
                    new Processor() {
                        @Override
                        public void process(Exchange exchange) throws Exception {
                            Message in = exchange.getIn();
                            in.setHeader(KEY, createKey(user));
                            in.setHeader(VALUE, user);
                        }
                    });
            assertNull(request.getException());

        }
    }

    @Test
    public void producerQueryOperationWithoutQueryBuilder() throws Exception {
        Exchange request = template.request("direct:start1", null);
        assertNull(request.getException());

        @SuppressWarnings("unchecked")
        List<User> queryResult = (List<User>) request.getIn().getHeader(RESULT);
        assertNull(queryResult);
    }

    @Test
    public void producerQueryWithoutResult() throws Exception {
        Exchange request = template.request("direct:start2", null);
        assertNull(request.getException());

        @SuppressWarnings("unchecked")
        List<User> queryResult = (List<User>) request.getIn().getHeader(RESULT);
        assertNotNull(queryResult);
        assertEquals(0, queryResult.size());
    }

    @Test
    public void producerQueryWithResult() throws Exception {
        Exchange request = template.request("direct:start3", null);
        assertNull(request.getException());

        @SuppressWarnings("unchecked")
        List<User> queryResult = (List<User>) request.getIn().getHeader(RESULT);
        assertNotNull(queryResult);
        assertEquals(2, queryResult.size());
        assertTrue(hasUser(queryResult, "nameA", "surnameA"));
        assertTrue(hasUser(queryResult, "nameA", "surnameB"));
    }
}
