/*
 * Copyright 2020 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.kie.kogito.rules.units;

import java.util.HashMap;
import java.util.Map;

import org.kie.kogito.rules.DataSource;

public interface AssignableChecker {

    boolean isAssignableFrom(Class<?> source, Class<?> target);

    static AssignableChecker create(ClassLoader classLoader) {
        return create(classLoader, classLoader != DataSource.class.getClassLoader());
    }

    static AssignableChecker create(ClassLoader classLoader, boolean classLoaderSafe) {
        return classLoaderSafe ? new ClassLoaderSafeAssignableChecker(classLoader) : DummyAssignableChecker.INSTANCE;
    }

    enum DummyAssignableChecker implements AssignableChecker {

        INSTANCE;

        @Override
        public boolean isAssignableFrom(Class<?> source, Class<?> target) {
            return source.isAssignableFrom(target);
        }
    }

    class ClassLoaderSafeAssignableChecker implements AssignableChecker {
        private final ClassLoader classLoader;
        private final Map<String, Class<?>> classes = new HashMap<>();

        public ClassLoaderSafeAssignableChecker(ClassLoader classLoader) {
            this.classLoader = classLoader;
        }

        @Override
        public boolean isAssignableFrom(Class<?> source, Class<?> target) {
            return classes.computeIfAbsent(source.getCanonicalName(), name -> {
                try {
                    return classLoader.loadClass(name);
                } catch (ClassNotFoundException e) {
                    return source;
                }
            }).isAssignableFrom(target);
        }
    }
}
