package com.redhat.installer.installation.validator;


import com.izforge.izpack.Pack;
import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.installer.ParameteredDataValidator;
import com.izforge.izpack.util.ErrorIzpackPathException;
import com.izforge.izpack.util.FileUtil;
import com.izforge.izpack.util.WarningIzpackPathException;

import java.nio.file.InvalidPathException;
import java.util.ArrayList;
import java.util.List;

/**
 * Treats a field as a path validator for packs specified in a list.
 * This is needed in installer-commons because of poor izpack design.
 * Created by thauser on 20/07/17.
 */
public class PathLengthValidator extends ParameteredDataValidator {
    private String message;


    public Status checkPathLength(String path, List<Pack> packs) {
        int longest = 0;
        for (Pack pi : packs) {
            longest = (longest < pi.longestPath) ? pi.longestPath : longest;
        }
        try {
            if (!FileUtil.isPathValidOffset(path, longest)) {
                return Status.ERROR;
            }
        } catch (WarningIzpackPathException wipe) {
            message = wipe.getMessage();
            return Status.WARNING;
        } catch (ErrorIzpackPathException eipe){
            message = eipe.getMessage();
            return Status.ERROR;
        }
        return Status.OK;
    }

    @Override
    public Status validateData(AutomatedInstallData adata) {
        if (getParameters().isEmpty()) {
            return Status.ERROR;
        }
        String packString = getParameters().get("packs");
        String variableString = getParameters().get("variable");
        List<Pack> packList = new ArrayList<>();
        for (String packId : packString.split(",")) {
            Pack toAdd = adata.getPackage(packId);
            if (toAdd == null) {
                // invalid configuration
                message = "Invalid configuration in IzPack xmls.";
                return Status.ERROR;
            }
            packList.add(adata.getPackage(packId));
        }
        return checkPathLength(adata.getVariable(variableString), packList);
    }

    @Override
    public String getErrorMessageId() {
        return null;
    }

    @Override
    public String getWarningMessageId() {
        return null;
    }

    @Override
    public boolean getDefaultAnswer() {
        return false;
    }

    @Override
    public String getFormattedMessage() {
        return message;
    }
}
