package com.redhat.installer.tests.installation.processpanel;

import com.redhat.installer.framework.mock.MockResourceBuilder;
import com.redhat.installer.framework.testers.ProcessPanelTester;
import com.redhat.installer.installation.processpanel.TomcatAddUser;
import com.redhat.installer.tests.TestUtils;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.parser.Parser;
import org.jsoup.select.Elements;
import org.junit.After;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.stream.Collectors;

import static junit.framework.TestCase.assertFalse;
import static junit.framework.TestCase.assertTrue;

public class TomcatAddUserTest extends ProcessPanelTester {

    private final String existingUsersXml = "/tomcat/tomcat-users.xml";
    private final String noExistingUsersXml = "/tomcat/tomcat-users-noexisting.xml";
    private final String mockDigestExecutable = "/tomcat/mock-digest.sh";
    private final String copiedDigest = "/bin/digest.sh";
    private final String copiedXml = "/conf/tomcat-users.xml";
    private final String copiedServer = "/conf/server.xml";
    private final String existingCredentialsServer = "/tomcat/server-existing-creds.xml";
    private final String userTemplate = "<user password=\"%s\" roles=\"%s\" username=\"%s\"/>";
    private final String testUsername = "testerinousername";
    private final String testPassword = "passw*rd1!";
    private final String testRoleList = "rolelisttest";
    private final String mockHashValue = "hashed-password-value-here";
    private final String[] testArgs = new String[]{testUsername, testPassword, testRoleList};

    @Before
    public void setUp() throws Exception {
        idata.langpack = TestUtils.createMockLangpack(tempFolder, "", "");
    }

    @After
    public void cleanup() throws Exception {
        if (Files.exists(Paths.get(idata.getInstallPath() + copiedXml))) {
            MockResourceBuilder.deleteFile(idata.getInstallPath() + copiedXml);
        }
    }

    @Test
    public void testSuccessExistingUsers() throws Exception {
        MockResourceBuilder.copyResourceToInstall(existingUsersXml, idata.getInstallPath() + copiedXml);
        boolean result = TomcatAddUser.run(handler, testArgs);
        assertCorrect(result);
    }

    @Test
    public void testNoExistingUsers() throws Exception {
        MockResourceBuilder.copyResourceToInstall(noExistingUsersXml, idata.getInstallPath() + copiedXml);
        boolean result = TomcatAddUser.run(handler, testArgs);
        assertCorrect(result);
    }

    @Ignore("Hashing functionality removed.")
    @Test
    public void testHashedPassword() throws Exception {
        MockResourceBuilder.copyResourceToInstall(noExistingUsersXml, idata.getInstallPath() + copiedXml);
        MockResourceBuilder.copyExecutableToInstall(mockDigestExecutable, idata.getInstallPath() + copiedDigest);
        MockResourceBuilder.copyResourceToInstall("/tomcat/server.xml", idata.getInstallPath() + copiedServer);
        boolean result = TomcatAddUser.run(handler, testArgs);
        assertTrue(result);
        String newFileString = getFileContentAsString();

        // check that the base tags are unmodified
        Document modifiedXml = Jsoup.parse(newFileString, "", Parser.xmlParser());
        Elements tomcatUsersElement = modifiedXml.getElementsByTag("tomcat-users");
        assertFalse(tomcatUsersElement.isEmpty());

        // check that our fake hashed value is present, and not the original password value.
        assertTrue(newFileString.contains(String.format(userTemplate, mockHashValue, testRoleList, testUsername)));
        assertFalse(newFileString.contains(testPassword));
    }

    @Ignore("Hashing functionality removed.")
    @Test
    public void testUseExistingCredentialHandler() throws Exception {
        MockResourceBuilder.copyResourceToInstall(noExistingUsersXml, idata.getInstallPath() + copiedXml);
        MockResourceBuilder.copyExecutableToInstall(mockDigestExecutable, idata.getInstallPath() + copiedDigest);
        MockResourceBuilder.copyExecutableToInstall(existingCredentialsServer, idata.getInstallPath() + copiedServer);
        boolean result = TomcatAddUser.run(handler, testArgs);
        assertTrue(result);

        String newFileString = getFileContentAsString();
        Document modifiedXml = Jsoup.parse(newFileString, "", Parser.xmlParser());
        Elements tomcatUsersElement = modifiedXml.getElementsByTag("tomcat-users");
        assertFalse(tomcatUsersElement.isEmpty());

        // check that our fake hashed value is present, and not the original password value.
        assertTrue(newFileString.contains(String.format(userTemplate, "used-existing-handler-settings", testRoleList, testUsername)));
        assertFalse(newFileString.contains(testPassword));
    }

    @Test
    public void testNoFileDetected() {
        boolean result = TomcatAddUser.run(handler, testArgs);
        assertFalse(result);
    }

    private void assertCorrect(boolean runResult) throws Exception {
        assertTrue(runResult);
        String newFileString = Files.lines(Paths.get(tempFolder.getRoot().getPath() + copiedXml)).collect(Collectors.joining("\n"));
        // check that the base tags are unmodified
        Document modifiedXml = Jsoup.parse(newFileString, "", Parser.xmlParser());
        Elements tomcatUsersElement = modifiedXml.getElementsByTag("tomcat-users");
        assertFalse(tomcatUsersElement.isEmpty());
        // our user is added
        assertTrue(newFileString.contains(String.format(userTemplate, testPassword, testRoleList, testUsername)));
    }

    private String getFileContentAsString() throws IOException {
        return Files.lines(Paths.get(tempFolder.getRoot().getPath() + copiedXml)).collect(Collectors.joining("\n"));
    }
}
