/**
 *  Copyright 2005-2016 Red Hat, Inc.
 *
 *  Red Hat licenses this file to you under the Apache License, version
 *  2.0 (the "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *  implied.  See the License for the specific language governing
 *  permissions and limitations under the License.
 */
package io.fabric8.itests.paxexam.basic;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.Set;

import io.fabric8.api.FabricService;
import io.fabric8.api.ServiceProxy;
import io.fabric8.itests.paxexam.support.ContainerBuilder;
import io.fabric8.itests.paxexam.support.ContainerProxy;
import io.fabric8.itests.paxexam.support.FabricTestSupport;
import io.fabric8.itests.paxexam.support.Provision;
import io.fabric8.zookeeper.ZkPath;
import io.fabric8.zookeeper.utils.ZooKeeperUtils;
import org.apache.curator.framework.CuratorFramework;
import org.apache.http.HttpResponse;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.nio.entity.NFileEntity;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.ops4j.pax.exam.Configuration;
import org.ops4j.pax.exam.Option;
import org.ops4j.pax.exam.junit.PaxExam;
import org.ops4j.pax.exam.karaf.options.KarafDistributionOption;
import org.ops4j.pax.exam.options.DefaultCompositeOption;
import org.ops4j.pax.exam.spi.reactors.ExamReactorStrategy;
import org.ops4j.pax.exam.spi.reactors.PerMethod;

@RunWith(PaxExam.class)
@ExamReactorStrategy(PerMethod.class)
public class FabricMavenProxyTest extends FabricTestSupport {

    @Test
    public void testUpload() throws Exception {
        String featureLocation = System.getProperty("feature.location");
        System.out.println("Testing with feature from:" + featureLocation);
        System.out.println(executeCommand("fabric:create -n --wait-for-provisioning"));
        //System.out.println(executeCommand("shell:info"));
        //System.out.println(executeCommand("fabric:info"));
        //System.out.println(executeCommand("fabric:profile-list"));

        ServiceProxy<FabricService> fabricProxy = ServiceProxy.createServiceProxy(bundleContext, FabricService.class);
        try {
            Set<ContainerProxy> containers = ContainerBuilder.create(fabricProxy, 2).withName("maven").withProfiles("fabric").waitForProvisioning(10 * 60 * 1000L).assertProvisioningResult().build();
            try {
                List<String> uploadUrls = new ArrayList<String>();
                ServiceProxy<CuratorFramework> curatorProxy = ServiceProxy.createServiceProxy(bundleContext, CuratorFramework.class);
                try {
                    CuratorFramework curator = curatorProxy.getService();
                    List<String> children = ZooKeeperUtils.getChildren(curator, ZkPath.MAVEN_PROXY.getPath("upload"));
                    for (String child : children) {
                        String uploadeUrl = ZooKeeperUtils.getSubstitutedPath(curator, ZkPath.MAVEN_PROXY.getPath("upload") + "/" + child);
                        uploadUrls.add(uploadeUrl);
                    }
                } finally {
                    curatorProxy.close();
                }
                //Pick a random maven proxy from the list.
                Random random = new Random();
                int index = random.nextInt(uploadUrls.size());
                String targetUrl = uploadUrls.get(index);

                String uploadUrl = targetUrl + "itest/itest/1.0/itest-1.0-features.xml";
                System.out.println("Using URI: " + uploadUrl);
                BasicCredentialsProvider credentialsProvider = new BasicCredentialsProvider();
                credentialsProvider.setCredentials(AuthScope.ANY, new UsernamePasswordCredentials("admin", "admin"));
                CloseableHttpClient client = HttpClientBuilder.create()
                        .setDefaultCredentialsProvider(credentialsProvider)
                        .build();
                HttpPut put = new HttpPut(uploadUrl);

                NFileEntity entity = new NFileEntity(new File(featureLocation), ContentType.TEXT_XML);
                put.setEntity(entity);
                HttpResponse response = client.execute(put);
                System.out.println("Response:" + response.getStatusLine());
                Assert.assertTrue(response.getStatusLine().getStatusCode() == 200 || response.getStatusLine().getStatusCode() == 202);

                System.out.println(executeCommand("fabric:profile-edit --repository mvn:itest/itest/1.0/xml/features default"));
                System.out.println(executeCommand("fabric:profile-edit --feature example-cbr default"));
                Provision.containerStatus(containers, PROVISION_TIMEOUT);
            } finally {
                ContainerBuilder.destroy(containers);
            }
        } finally {
            fabricProxy.close();
        }
    }

    @Configuration
    public Option[] config() {
        return new Option[] {
                new DefaultCompositeOption(fabricDistributionConfiguration()),
                mavenBundle("org.apache.httpcomponents", "httpcore-osgi").versionAsInProject(),
                mavenBundle("org.apache.httpcomponents", "httpclient-osgi").versionAsInProject(),
                mavenBundle("io.fabric8", "fabric-project-deployer").versionAsInProject(),
                mavenBundle("io.fabric8", "fabric-maven-proxy").versionAsInProject(),
                KarafDistributionOption.editConfigurationFilePut("etc/system.properties", "feature.location",
                        FabricMavenProxyTest.class.getResource("/test-features.xml").getFile()), KarafDistributionOption.debugConfiguration("5005", false) };
    }
}
