/*
 * RHQ Management Platform
 * Copyright (C) 2005-2008 Red Hat, Inc.
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package org.rhq.enterprise.agent.promptcmd;

import java.io.PrintWriter;
import java.util.prefs.Preferences;

import mazz.i18n.Msg;

import org.rhq.core.util.exception.ThrowableUtil;
import org.rhq.enterprise.agent.AgentMain;
import org.rhq.enterprise.agent.i18n.AgentI18NFactory;
import org.rhq.enterprise.agent.i18n.AgentI18NResourceKeys;

/**
 * Sets an agent configuration preference and sets a system property to enable it to override any existing preference.
 *
 * @author John Mazzitelli
 */
public class SetConfigPromptCommand implements AgentPromptCommand {
    private static final Msg MSG = AgentI18NFactory.getMsg();

    /**
     * @see AgentPromptCommand#getPromptCommandString()
     */
    public String getPromptCommandString() {
        return MSG.getMsg(AgentI18NResourceKeys.SETCONFIG);
    }

    /**
     * @see AgentPromptCommand#execute(AgentMain, String[])
     */
    public boolean execute(AgentMain agent, String[] args) {
        PrintWriter out = agent.getOut();

        if (args.length != 2) {
            out.println(MSG.getMsg(AgentI18NResourceKeys.HELP_SYNTAX_LABEL, getSyntax()));
        } else {
            String prop = args[1];
            int i = prop.indexOf("=");
            String name;
            String value;

            Preferences prefs = agent.getConfiguration().getPreferences();

            if (i == -1) {
                name = prop;

                // removes the named preference from both system properties and agent configuration
                System.getProperties().remove(name);
                prefs.remove(name);

                out.println(MSG.getMsg(AgentI18NResourceKeys.SETCONFIG_REMOVED, name));
            } else {
                name = prop.substring(0, i);
                value = prop.substring(i + 1, prop.length());

                // set the named preference in both system properties and agent configuration
                System.setProperty(name, value);
                prefs.put(name, value);

                out.println(MSG.getMsg(AgentI18NResourceKeys.SETCONFIG_SET, name, value));
            }

            // make sure we push the changes to the backing store
            flush(prefs, out);
        }

        return true;
    }

    /**
     * @see AgentPromptCommand#getSyntax()
     */
    public String getSyntax() {
        return MSG.getMsg(AgentI18NResourceKeys.SETCONFIG_SYNTAX);
    }

    /**
     * @see AgentPromptCommand#getHelp()
     */
    public String getHelp() {
        return MSG.getMsg(AgentI18NResourceKeys.SETCONFIG_HELP);
    }

    /**
     * @see AgentPromptCommand#getDetailedHelp()
     */
    public String getDetailedHelp() {
        return MSG.getMsg(AgentI18NResourceKeys.SETCONFIG_DETAILED_HELP);
    }

    private void flush(Preferences prefsToFlush, PrintWriter out) {
        try {
            prefsToFlush.flush();
        } catch (Exception e) {
            out.println(MSG.getMsg(AgentI18NResourceKeys.SETCONFIG_FLUSH_FAILED, ThrowableUtil.getAllMessages(e)));
        }
    }
}