/*
 * Copyright (C) 2018 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.optaplanner.openshift.employeerostering.gwtui.client.rostergrid.list;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;

import elemental2.dom.HTMLElement;

public class ListView<C, T> {

    private Supplier<ListElementView<C, T>> viewFactory;
    private C container;
    private HTMLElement htmlParentElement;

    private List<T> objects;
    private Map<T, ListElementView<C, T>> views;

    private boolean init = false;

    public void init(final HTMLElement htmlParentElement,
                     final C container,
                     final List<T> objects,
                     final Supplier<ListElementView<C, T>> viewFactory) {

        reset();
        this.init = true;
        this.container = container;
        this.htmlParentElement = htmlParentElement;
        this.viewFactory = viewFactory;
        this.objects = objects;
        this.views = new HashMap<>();
        objects.forEach(this::addViewFor);
    }

    private void reset() {
        if (init) {
            // Has to be a copy because the remove method removes things from the objects list.
            new HashSet<>(objects).forEach(this::remove);
        }
    }

    public void addAfter(final T object, final T newObject) {
        final ListElementView<C, T> view = viewFactory.get().setup(newObject, this);
        views.put(newObject, view);

        int next = objects.indexOf(object) + 1;

        if (next <= 0 || next >= objects.size()) {
            htmlParentElement.appendChild(view.getElement());
            objects.add(newObject);
        } else {
            htmlParentElement.insertBefore(view.getElement(), views.get(objects.get(next)).getElement());
            objects.add(next, newObject);
        }
    }

    public void add(final T newObject) {
        addViewFor(newObject);
        objects.add(newObject);
    }

    private void addViewFor(final T newObject) {
        final ListElementView<C, T> view = viewFactory.get().setup(newObject, this);
        views.put(newObject, view);
        htmlParentElement.appendChild(view.getElement());
    }

    public void remove(final T object) {
        final ListElementView<C, T> view = views.remove(object);
        view.destroy();
        view.getElement().remove();
        objects.remove(object);
    }

    public void clear() {
        reset();
    }

    public List<T> getObjects() {
        return objects;
    }

    public boolean isEmpty() {
        return objects.isEmpty();
    }

    public ListElementView<C, T> getView(final T obj) {
        return views.get(obj);
    }

    public void addOrReplace(final T obj, final T newObj) {
        if (!objects.contains(obj)) {
            add(newObj);
        } else {
            addAfter(obj, newObj);
            remove(obj);
        }
    }

    public C getContainer() {
        return container;
    }

    public HTMLElement getHTMLParentElement() {
        return htmlParentElement;
    }
}
