//////////////////////////////////////////////////////////////////////////////
// check.C

#define WITH_DEPGRAPH

#include "dl.h"
#include "check.h"
#include "timer.h"
#include <stdio.h>
#include "generate.h"

extern bool run_satz(
    const GRULES &,
    const GCONSTRAINTS &,
    const GATOMSET &,
    GATOMSET * );

// Activate some global timers which are displayed with -stats and -stats++.
#define CHECK_BENCHMARK
#ifdef CHECK_BENCHMARK
static double   global_worst_time = 0.0;
static double   global_worst_inner_time = 0.0;
static unsigned global_max_rwiplus = 0;
#endif

static GATOMSET *global_ufset;        // Not really elegant, but works as long
static bool *global_ufsetInitialized; // as we do not need to be reentrant.

//////////////////////////////////////////////////////////////////////////////
/** performs one iteration of the R operator
*/
template< class GPROGRAM_TYPE >
static bool R(
    const GPROGRAM_TYPE &P,
    const GINTERPRET &I,
          GATOMSET &X,
    unsigned &sizeX )
   {
   bool XwasChanged=false;
   for( typename GPROGRAM_TYPE::const_iterator R = P.begin(); 
        R != P.end(); 
        R++ )
      {
      bool firstOcc=true;
      for( GDISJUNCTION::const_iterator atom = (*R).getHead().begin();
           atom != (*R).getHead().end();
           atom++ )
         if( X.contains(*atom) )
            {
            if( firstOcc )
               {
               firstOcc=false;
               if( (*R).getBody() )
                  {
                  bool condII=true;
                  for( GCONJUNCTION::const_iterator L=(*R).getBody()->begin();
                       (L != (*R).getBody()->end() && condII);
                       L++ )
                     {
                     if( (*L).isNegative() )
                        {
                        if( I.isTotallyTrue((GATOM)*L) )
                           condII=false; // not violated
                        }
                     else
                        {
                        if( I.isFalse((GATOM)*L) || X.contains((GATOM)*L) )
                           condII=false;
                        }
                     }
                  if( ! condII )
                     break; // check next atom in disjunction
                  }
               }
            bool condIII=true;
            for( GDISJUNCTION::const_iterator A=(*R).getHead().begin();
                 ( (A != (*R).getHead().end()) && condIII );
                 A++ )
               {
               // The first condition below relies on the fact that we do
               // not have duplicate atoms in the head.
               if( A != atom  &&  I.isTotallyTrue(*A) )
                  condIII=false; // not violated
               }

            if( condIII ) // condII and condIII both violated
               {
               sizeX--;
               X.erase(*atom);
               XwasChanged=true;
               }
            } // for atom in disj; if x.contains(atom)
      } // for rule
   return XwasChanged;
   }

//////////////////////////////////////////////////////////////////////////////
template< class GPROGRAM_TYPE >
void FixpointR(
    const GPROGRAM_TYPE &P,
    const GINTERPRET &I,
          GATOMSET   &X,
          unsigned   &sizeX )
    {
    while( R(P, I, X, sizeX) ) { };
    }

//////////////////////////////////////////////////////////////////////////////
// RULE_FACTORY is used in addTransformedRule() to abstract from the rule
// generation which may be different for TRULE<GATOM,GLITERAL> and other
// forms of rules.

template<class DUMMY> class RULE_FACTORY { };


template <>
//////////////////////////////////////////////////////////////////////////////
class RULE_FACTORY<GRULE>
   {
   TDISJUNCTION<GATOM>    head;
   TCONJUNCTION<GLITERAL> body;
   vector< GRULE > *r;
   vector< GCONSTRAINT > *c;
   bool head_empty, body_empty;

   public:
   RULE_FACTORY(vector<GRULE> &_r, vector<GCONSTRAINT> &_c) : r(&_r), c(&_c),
      head_empty(true), body_empty(true)
      { }
   inline void addToHead(const GATOM &atom)
      {
      head.add(atom);
      head_empty = false;
      }
   inline void addToBody(const GLITERAL &lit)
      {
      assert(! lit.isNegative());
      body.add(lit);
      body_empty = false;
      }

   // returns true iff a rule or constraint was added.
   inline bool commit(void)
      {
      if(! head_empty)
         {
         r->push_back(GRULE(&head, &body));

         return true;
         }
      else if(! body_empty)
         {
         c->push_back(GCONSTRAINT(body));

         return true;
         }
      else
         {
         // empty rule (!)
         assert(0);
         return false;
         }
      }
   };
// end of class RULE_FACTORY< TRULE<GATOM,GLITERAL> >

//////////////////////////////////////////////////////////////////////////////
/** This function is used by addTransformedRule.
   A rule is relevant if
   * its body is not false and 
   * its head intersect I+ contains atoms from Rw(I+) and
   * it is not positive recursive
*/
template<class RULE_TYPE>
static bool isRuleRelevantForTransformation(
    const RULE_TYPE &R,
    const GINTERPRET &I,
    const GATOMSET &Rw_Iplus )
   {
   if( R.getBody() )
   for( GCONJUNCTION::const_iterator lit = R.getBody()->begin();
        (lit != R.getBody()->end());
        lit++ )
      {
      // FUTURE FIXME: If we introduce must-be-false, then the
      // following call should be updated to isTotallyFalse()
      // [succeeds if the atom is either false or must-be-false].
      if( I.isFalse(*lit) )
        {
        return false; // body is false -> rule does not restrict X
        }

      // the following code is executed if the literal *lit is not negated -
      // this is independent from the interpretation I.
      if( ! (*lit).isNegative() )
         {
         // check if body atom also appears in the head
         for( GDISJUNCTION::const_iterator atom = R.getHead().begin();
            atom != R.getHead().end();
            atom++ )
            {
            if(*lit == *atom)
               {
               return false; // rule is positive recursive and does not
                             // restrict any unfounded set X
               }
            }
         }
      }

   bool headtrue = false;

   for( GDISJUNCTION::const_iterator atom = R.getHead().begin();
      atom != R.getHead().end();
      atom++ )
      {
      if(I.isTotallyTrue(*atom))
         {
         headtrue = true;

         if(! Rw_Iplus.contains(*atom))
            {
            return false; // head intersect i is not a subset of Rw ->
                          // rule does not restrict X
            }
         }
      }

   return headtrue;
   }


//////////////////////////////////////////////////////////////////////////////
/** This function transforms a rule in a program which is to be checked for
   unfounded-freeness into a rule for UNSAT.
   If a rule is irrelevant, it is not inserted into the program dest_rules.
   Returns true iff a rule was added to dest_rules
*/
template<class RULE_FACTORY_TYPE>
static bool addTransformedRule(
    const TRULE<GATOM,GLITERAL> &src_rule,
    const GINTERPRET            &I,
    const GATOMSET              &Rw_Iplus,
    RULE_FACTORY_TYPE           &rf,
    bool transform = true )
   {
   if( isRuleRelevantForTransformation(src_rule, I, Rw_Iplus) )
      {
      if( src_rule.getBody() )
      for( GCONJUNCTION::const_iterator lit = src_rule.getBody()->pos_begin();
           lit != src_rule.getBody()->pos_end();
           lit++ )
         {
         if( Rw_Iplus.contains(lit->getAtom()) )
            {
            if(transform)
               {
               // body atoms are added to head
               rf.addToHead(lit->getAtom());
               }
            else
               {
               rf.addToBody(*lit);
               }
            }
         }

      for( GDISJUNCTION::const_iterator atom = src_rule.getHead().begin();
           atom != src_rule.getHead().end();
           atom++ )
         {
         if( I.isTotallyTrue(*atom) )
            {
            if(transform)
               {
               // head atoms are added to body
               rf.addToBody(*atom);
               }
            else
               {
               rf.addToHead(*atom);
               }
            }
         }

      rf.commit();

      return true;
      }
   else
      {
      // no rule added
      return false;
      }
   }


//////////////////////////////////////////////////////////////////////////////
template<class RULE_FACTORY_TYPE>
static bool addSimplifiedRule(
    const TRULE<GATOM,GLITERAL> &src_rule,
    const GINTERPRET            &I,
    const GATOMSET              &Rw_Iplus,
    RULE_FACTORY_TYPE           &rf )
   {
   return addTransformedRule(src_rule, I, Rw_Iplus, rf, false);
   }
// end of addSimplifiedRule()


//////////////////////////////////////////////////////////////////////////////
template<class T>
static ostream &operator<<(ostream &o, const vector<T> &v)
   {
   for(unsigned i = 0; i < v.size(); i++)
      {
      o << v[i] << "  ";
      }
   o << endl;
   return o;
   }

//////////////////////////////////////////////////////////////////////////////
// returns the number of transformed rules; excluding the big disjunction
template< class GPROGRAM_TYPE >
static unsigned TransformAllRules(
    const GPROGRAM_TYPE &P,
    const GINTERPRET    &I,
    const GATOMSET      &posAssumptions,
    const GATOMSET      &Rw_Iplus,
    GRULES       &r,
    GCONSTRAINTS &c )
    {
    assert(r.size() == 0);
    assert(c.size() == 0);

    // First generate a big disjunction that enforces that the unfounded set
    // to be found is nonempty. Instead of considering all atoms in the
    // (residual) Herbrand Base, we only take those that are also contained
    // in Rw_Iplus and the set of positive assumptions in the Model Generator.
    GDISJUNCTION head;
    for(unsigned i = 0; i < GATOMSET::getSize(); i++)
        {
#if CHECK_OPT >= 1
        const GATOM a(i);

        if( Rw_Iplus.contains(a)  &&  posAssumptions.contains(a) )
            head.add(a);
#else
        const ATOM &a = GATOM(i).getTheRealOne();

        if( Rw_Iplus.contains(GATOM(a)) && posAssumptions.contains(GATOM(a)) )
            head.add(GATOM(i));
#endif
        }

    assert( head.begin() != head.end() );
    r.push_back(GRULE(&head));

    if( CTraceLevel >= 2 )
        cdebug << " Rw_Iplus = " << Rw_Iplus << endl
               << " posAssumptions = " << posAssumptions << endl;

    unsigned cnt_transformed = 0;

    // Now transform the remaining rules.
    for( typename GPROGRAM_TYPE::const_iterator R = P.begin();
         R != P.end();
         R++ )
        {
        RULE_FACTORY<GRULE> rf(r, c);

        if(addTransformedRule(*R, I, Rw_Iplus, rf))
            {
            cnt_transformed++;
            }
        }

    if( CTraceLevel >= 2 )
        {
        cdebug << " Input for SAT solver:" << endl << "  ";
        print_list(cdebug,r,"\n  ");
        cdebug << endl;
        if( ! c.empty() )
            {
            cdebug << "  ";
            print_list(cdebug,c,"\n  ");
            }
        cdebug << endl;
        }

    return cnt_transformed;
    }



//////////////////////////////////////////////////////////////////////////////
/** This function invokes the Model Generator to solve the SAT problem
   obtained by transforming the program. 
*/
template< class GPROGRAM_TYPE >
static bool isUNSAT_MG(
    const GPROGRAM_TYPE &P,
    const GINTERPRET    &I,
    const GATOMSET      &posAssumptions,
    const GATOMSET      &Rw_Iplus )
    {
    GRULES       r;
    GCONSTRAINTS c;

    if( CTraceLevel >= 1 )
        {
        cdebug << endl
               << ">>>>> Model Checker -> Model Generator begin"
               << endl;
        }

    if( TransformAllRules(P, I, posAssumptions, Rw_Iplus, r, c) == 0 )
        {
        // satisfiable - there are atoms left but no rules that constrain
        // them -> not unfounded-free

        if( CTraceLevel >= 1 )
            {
            cdebug << "<<<<< Model Checker -> Model Generator end"
                   << endl
                   << endl;
            }

        return false;
        }


    // data structures are ready

    if( CTraceLevel >= 2 )
        {
        cdebug << "Trying to find a model for" << endl << "   ";
        print_list(cdebug,r,"\n   ");
        print_list(cdebug,c,"\n   ");
        cdebug << endl;
        }

    // Dummy required by the constructor of MODEL_GENERATOR.
    GWEAKCONSTRAINTS dummy_weakconstraints;

    MODEL_GENERATOR mg(r,c,
        dummy_weakconstraints,
        0,0,0,
        1,
        0,
        MODEL_GENERATOR::DEFAULT,
        0,0,false,0,0,0,0,0,0,0,
        HeuristicSequenceChecker,HeuristicCombinationSequence );


    mg.run();

    assert( mg.getNumberOfModelsComputed() <= 1 );

    bool unsat = (mg.getNumberOfModelsComputed() == 0);

    if( CTraceLevel >= 1 )
        {
        cdebug << "<<<<< Model Checker -> Model Generator end"
               << endl
               << endl;
        }

    return unsat;
    }

//////////////////////////////////////////////////////////////////////////////
/** This procedure just decides which "inner" algorithm to select and runs
   the right code.  It also updates global_worst_inner_time.

   When this procedure is called from the monolithic outer algorithm,
   P can be either a component subprogram or the full GroundIDB.
   The size of Rw_Iplus must be greater than 0.
   Rw_Iplus must be a subset of the positive part of I.

   The template argument is needed because in isUnfoundedFree_Monolithic(),
   a GPROGRAM_TYPE is used which is different from all other calling procedures.
*/
template< class GPROGRAM_TYPE >
static bool ExecuteInnerMCAlgorithm(
    const GPROGRAM_TYPE &P,
    const GINTERPRET &I,
    const GATOMSET   &posAssumptions,
    const GATOMSET   &Rw_Iplus,
    const unsigned    Rw_Iplus_size,
    unsigned mode )
    {
    bool uff = true;

#ifdef CHECK_BENCHMARK
    if(Rw_Iplus_size > global_max_rwiplus)
        {
        // global_max_rwiplus is a global variable
        global_max_rwiplus = Rw_Iplus_size;
        }
#endif

    if( CTraceLevel >= 1 )
        cdebug << "Starting inner model checking algorithm for non-hcf "
                  "components (|Rw(I+)| = " << Rw_Iplus_size << ")"
               << endl;

#ifdef CHECK_BENCHMARK
    TIMER t;
    t.start();
#endif

    if(Rw_Iplus_size == 0)
        {
        // unfounded-free - the transformed program is empty.
        // This branch can only be taken when the computation of the
        // fixpoint of the R operator was bypassed.
        uff = true;
        }
    else if (mode == MCM_USEMODELGENERATOR)
        {
        uff = isUNSAT_MG(P, I, posAssumptions, Rw_Iplus);
        }
    else if (mode == MCM_CALLDAVISPUTNAM)
        {
        GRULES       r;
        GCONSTRAINTS c;

        if( TransformAllRules(P, I, posAssumptions, Rw_Iplus, r, c) == 0 )
            {
            if( CTraceLevel >= 1 )
                cdebug << "Empty input after transformation, "
                          "not invoking SAT solver." << endl;

            uff = false;
            }
        else
            {
            uff = ! run_satz(r, c, Rw_Iplus, global_ufset);
            if( ! uff )
                if( global_ufsetInitialized )
                    *global_ufsetInitialized=true;

#ifndef NDEBUG
            if( ! uff  &&  global_ufset != 0 )
                {
                const unsigned oldsize=global_ufset->size();
                unsigned size=oldsize;

                R(P,I,*global_ufset,size);
                
                assert( size == oldsize );
                }
#endif
            }
        }
    else
        InternalError("model check");

#ifdef CHECK_BENCHMARK
    t.stop();
    double alg_time = t.getAsDouble();

    if( alg_time > global_worst_inner_time )
        global_worst_inner_time = alg_time;
#endif

    return uff; // return if unfounded-free
    }


//////////////////////////////////////////////////////////////////////////////
/** used by IsUnfoundedFree_Split() and the Model Generator.
*/
bool IsComponentStable(
    DEPGRAPH<GATOM,GCONSTRAINTS,GPROGRAM> *depgraph,
    const GINTERPRET &interpretation,
    const GATOMSET   &posAssumptions,
    const unsigned    i, // component_index
    const MCM_MODE    nonhcf_mode )
    { 
    GATOMSET X;
    unsigned sizeX=0;

    for( unsigned j = 0; j < (depgraph->getComponents())[i].size(); j++ )
        {
        const GATOM &atom=GATOM(((depgraph->getComponents())[i][j]));
        
        if( interpretation.isTotallyTrue(atom)
            && atom.hasToBeCheckedForSupportedness() // exclude aggregates
            )
            {
            sizeX++;
            X.add(atom);
            }
        }

    if( CTraceLevel >= 2 )
        {
        cdebug << "Unfounded-freeness for comp " << i
               << " of " << depgraph->getComponents().size() << " [ ";
        for( unsigned j=0;j< (depgraph->getComponents())[i].size();j++)
            {
            cdebug << GATOM((depgraph->getComponents())[i][j]) 
                   << " ";
            }
        cdebug << "]" << endl;
        cdebug << " GroundSubP=" << depgraph->getComponentRules(i) 
               << endl
               << " X=" << X   // component-specific part of I
               << endl;
        }
    
    
    FixpointR(depgraph->getComponentRules(i),interpretation,X,sizeX);
    if( CTraceLevel >= 2 )
        cdebug << "Fixed point of R = " << X << endl;
    
    if( sizeX > 0 ) // The fixpoint of R is non-empty.
        {
        if( depgraph->isHCF(i) )
            {
            if( CTraceLevel >= 2 )
                cdebug << "Comp. is hcf => NOT unfounded-free" << endl;
            
            return false;
            }
        else
            {
            if( ! ExecuteInnerMCAlgorithm( depgraph->getComponentRules(i),
                                           interpretation,
                                           posAssumptions,
                                           X, sizeX, nonhcf_mode) )
                return false;
            }
        }

    return true;
    }

//////////////////////////////////////////////////////////////////////////////
/** implements the old "outer" algorithm which first splits up programs into
   components and does the following with them:

   First, the fixpoint of R is computed. Then, the procedure finishes if it
   it empty or the component is hcf.
   Otherwise, ExecuteInnerAlgorithm() is called.
*/
static bool IsUnfoundedFree_Split( 
          DEPGRAPH<GATOM,GCONSTRAINTS,GPROGRAM> *depgraph,
    const GINTERPRET &interpretation,
    const GATOMSET   &posAssumptions,
    const MCM_MODE    nonhcf_mode )
//
//  Returns: true, if interpretation is unfounded-free with respect to all the 
//           subprograms generated by the *depgraph, false else.
   {
   for( unsigned i = 0; i < (depgraph->getComponents()).size(); i++ )
      {
      if( ! IsComponentStable( depgraph, interpretation,
                               posAssumptions,
                               i, nonhcf_mode) )
         {
         return false;
         }
      }
   return true;
   }


//////////////////////////////////////////////////////////////////////////////
/** implements the new "outer" algorithm that is proposed in the AI-JOURNAL
   paper.
   First, the fixpoint of R is computed for the whole program (since that
   can be computed in linear time), and only after that, the components
   of the already simplified program are computed.
*/
static bool IsUnfoundedFree_R_early(
    const GRULES     &GroundIDB,
    const GINTERPRET &interpretation,
    const GATOMSET   &posAssumptions,
    const unsigned    mode)
   {
   if(CTraceLevel >= 1)
      cdebug << "begin of IsUnfoundedFree_R_early  mode=" << mode << endl;

   GATOMSET X;
   for( size_t i=0; i < interpretation.size(); i++ )
      {
      if( interpretation.isTotallyTrue(GATOM(i)) &&
          GATOM(i).hasToBeCheckedForSupportedness() ) // exclude aggregates
          {
          X.add(GATOM(i));
          }
      }

   unsigned X_size = X.size();

#ifdef CHECK_BENCHMARK
   TIMER r_t;
   r_t.start();
#endif
   FixpointR(GroundIDB, interpretation, X, X_size);
#ifdef CHECK_BENCHMARK
   r_t.stop();
   if(CTraceLevel >= 1)
      cdebug << "Big R time = " << r_t << endl;
#endif

   GRULES simplified_program;
   GCONSTRAINTS dummy;

   for(GRULES::const_iterator i = GroundIDB.begin(); i != GroundIDB.end(); i++)
      {
      RULE_FACTORY<GRULE> rf(simplified_program, dummy);
      addSimplifiedRule(*i, interpretation, X, rf);
      }
   assert(dummy.size() == 0);

   if(CTraceLevel >= 1)
      cdebug << "GroundIDB: " << GroundIDB << endl
             << "Rw(I+): " << X << endl
             << "simp(P,I,Rw): " << simplified_program << endl;

   if(simplified_program.size() == 0)
      {
      if(CTraceLevel >= 1)
         cdebug << "simplified program is empty." << endl;

      return (X_size == 0); // no rules left -> unfounded-free
      }

   assert(X_size > 0);

   GCONSTRAINTS dummy_cons;

#ifdef CHECK_BENCHMARK
   TIMER dg_t;
   dg_t.start();
#endif
   DEPGRAPH<GATOM,GCONSTRAINTS,TPROGRAM<TRULE<GATOM,GLITERAL> > >
      simplified_depgraph(simplified_program, dummy_cons, 0);
#ifdef CHECK_BENCHMARK
   dg_t.stop();
   if(CTraceLevel >= 1)
      cdebug << "DG time = " << dg_t << endl;
#endif

   if(simplified_depgraph.isHCF())
      {
      if(CTraceLevel >= 1)
         cdebug << "whole program is HCF -> not unfounded-free" << endl;

      return false;
      }

   unsigned atoms_in_dg = 0;

   for( unsigned i = 0; i < simplified_depgraph.getComponents().size(); i++ )
      {
      if(CTraceLevel >= 1)
         cdebug << "Component " << i+1 << " ..." << endl
                << "SubP: " << simplified_depgraph.getComponentRules(i) << endl;

      GATOMSET atoms;
      unsigned y_size = 0;

      for( unsigned j = 0;
           j < (simplified_depgraph.getComponents())[i].size();
           j++ )
         {
         atoms.add(GATOM(((simplified_depgraph.getComponents())[i][j])));
         y_size++;
         atoms_in_dg++;
         }

      if(CTraceLevel >= 1)
         cdebug << "Atoms: " << atoms << endl;

      if(simplified_depgraph.isHCF(i))
         {
         unsigned y_size1 = y_size;

#ifdef CHECK_BENCHMARK
         TIMER r_t;
         r_t.start();
#endif
         FixpointR( simplified_depgraph.getComponentRules(i), interpretation,
                    atoms, y_size);
#ifdef CHECK_BENCHMARK
         r_t.stop();
         if(CTraceLevel >= 1)
            cdebug << "Sub - R time = " << r_t << endl;
#endif

         if(CTraceLevel >= 1)
            cdebug << "Atoms_R: " << atoms << endl
                   << "size before second application of R: " << y_size1
                   << ", and afterwards: " << y_size << endl;

         if(y_size != 0)
            return false;
         }
      else
         {
         if(CTraceLevel >= 1)
            cdebug << "non-HCF" << endl;

/*
         if(y_size < X_size)
            {
            if(CTraceLevel >= 1)
               cdebug << "Unconstrained atom(s) -> not unfounded-free" << endl;

            return false;
            }
*/

         bool unsat = ExecuteInnerMCAlgorithm(
             simplified_depgraph.getComponentRules(i),
             interpretation, posAssumptions,
             atoms, y_size, mode);

         if(! unsat)
            {
            if(CTraceLevel >= 1)
               cdebug << "non-hcf comp was not unfounded-free" << endl;
            return false;
            }
         }
      }

   if(atoms_in_dg < X_size)
      {
      if(CTraceLevel >= 1)
         cdebug << "Unconstrained atom(s) -> not unfounded-free" << endl;

      return false;
      }

   if(CTraceLevel >= 1)
      cdebug << "At the end. unfounded-free" << endl;
   return true;
   }
// end of IsUnfoundedFree_R_early()



//////////////////////////////////////////////////////////////////////////////
/** does not break down programs into their components, but executes
   R on demand.
   Only two kinds of "inner" algorithms for model checking are currently
   supported here.
*/
static bool IsUnfoundedFree_Monolithic(
          DEPGRAPH<GATOM,GCONSTRAINTS,GPROGRAM> *depgraph,
    const GRULES     &GroundIDB,
    const GINTERPRET &interpretation,
    const GATOMSET   &posAssumptions,
    const bool        use_R,
    const unsigned    inner_mode )
   {
   if( CTraceLevel >= 1 )
      cdebug << "Running monolithic model checking algorithm." << endl;

   GATOMSET X;
   for( size_t i=0; i < interpretation.size(); i++ )
      {
      if( interpretation.isTotallyTrue(GATOM(i)) &&
          GATOM(i).hasToBeCheckedForSupportedness() ) // exclude aggregates
          {
          X.add(GATOM(i));
          }
      }
   unsigned X_size = X.size();

   if(use_R)
      {
      FixpointR(GroundIDB, interpretation, X, X_size);

        if( X_size == 0 )
            {
            if( CTraceLevel >= 1 )
                cdebug << "R has an empty fixpoint -> stable"
                       << endl;

            return true;
            }
        else if( depgraph->isHCF() )
            {
            if( CTraceLevel >= 1 )
                cdebug << "R has non-empty fixpoint, graph is HCF -> unfounded"
                       << endl;

            return false;
            }
      }

   return ExecuteInnerMCAlgorithm( GroundIDB, interpretation,
                                   posAssumptions,
                                   X, X_size, inner_mode);
   }


//////////////////////////////////////////////////////////////////////////////
enum IsUnfoundedReturn { not_applicable, unfounded, not_unfounded };
static inline IsUnfoundedReturn IsUnfounded(
// Check whether Y is an unfounded set with respect to a (possibly partial)
// interpretation I and a single rule r.
//
// Y is unfounded, iff for every rule which contains a literal from Y
// in its head at least one of the following conditions holds:
//   
//    I: a positive body literal is in Y.
//   II: the body is false wrt. I.
//  III: a head literal which is not in Y is true in I.
//
    const GATOMSET   &Y,
    const GINTERPRET &I,
    const GRULE      &r )
    {
    // A rule is only applicable, if it contains at least one atom from
    // Y in the head.
    bool applicable=false;
    // A head atom which is not in Y is true.
    bool condIII=false;

    for( GDISJUNCTION::const_iterator i=r.getHead().begin();
         i != r.getHead().end();
	 // Alternative:
         // i != r.getHead().end() && ( ! condIII || ! applicable);
         i++ )
        {
        if( Y.contains(*i) )
            applicable=true;
        else if( I.isTotallyTrue(*i) )
            condIII=true;
        }

    if( ! applicable )
        {
        if( CTraceLevel >= 2 )
            cdebug << "  - not applicable " << r << endl;

        return not_applicable;
        }

    if( condIII )
        {
        if( CTraceLevel >= 2 )
            cdebug << "  - different head atom true " << r << endl;
        
        return unfounded;
        }

    if( r.getBody() )
        {
        for( GCONJUNCTION::const_iterator i=r.getBody()->pos_begin();
             i != r.getBody()->pos_end();
             i++ )
            {
            if( Y.contains((GATOM)*i) )
                {
                if( CTraceLevel >= 2 )
                    cdebug << "  - self-foundation " << r << endl;
                
                return unfounded;
                }
            
            // We are dealing with partial models here and have to be
            // conservative; for total models we'd use ! I.isTotallyTrue().
            if( I.isFalse((GATOM)*i) )
                {
                if( CTraceLevel >= 2 )
                    cdebug << "  - positive body false " << r << endl;
                
                return unfounded;
                }
            }

        for( GCONJUNCTION::const_iterator i=r.getBody()->neg_begin();
             i != r.getBody()->neg_end();
             i++ )
            {
            if( I.isTotallyTrue((GATOM)*i) )
                {
                if( CTraceLevel >= 2 )
                    cdebug << "  - negative body false " << r << endl;
                
                return unfounded;
                }
            }
        }
    else
        // If the body is empty, conditions I and II are trivially violated.
        ;

    // All 3 conditions are violated; we do not have an unfounded set.
    if( CTraceLevel >= 2 )
        cdebug << "  No, due to " << r << endl;

    return not_unfounded;
    }

//////////////////////////////////////////////////////////////////////////////
bool IsUnfoundedSet(
// Check whether Y is an unfounded set with respect to a (possibly partial)
// interpretation I and a set of rules P.
//
    const GATOMSET   &Y,
    const GINTERPRET &I,
    const GRULES     &P,
    vector<const GRULE*> *unfoundedRules )
    {
    if( CTraceLevel >= 2 )
        {
        cdebug << "IsUnfoundedSet? " << Y << endl
               << "           wrt. " << '{';
        I.printTruthValue(cdebug,static_cast<TruthValue> (MUST_BE_TRUE_V
                                 | TRUE_FROM_MUST_BE_TRUE_V
                                 | TRUE_FROM_UNDEFINED_V) );
        cdebug << "} , {";
        I.printTruthValue(cdebug, UNDEFINED_V);
        cdebug << '}' << endl;
        }

    // Try to find a rule which violates the definition of an unfounded set,
    // that is, a rule which violates _all_ three conditions.
    for( GRULES::const_iterator r=P.begin(); r != P.end(); r++ )
        switch( IsUnfounded(Y,I,*r) )
            {
            case not_applicable:
                break;
            case unfounded:
                if( unfoundedRules )
                    unfoundedRules->push_back(&*r);
                break;
            case not_unfounded:
                return false;
            default:
                assert( 0 );
            }

    if( CTraceLevel >= 2 )
        {
        cdebug << "  Yes, the set is unfounded."; 
        if( unfoundedRules )
            cdebug << " And we put " << unfoundedRules->size() 
                   << " of " << P.size() << " rules in the cache.";
        cdebug << endl;
        }

    return true;
    }

//////////////////////////////////////////////////////////////////////////////
bool IsUnfoundedSet(
// Check whether Y is an unfounded set with respect to a (possibly partial)
// interpretation I and a set of rules unfoundedRules.
//
    const GATOMSET   &Y,
    const GINTERPRET &I,
    const vector<const GRULE*> &unfoundedRules )
    {
    if( CTraceLevel >= 2 )
        {
        cdebug << "IsUnfoundedSet (using rule cache)? " << Y << endl
               << "           wrt. " << '{';
        I.printTruthValue(cdebug,static_cast<TruthValue> (MUST_BE_TRUE_V
                                 | TRUE_FROM_MUST_BE_TRUE_V
                                 | TRUE_FROM_UNDEFINED_V) );
        cdebug << "} , ";
        I.printTruthValue(cdebug, UNDEFINED_V);
        cdebug << '}' << endl;
        }

    // We never should encounter an empty rule cache here.
    assert( unfoundedRules.size() > 0 );

    // Try to find a rule which violates the definition of an unfounded set,
    // that is, a rule which violates _all_ three conditions.
    for( vector<const GRULE*>::const_iterator r=unfoundedRules.begin();
         r != unfoundedRules.end();
         r++ )
        if( IsUnfounded(Y,I,**r) == not_unfounded )
            return false;

    if( CTraceLevel >= 2 )
        cdebug << "  Yes, the set is unfounded." << endl;

    return true;
    }

//////////////////////////////////////////////////////////////////////////////
/** Check whether interpretation is stable wrt. to GroundIDB.
*
*   If ufset is non-NULL, the corresponding GATOMSET will be filled with a
*   maximum unfounded set in case the check failed.
*/
bool IsStable(
    const GINTERPRET &interpretation,
    const GRULES     &GroundIDB,
    const bool        knownHCF,
          DEPGRAPH<GATOM,GCONSTRAINTS,GPROGRAM> *depgraph,
    const GATOMSET   &posAssumptions,
          GATOMSET   *ufset,
          bool       *ufsetInitialized )
    {
    bool uff=false;

    global_ufset=ufset;
    global_ufsetInitialized=ufsetInitialized;
    if( global_ufsetInitialized )
        *global_ufsetInitialized=false;

    assert( ( depgraph != 0 ) || knownHCF );

#ifdef CHECK_BENCHMARK
    TIMER t;
    t.start();
#endif

    if( CTraceLevel >= 1 )
        cdebug << endl << "Checking " << interpretation << endl;

    switch(OptionModelCheckMode)
        {
        case 0:
            if( knownHCF || depgraph->isHCF() )
                {
                // ignore components
                uff = IsUnfoundedFree_Monolithic( depgraph, GroundIDB,
                                                  interpretation,
                                                  posAssumptions,
                                                  true, MCM_THROWASSERTION );
                }
            else
                {
                uff = IsUnfoundedFree_Monolithic( depgraph, GroundIDB,
                                                  interpretation,
                                                  posAssumptions,
                                                  true, MCM_CALLDAVISPUTNAM );
                }
            break;
        case 1:
            uff = IsUnfoundedFree_Split(
                depgraph, interpretation,
                posAssumptions,
                MCM_CALLDAVISPUTNAM );
            break;
        case 5:
            uff = IsUnfoundedFree_R_early ( GroundIDB, interpretation,
                                            posAssumptions,
                                            MCM_CALLDAVISPUTNAM );
            break;
        case 9:
            uff = IsUnfoundedFree_Monolithic( depgraph, GroundIDB,
                                              interpretation,
                                              posAssumptions,
                                              false, MCM_CALLDAVISPUTNAM );
            break;
        default:
            InternalError("model check");
        } // switch end

#ifdef CHECK_BENCHMARK
    t.stop();
    double mc_time = t.getAsDouble();
    
    if( CTraceLevel >= 2 )
        cdebug << "Model checking time: " << mc_time << endl;

    if(mc_time > global_worst_time)
        global_worst_time = mc_time;
#endif
    
    return uff;
    }


void ModelCheckerStats()
   {
#ifdef CHECK_BENCHMARK
   cstats << "Worst single MC time      : " << global_worst_time << "s"
                                            << endl
          << "Worst single inner MC time: " << global_worst_inner_time << "s"
                                            << endl
          << "Max. atoms for inner MC   : " << global_max_rwiplus
          << endl;
#endif
   }

// Local Variables:
// c-file-style: "dl"
// End:
