//////////////////////////////////////////////////////////////////////////////
// gatomset.h 

#ifndef GATOMSET_H
#define GATOMSET_H

//////////////////////////////////////////////////////////////////////////////
class GATOMSET
//
//  GATOMSET represents a set of grounded atoms; in fact, an array
//  of booleans is used where every atom's state (included/not included)
//  is indicated (true/false) at the array[atom.index] position.
//
    {
    // For the actual implementation we use the abstraction of bool_type.
    // (Some compilers on some systems have sizeof(bool) > 1. egcs 1.0.x,
    // for example, has sizeof(bool) == 8 on alphaev56-dec-osf4.0d.) 

    typedef char bool_type;

    static size_t SIZE;
    bool_type *array;

public:
    static void setSize(const size_t new_size)
        {
        assert( SIZE == 0  &&  GATOM::isfrozen() );

        SIZE=new_size;
        }

    static size_t getSize()
        {
        return SIZE;
        }
    
    GATOMSET()
        {
        assert( GATOM::isfrozen() );
        array = new bool_type[SIZE];
        memset( array, 0, SIZE * sizeof (bool_type) );
        }
        
    GATOMSET(const GATOMSET& atomset)
        {
        assert( GATOM::isfrozen() );
        array = new bool_type[SIZE];
        memcpy( array, atomset.array, SIZE * sizeof (bool_type) );
        }
            
    ~GATOMSET()
        {
        delete[] array;
        }

    void operator= (const GATOMSET& atomset)
        {
        assert( GATOM::isfrozen() );
        memcpy( array, atomset.array, SIZE * sizeof (bool_type) );
        }

    bool contains(const GATOM& atom) const
        {
        assert( atom.getIndex() < SIZE );

        return array[atom.getIndex()];
        }

    void add(const GATOM& atom)
        {
        assert( atom.getIndex() < SIZE );

        array[atom.getIndex()]=true;
        }

    void erase(const GATOM& atom)
        {
        assert( atom.getIndex() < SIZE );

        array[atom.getIndex()]=false;
	}

#ifndef NEVER
    size_t size() const
        {
        assert( this );

        size_t s=0;

        for(unsigned i=0; i < SIZE; i++)
            if( array[i] )
                s++;

        return s;
        }    
#endif

    bool operator[] (const unsigned index) const
        {
        assert( index < SIZE );
        return array[index];
        }

    bool operator< (const GATOMSET& set2) const
        {
	assert ( SIZE == set2.SIZE );
        for(unsigned i=0; i < SIZE; i++)
            if( array[i] < set2.array[i])
                return true;
            else if( array[i] > set2.array[i] )
                return false;

        return false;
        } 

    // "continuing" indicates whether this ATOMSET should be printed
    // as a continuation of another ATOMSET, which is not the case by
    // default.
    // returns true if anything was printed
    bool print(ostream& out, bool continuing=false) const
	{
	// Do not print a comma before the first atom, unless in
	// "continuing" mode. I.e., if first is true, there is a special
	// treatment for the first atom.
	bool first= !continuing;
	bool printed=false;

	for(unsigned i=0; i < SIZE; i++)
	    if( (*this)[i] )
		{
		if( first )
		    first=false;
		else
		    out << ", ";
		
		printed=true;
		out << GATOM(i).getTheRealOne();
		}

	return printed;
	}

    /** prints a censored version of this GATOMSET to the given output stream.
     * Uses globals PredFilter and PredPFilter that specify the wanted
     * predicates. If both are empty, no filtering will take place, and the
     * printout is verbatim.
     * @param out the destination output stream.
     * @param continuing whether the output continues preceding output.
     * @return whether something was printed.
     */
    bool printCensored(ostream& out, bool continuing=false) const
	{
	if ( !PredFilter.empty() || !PredPFilter.empty() || ProjectionsExist )
	    {
	    bool didPrint=false;
	    for (unsigned i=0; i<getSize(); i++)
		if( contains(GATOM(i)) )
                    {
                    const ATOM &atom=GATOM(i).getTheRealOne();

                    if( ! atom.getPredItem().isCensored() )
                        {
                        if (continuing)
                            out << ", ";
                        else
                            continuing=true;

                        out << atom;

                        didPrint=true;
                        }
		    }
	    return didPrint;
	    }
	else
	    return print(out,continuing);
	}
    };

//////////////////////////////////////////////////////////////////////////////
inline ostream& operator<< (ostream& out, const GATOMSET& atomset)
//
    {
    out << "{";
    atomset.print(out);
    out << "}";

    return out;
    }

#endif

// Local Variables:
// mode: c++
// c-file-style: "dl"
// End:
