///////////////////////////////////////////////////////////////////////////////
// generate.C

#include <numeric>
#define WITH_DEPGRAPH
#include "dl.h"
#include "check.h"
#include "generate.h"

////////////////////////////////////////////////////////////////////////////
// Given a GATOM atom and a GRULE r, check whether the head of r contains
// at least one other GATOM (different from atom) that belongs to the same
// component as atom.
bool sameComponentInHead(
    const DEPGRAPH<GATOM,GCONSTRAINTS,GPROGRAM> &g, 
    const GRULE &r,
    const GATOM &atom ) 
    {
    // First, take the index of the component containing 'atom'.
    const unsigned componentIndex = g.getAtomComponent(atom);

    // Then we take the head of 'r'...
    const TDISJUNCTION<GATOM> &head = r.getHead();

    // ...and check the condition.
    for (TDISJUNCTION<GATOM>::const_iterator i = head.begin(); 
         i != head.end(); i++) 
        {
        if ( *i != atom  &&  g.getAtomComponent(*i) == componentIndex )
            {
            return true;
            }
        }

    return false;
    }

///////////////////////////////////////////////////////////////////////////////

inline bool MODEL_GENERATOR::isPT(const GATOM& atom, const GINTERPRET& I) const
    {
    // atom is possibly true if it is undefined in I and occurs in the
    // head of an unsatisfied (w.r.t. I) rule, the positive body of
    // which is true in I.
    if( ! I.isUndefined(atom) )
        return false;

    for( GRULEPTRS::const_iterator r = headRules_begin(atom);
         r != headRules_end(atom);
         r++ )
        if( ! (**r).satisfied && (**r).undefPosBody == 0 )
            return true;

    return false;
    }

inline bool MODEL_GENERATOR::isPositivePT(
    const GATOM& atom,
    const GINTERPRET& I ) const
    {
    if( ! I.isUndefined(atom) )
        return false;

    if( recursiveAggregatesExist && atom.isAggregate() )
        {
        for( GRULEPTRS::const_iterator r = posBodyRules_begin(atom);
             r != posBodyRules_end(atom);
             r++ )
            {
            assert( posGroundDG );
            if( sameComponentInHead( *posGroundDG,(**r),atom) )
                return true;
            }
        }
    else
        {
        for( GRULEPTRS::const_iterator r = headRules_begin(atom);
             r != headRules_end(atom);
             r++ )
            if( ! (**r).satisfied
                && ((**r).undefNegBody + (**r).undefPosBody == 0) )
                return true;
        }

    return false;
    }

inline bool MODEL_GENERATOR::isNegativePT(
    const GATOM& atom,
    const GINTERPRET& I ) const
    {
    if( ! I.isUndefined(atom) )
        return false;

    for( GRULEPTRS::const_iterator r = negBodyRules_begin(atom);
         r != negBodyRules_end(atom);
         r++ )
        if( ! (**r).satisfied && ((**r).undefPosBody == 0) )
            return true;

    return false;
    }

inline void MODEL_GENERATOR::HeuristicPT::getFirstRealPT()
// Get the first possibly true literal. Since the lookahead might have
// entailed a propagation, some of the previously found possibly true
// literals may no longer be possibly true.
    {
    while( ! PTCache.empty() )
        {
        if( OptionUseOldPTs )
            {
            if( PTCache.front().literal.isNegative() )
                {
                if( MG.isNegativePT(PTCache.front().literal,I) )
                    break;
                }
            else
                {
                if( MG.isPositivePT(PTCache.front().literal, I) )
                    break;
                }
            }
        else
            {
            if( MG.isPT(PTCache.front().literal, I) )
                break;
            }
        
        if( HTraceLevel >= 2 )
            cdebug << indent(MG.level) << PTCache.front() 
                   << " is no longer a PT." << endl;
        
        // Here the PT in the list is no longer PT.
        PTCache.pop_front();
        }
    }

inline bool MODEL_GENERATOR::HeuristicPT::init()
    {
    do
        {
        // Reset the flags indicating whether the heuristic values
        // of some PT have already been considered.
        MG.initialiseAlreadyConsideredAsPT();
        
        MG.clearBinaryClauseOccurrences();
        
        // Perform lookahead and immediately return if an
        // inconsistency is detected.
        if( ! createPTs() )
            {
            if( HTraceLevel >= 2 )
                cdebug << "Propagating compliment of inconsistent PTs "
                    "creates inconsistency as well. "
                    "Aborting entire subtree." << endl;
            return false;
            }
        
        if( HTraceLevel >= 2 )
            {
            copy(PTCache.begin(),
                 PTCache.end(),
                 ostream_iterator<HeuristicItem>(cdebug, "  ->  "));
            cdebug << "[]" << endl;
            }
        
        // Set the flag if no possibly true literals have been
        // found at all.
        if( PTCache.empty() )
            noMorePTs = true;
        else
            getFirstRealPT();
        }
    // If there were possibly true literals, but all of them are
    // no longer possibly true, we must perform another lookahead,
    // since some literals may have become possibly true after the
    // final propagation. If there was no possibly true literal at
    // all, we are finished since no propagation may have taken
    // place (only the "negation" of possibly true literals which
    // directly lead to an inconsistency are propagated).
    while( ! noMorePTs && PTCache.empty() );
    
    return true;
    }

inline bool MODEL_GENERATOR::HeuristicPT::next()
    {
    assert( ! atEnd() );
    assert( ! PTCache.empty() );

    PTCache.pop_front();
    
    getFirstRealPT();
    
    if( PTCache.empty() )
        return init();
    
    return true;
    }

inline bool MODEL_GENERATOR::HeuristicPT::createPTs()
    {
    // Perform lookahead and order the possibly true literals...
    
    assert( PTCache.empty() );
    
    currentCacheSize = 0;
    
    unsigned numberOfConsPTs, numberOfInconsPTs;
    
    bool firstPT = true;
    
    do
        {
        numberOfConsPTs = 0;
        numberOfInconsPTs = 0;
        
        unsigned long binaryClauseOccurrenceSum = 0;
        unsigned long binaryClauseOccurrenceCount = 0;

        // For every atom in the Herbrand Base:
        for( unsigned i = 0; i < GATOMSET::getSize(); i++ )
            {
            GATOM a(i);
            
            if( OptionUseOldPTs )
                {
                // check whether its pos. or neg. literal is possibly true
                if( ! MG.isAlreadyConsideredAsPT(a,false)
                    && MG.isPositivePT(a,I) )
                    {
                    MG.initialiseBinaryClauseOccurrences(a,false);
                    binaryClauseOccurrenceCount++;
                    binaryClauseOccurrenceSum +=
                        MG.getBinaryClauseOccurrences(a,false)
                        + MG.getBinaryClauseOccurrences(a,true);
                    }
                
                if( ! MG.isAlreadyConsideredAsPT(a,true)
                    && MG.isNegativePT(a,I) )
                    {
                    MG.initialiseBinaryClauseOccurrences(a,true);
                    binaryClauseOccurrenceCount++;
                    binaryClauseOccurrenceSum +=
                        MG.getBinaryClauseOccurrences(a,true)
                        + MG.getBinaryClauseOccurrences(a,false);
                    }
                }
            else
                {
                if( ! MG.isAlreadyConsideredAsPT(a,false)
                    && MG.isPT(a,I) )
                    {
                    MG.initialiseBinaryClauseOccurrences(a,false);
                    binaryClauseOccurrenceCount++;
                    binaryClauseOccurrenceSum +=
                        MG.getBinaryClauseOccurrences(a,false)
                        + MG.getBinaryClauseOccurrences(a,true);
                    }
                }
            }
        
        unsigned binaryClauseOccurrenceMean = 0;
        
        if( binaryClauseOccurrenceCount != 0 && MG.level > 1 )
            {
            if( binaryClauseOccurrenceSum 
                < binaryClauseOccurrenceCount
                && binaryClauseOccurrenceSum > 0 )
                binaryClauseOccurrenceMean = 1;
            else
                binaryClauseOccurrenceMean =
                    binaryClauseOccurrenceSum
                    / binaryClauseOccurrenceCount;
            }
        
        // For every atom in the Herbrand Base:
        for( unsigned i = 0; i < GATOMSET::getSize(); i++ )
            {
            GATOM a(i);
            
            if( OptionUseOldPTs )
                {
                // check whether its pos. or neg. literal is possibly true
                if( ! MG.isAlreadyConsideredAsPT(a,false)
                    && ( MG.getBinaryClauseOccurrences(a,false)
                         + MG.getBinaryClauseOccurrences(a,true)
                         >= binaryClauseOccurrenceMean )
                    && MG.isPositivePT(a,I) )
                    {
                    if( firstPT )
                        {
                        MG.statChoicePoints++;
                        firstPT = false;
                        }
                    
                    if( ! HandlePT(a,false,I,
                                   numberOfConsPTs,numberOfInconsPTs) )
                        return false;
                    }
                
                if( ! MG.isAlreadyConsideredAsPT(a,true)
                    && ( MG.getBinaryClauseOccurrences(a,true)
                         + MG.getBinaryClauseOccurrences(a,false)
                         >= binaryClauseOccurrenceMean )
                    && MG.isNegativePT(a,I) )
                    {
                    if( firstPT )
                        {
                        MG.statChoicePoints++;
                        firstPT = false;
                        }
                    
                    if( ! HandlePT(a,true,I,
                                   numberOfConsPTs,numberOfInconsPTs) )
                        return false;
                    }
                }
            else
                {
                if( ! MG.isAlreadyConsideredAsPT(a,false)
                    && ( MG.getBinaryClauseOccurrences(a,false)
                         + MG.getBinaryClauseOccurrences(a,true)
                         >= binaryClauseOccurrenceMean )
                    && MG.isPT(a,I) )
                    {
                    if( firstPT )
                        {
                        MG.statChoicePoints++;
                        firstPT = false;
                        }
                    
                    if( ! HandlePT(a,false,I,
                                   numberOfConsPTs,numberOfInconsPTs) )
                        return false;
                    }
                }
            
            }
        }
    // Note: If no PT has been found, but there were PTs
    // immediately leading to inconsistency, the complement of
    // these "inconsistent PTs" has been assumed and
    // propagated. This could have turned some previously non-PT
    // atoms into PT atoms, so we have to reexamine the atoms.
    while( numberOfConsPTs == 0 && numberOfInconsPTs > 0 );
    
    if( HTraceLevel >= 1 )
        cdebug << indent(MG.level) << numberOfConsPTs 
               << " consistent PTs and " << numberOfInconsPTs 
               << " inconsistent PTs." << endl;
    
    return true;
    }

//////////////////////////////////////////////////////////////////////////////
/// prints the undefined-counters of all ground rules
/// @param the stream to print on
void MODEL_GENERATOR::PrintGruleCounters(
    ostream&      out )
    {
    // simply traverse all rules ...
    for( GRULES::const_iterator r=rules.begin(); 
	 r != rules.end(); 
	 r++ )
	{
	// ... and pretty-print rules and counters
	out << *r << " has "
	    << r->undefHead << "/"
	    << r->undefPosBody << "/"
	    << r->undefNegBody
            << " undefined literals in H/B+/B-"
            << endl;
	}
    }

//////////////////////////////////////////////////////////////////////////////
/// prints the undefined-counters of all ground constraints
/// @param the stream to print on
void MODEL_GENERATOR::PrintGconstraintCounters(
    ostream&            out )
    {
    // simply traverse all constraints ...
    for( GCONSTRAINTS::const_iterator c=constraints.begin(); 
	 c != constraints.end(); 
	 c++ )
	{
	// ... and pretty-print constraints and counters
	out << *c << " has " 
	    << c->undefPosBody << "/"
	    << c->undefNegBody 
            << " undefined literals in B+/B-"
	    << endl;
	}
    }

//////////////////////////////////////////////////////////////////////////////
/// prints all ground atoms, and the rules/constraints in which they occur,
/// respectively
/// @param the stream to print on
void MODEL_GENERATOR::PrintGatomRulesConstraints(
    ostream&            out )
    {
    // print information
    out << "Ground atoms and where they occur:"
	<< endl;
    
    // Since there is no global datastructure in which ground atoms are
    // stored, we have to build one such structure locally.
    set<GATOM> allGatoms;
    // traverse all ground rules...
    for( GRULES::const_iterator r=rules.begin(); 
	 r != rules.end(); 
	 r++ )
	{
	assert( r->hasHead() );
	// collect all ground atoms from the head ...
	for( GDISJUNCTION::const_iterator a=r->getHead().begin(); 
	     a != r->getHead().end(); 
	     a++ )
	    allGatoms.insert(*a);
	// ... and from the body, if it exists
	if( const GCONJUNCTION *body = (*r).getBody() )
	    for( GCONJUNCTION::const_iterator a=body->begin(); 
		 a != body->end(); a++ )
		allGatoms.insert(*a);
	}
    // ...and all ground constraints
    for( GCONSTRAINTS::const_iterator c=constraints.begin();
	 c != constraints.end();
	 c++ )
	{
	for( GCONSTRAINT::const_iterator l=(*c).begin();
	     l != (*c).end();
	     l++ )
	    {
	    allGatoms.insert(*l);
	    }
	}

    // All ground atoms are stored in allGatoms. Now traverse it and print
    // out the rule information:
    for( set<GATOM>::const_iterator a=allGatoms.begin(); 
	 a != allGatoms.end(); a++ )
	{
	out << *a 
            << " has "
            << getPotentialSupport(*a)
            << " rule(s) potentially supporting it and occurs" << endl;
        
        if( headRules_begin(*a) != headRules_end(*a) )
            {
            out << "\tin the head of " << endl;
            }
	for( GRULEPTRS::const_iterator r = headRules_begin(*a); 
	     r != headRules_end(*a); r++ )
	    {
	    out << "\t   " << **r << endl;
	    }
        
        if( posBodyRules_begin(*a) != posBodyRules_end(*a)
         || posConstraints_begin(*a) != posConstraints_end(*a) )
            {
            out << "\tin the positive body of " << endl;
            }
        for( GRULEPTRS::const_iterator r = posBodyRules_begin(*a);
             r != posBodyRules_end(*a); r++ )
            {
            out << "\t   " << **r << endl;
            }
 	for( GCONSTRAINTPTRS::const_iterator c = posConstraints_begin(*a);
	     c != posConstraints_end(*a); c++ )
	    {
	    out << "\t   " << **c << endl;
	    }
        
        if( negBodyRules_begin(*a) != negBodyRules_end(*a)
         || negConstraints_begin(*a) != negConstraints_end(*a) )
            {
            out << "\tin the negative body of " << endl;
            }
	for( GRULEPTRS::const_iterator r = negBodyRules_begin(*a); 
	     r != negBodyRules_end(*a); r++ )
	    {
            out << "\t   " << **r << endl;
	    }
	for( GCONSTRAINTPTRS::const_iterator c = negConstraints_begin(*a);
	     c != negConstraints_end(*a); c++ )
	    {
            out << "\t   " << **c << endl;
	    }
	}
    out << endl;
    }

//////////////////////////////////////////////////////////////////////////////
// Initialise all counters in the GRULES rules w.r.t. a given
// GINTERPRET and register these rules with the head-, positive-body-,
// and negative-body-occurrences data structures.
// A rule can yield an inconsistency if all head atoms are false and
// the body is true (or mbt). In this case false is returned.
bool MODEL_GENERATOR::initialiseRules(
    const GINTERPRET &I )
    {
    // For each rule...
    for( GRULES::iterator ri = rules.begin();
         ri != rules.end();
         ri++ )
	{
        bool inconsistent = true;

        // ... reset rule counters
	(*ri).undefHead = 0;
	(*ri).undefPosBody = 0;
	(*ri).undefNegBody = 0;
        (*ri).satisfied = GRULE::UNSATISFIED;
        (*ri).potentiallySupportedAtom = 0;
        (*ri).mbtHead = 0;
        (*ri).mbtPosBody = 0;

        // Rules must have heads.
	assert( ri->hasHead() );

        // ... add rule to atom head rules, adjust undefined and mbt
        // rule counters and handle satisfied rules
	for( GDISJUNCTION::const_iterator a = ri->getHead().begin();
	     a != ri->getHead().end(); a++ )
	    {
	    addHeadRule(*a, ri); 

	    if( I.isUndefined(*a) )
                {
		(*ri).undefHead++;
                inconsistent = false;
                }
            else if( I.isMustBeTrue(*a) )
                {
                (*ri).mbtHead++;
		(*ri).undefHead++;
                inconsistent = false;
                }
            else if( I.isTrue(*a) )
                {
                inconsistent = false;
                // The rule is satisfied.
                if( (*ri).satisfied )
                    {
                    // If it was already satisfied before, check
                    // whether the rule potentially supports some head
                    // atom. If not, it is already known that no atom
                    // can be supported by the rule (only satisfaction
                    // by the head has been considered so far).
                    if( (*ri).potentiallySupportedAtom != 0 )
                        {
                        // If there was a potentially supported atom,
                        // it can no longer be supported, since no
                        // duplicate atoms should occur in the head,
                        // and therefore two different true atoms
                        // occur in the head.

                        // Make sure that the current atom is not the
                        // supported one (if the assertion triggers,
                        // there are duplicate atoms in the head).
                        assert( ! (*ri).potentiallySupports(*a) );

                        (*ri).potentiallySupportedAtom = 0;
                        }
                    }
                else
                    {
                    // Unsatisfied rules cannot potentially support a
                    // particular atom.
                    assert( (*ri).potentiallySupportedAtom == 0 );
                    // The rule becomes satisfied and potentially
                    // supports exactly the current atom.
                    (*ri).potentiallySupportedAtom =
                        a - ri->getHead().begin() + 1;
		    (*ri).satisfied |= GRULE::SATISFIED_BY_HEAD;
                    }
                }
	    }


        // ... add rule to the respective atom body rules and handle
        // satisfied rules.
	if( (*ri).hasBody() )
	    {
	    const GCONJUNCTION *body = (*ri).getBody();
	    for( GCONJUNCTION::const_iterator l = body->pos_begin();
		 l != body->pos_end(); l++ )
		{
		addPosBodyRule(*l, ri);

                if( (*l).isAggregate() )
                    updateAggregatesLinearStructures(*l);
                if( I.isUndefined(*l) )
                    {
                    (*ri).undefPosBody++;
                    inconsistent = false;
                    }
                else if( I.isMustBeTrue(*l) )
                    {
                    (*ri).mbtPosBody++;
                    (*ri).undefPosBody++;
                    }
                else if( I.isFalse(*l) )
                    {
                    inconsistent = false;
                    // The rule is satisfied and cannot potentially
                    // support a particular atom.
		    (*ri).satisfied |= GRULE::SATISFIED_BY_BODY;
                    (*ri).potentiallySupportedAtom = 0;
                    }
		}
	    for( GCONJUNCTION::const_iterator l = body->neg_begin();
		 l != body->neg_end(); l++ )
		{
		addNegBodyRule(*l,ri);
                if( (*l).isAggregate() )
                    updateAggregatesLinearStructures(*l);
		if( I.isUndefined(*l) )
                    {
                    inconsistent = false;
		    (*ri).undefNegBody++;
                    }
                else if( I.isFalse(*l) )
                    {
                    inconsistent = false;
                    // The rule is satisfied and cannot potentially
                    // support a particular atom.
                    (*ri).satisfied |= GRULE::SATISFIED_BY_BODY;
                    (*ri).potentiallySupportedAtom = 0;
                    }
		}
	    }

        // Finally, adjust the atoms' potentially-supported counters.
        // FIXME: This could be done in a smarter way by checking
        // satisfied and potentiallySupportedAtom directly.
        for( GDISJUNCTION::const_iterator a = ri->getHead().begin();
             a != ri->getHead().end(); a++ )
            if( (*ri).potentiallySupports(*a) )
                incPotentialSupport(*a);

        // Detect inconsistency.
        if( inconsistent )
            return false;
	}

    return true;
    }

//////////////////////////////////////////////////////////////////////////////
// Initialise all counters in the GCONSTRAINTS constraints and register these
// constraints with the positive- and negative-occurrences data structures.
// A constraint can yield an inconsistency if literals are true (or
// mbt). In this case false is returned.
bool MODEL_GENERATOR::initialiseConstraints(
    const GINTERPRET   &I )
    {
    for( GCONSTRAINTS::iterator ci = constraints.begin();
         ci != constraints.end();
         ci++ )
	{
        bool inconsistent = true;

	(*ci).undefPosBody = 0;
	(*ci).undefNegBody = 0;
        (*ci).satisfied = false;

	for( GCONJUNCTION::const_iterator l = (*ci).pos_begin();
             l != (*ci).pos_end(); l++ )
	    {
	    addPosConstraint(*l,ci);
            if( (*l).isAggregate() )
                updateAggregatesLinearStructures(*l);
	    if( I.isUndefined(*l) )
                {
                inconsistent = false;
                (*ci).undefPosBody++;
                }
            else if( I.isFalse(*l) )
                {
                inconsistent = false;
                (*ci).satisfied = true;
                }
	    }
	for( GCONJUNCTION::const_iterator l = (*ci).neg_begin();
	     l != (*ci).neg_end(); l++ )
	    {
	    addNegConstraint(*l,ci);
            if( (*l).isAggregate() )
                updateAggregatesLinearStructures(*l);
	    if( I.isUndefined(*l) )
                {
                inconsistent = false;
		(*ci).undefNegBody++;
                }
            else if( I.isFalse(*l) )
                {
                inconsistent = false;
                (*ci).satisfied = true;
                }
	    }

        // Detect inconsistency.
        if( inconsistent )
            return false;
	}

    return true;
    }

//////////////////////////////////////////////////////////////////////////////
// Initialise all counters in the GWEAKCONSTRAINTS constraints and register
// these weak constraints with the positive- and negative-occurrences data
// structures.
// A weak constraint can yield a violation if literals are true (or mbt).
void MODEL_GENERATOR::initialiseWeakConstraints(
    GINTERPRET   &I )
    {
    for( GWEAKCONSTRAINTS::iterator ci = wconstraints.begin();
                                    ci != wconstraints.end();
                                    ci++ )
        {
        bool violated = true;

        (*ci).undefPosBody = 0;
        (*ci).undefNegBody = 0;
        (*ci).satisfied = false;

        for( GCONJUNCTION::const_iterator l = (*ci).pos_begin();
                                          l != (*ci).pos_end(); 
                                          l++ )
            {
            addPosWeakConstraint(*l,ci);
            if( (*l).isAggregate() )
                updateAggregatesLinearStructures(*l);
            if( I.isUndefined(*l) )
                {
                violated = false;
                (*ci).undefPosBody++;
                }
            else if( I.isFalse(*l) )
                {
                violated = false;
                (*ci).satisfied = true;
                }
            }
        for( GCONJUNCTION::const_iterator l = (*ci).neg_begin();
                                          l != (*ci).neg_end(); 
                                          l++ )
            {
            addNegWeakConstraint(*l,ci);
            if( (*l).isAggregate() )
                updateAggregatesLinearStructures(*l);
            if( I.isUndefined(*l) )
                {
                violated = false;
                (*ci).undefNegBody++;
                }
            else if( I.isFalse(*l) )
                {
                violated = false;
                (*ci).satisfied = true;
                }
            }
        // Detect violation and increase the cost of the interpretation.
        if(  violated )
            {
            // Currently we haven't stored a Best Model yet, so we can
            // increase the cost directly.
            I.increaseCost((*ci).getWeights());
            if(TraceLevel >=3)
                {
                cdebug << indent(level)
                       << "WeakConstraint " << (*ci) << "is violated" 
                       << endl << indent(level) 
                       << "Increased cost of the interpretation "
                          "([Weight:Level]): ";
                I.printCost(cdebug);
                }
            }         
        }
    }

//////////////////////////////////////////////////////////////////////////////
// Initialise counters and rule/constraint references for our linear-time
// operators (and more).
// If an inconsistency is detected, false is returned.
bool MODEL_GENERATOR::initialiseLinearDataStructures(
    GINTERPRET   &I )
    {
    const size_t N=GATOMSET::getSize();

    // Reserve the amount of memory we need in advance, so that we avoid
    // successive growth in the initialization loop below.
    // (This saves needless copying and avoids allocation of unused memory;
    // STL by default allocates memory in chunks of 2^n bytes.)
    HeadOccurrences.reserve(N);
    PosBodyOccurrences.reserve(N);
    NegBodyOccurrences.reserve(N);
    PosConstraintOccurrences.reserve(N);
    NegConstraintOccurrences.reserve(N);
    PosWConstraintOccurrences.reserve(N);
    NegWConstraintOccurrences.reserve(N);
    occursInAggregateFunction.reserve(N);
    occursInGatom.reserve(aggregateFunctionsCount);

    if( HeadOccurrences.size() < N )
        HeadOccurrences.insert(HeadOccurrences.end(),
                               N - HeadOccurrences.size(),GRULEPTRS());

    if( PosBodyOccurrences.size() < N )
        PosBodyOccurrences.insert(PosBodyOccurrences.end(),
                                  N - PosBodyOccurrences.size(),GRULEPTRS());

    if( NegBodyOccurrences.size() < N )
        NegBodyOccurrences.insert(NegBodyOccurrences.end(),
                                  N - NegBodyOccurrences.size(),GRULEPTRS());

    if( PosConstraintOccurrences.size() < N )
        PosConstraintOccurrences.insert(PosConstraintOccurrences.end(),
                                        N - PosConstraintOccurrences.size(),
                                        GCONSTRAINTPTRS());

    if( NegConstraintOccurrences.size() < N )
        NegConstraintOccurrences.insert(NegConstraintOccurrences.begin(),
                                        N - NegConstraintOccurrences.size(),
                                        GCONSTRAINTPTRS());

    if( PosWConstraintOccurrences.size() < N )
        PosWConstraintOccurrences.insert(PosWConstraintOccurrences.end(),
                                        N - PosWConstraintOccurrences.size(),
                                        GWEAKCONSTRAINTPTRS());

    if( NegWConstraintOccurrences.size() < N )
        NegWConstraintOccurrences.insert(NegWConstraintOccurrences.begin(),
                                        N - NegWConstraintOccurrences.size(),
                                        GWEAKCONSTRAINTPTRS());

    if( potentiallySupportingRules.size() < N )
        potentiallySupportingRules.insert(potentiallySupportingRules.end(),
                               N - potentiallySupportingRules.size(),0);

    if( alreadyConsideredAsPosPT.size() < N )
        alreadyConsideredAsPosPT.insert(alreadyConsideredAsPosPT.end(),
                               N - alreadyConsideredAsPosPT.size(),0);

    if( alreadyConsideredAsNegPT.size() < N )
        alreadyConsideredAsNegPT.insert(alreadyConsideredAsNegPT.end(),
                               N - alreadyConsideredAsNegPT.size(),0);

    if( binaryClauseOccurrencesPos.size() < N )
        binaryClauseOccurrencesPos.insert(binaryClauseOccurrencesPos.end(),
                               N - binaryClauseOccurrencesPos.size(),0);

    if( binaryClauseOccurrencesNeg.size() < N )
        binaryClauseOccurrencesNeg.insert(binaryClauseOccurrencesNeg.end(),
                               N - binaryClauseOccurrencesNeg.size(),0);

    if( occursInAggregateFunction.size() < N )
        occursInAggregateFunction.insert(occursInAggregateFunction.end(),
                               N - occursInAggregateFunction.size(),
                               vector< pair<unsigned,TERMS> >());

    if( occursInGatom.size() < aggregateFunctionsCount )
        occursInGatom.insert(occursInGatom.end(),
                               aggregateFunctionsCount - occursInGatom.size(),
                               vector<GATOM>());

    if( initialiseRules(I) && initialiseConstraints(I) )
        {
        initialiseWeakConstraints(I);
        return true;
        }
    else
        return false;
    }

//////////////////////////////////////////////////////////////////////////////
// Create a datastructure holding the current potentiallySupportingRules
// counters of GATOMs.
//
// @return a vector indexed by GATOMs, containing their current
//         potentiallySupportingRules counter values.

vector<unsigned> *
    MODEL_GENERATOR::createPotentiallySupportingRulesCounters()
    {
    vector<unsigned>* oldPotentiallySupportingRulesP = 
        new vector<unsigned>;

    oldPotentiallySupportingRulesP->reserve(GATOMSET::getSize());

    if( oldPotentiallySupportingRulesP->size() < GATOMSET::getSize() )
        oldPotentiallySupportingRulesP->insert(
            oldPotentiallySupportingRulesP->end(),
            GATOMSET::getSize() - oldPotentiallySupportingRulesP->size(),
            0);

    // for every atom in the HB
    for( unsigned i = 0; i < GATOMSET::getSize(); i++ )
        (*oldPotentiallySupportingRulesP)[i] 
            = getPotentialSupport(GATOM(i));

    return oldPotentiallySupportingRulesP;
    }

//////////////////////////////////////////////////////////////////////////////
// Compares the potentiallySupportingRules counters of GATOMs with
// previously cached copies.
// 
// @return true if changes occured, false if no changes occured

bool MODEL_GENERATOR::checkPotentiallySupportingRulesCounters(
    vector<unsigned> * oldPotentiallySupportingRulesP )
    {
    bool change=false;

    // for every atom in the HB
    for( unsigned i = 0; i < GATOMSET::getSize(); i++ )
        {
        if( (*oldPotentiallySupportingRulesP)[i]
            != getPotentialSupport(GATOM(i)) )
            {
            if( TraceLevel >= 3 )
                cdebug << GATOM(i)
                       << "'s potential support counter " 
                       << (*oldPotentiallySupportingRulesP)[i] << " -> " 
                       << getPotentialSupport(GATOM(i)) 
                       << endl;
            change = true;
            }
        }

    return change;
    }


//////////////////////////////////////////////////////////////////////////////
/// recomputes the undefined counters of a ground rule, optionally checking
/// whether changes occured
///
/// @param r, the GRULE whose counters are recomputed.
/// @param I, a GINTERPRET used to determine the undefinedness of atoms.
/// @param check, specifies whether a check on changes should be performed
///        defaults to false
///
/// @return if check was true, returns true if a change occured; if check was
///         false, true is returned in any case

bool MODEL_GENERATOR::RecomputeCounters(
          GRULE      &r,
    const GINTERPRET &I,
          bool        check ) 
    {
    bool change = false;

    // If we have to check for changes, record current counter values
    // in local variables.
    unsigned uHead = 0, uPosBody = 0, uNegBody = 0,
             uPotentiallySupportedAtom = 0;
    unsigned short uSatisfied = GRULE::UNSATISFIED;

    // If a body exists...
    if (r.hasBody())
        {
        const GCONJUNCTION *body=r.getBody();

        // ...check the positive body for undefined or must-be-true
        // (w.r.t. I) atoms...
        for (GCONJUNCTION::const_iterator l=body->pos_begin();
             l!=body->pos_end(); l++)
            {
            if( I.isUndefined(*l) || I.isMustBeTrue(*l) )
                uPosBody++;
            else if (I.isFalse(*l))
                // one false literal in the body satisfies the rule
                uSatisfied = GRULE::SATISFIED_BY_BODY;
            }
        // ... and check the negative body for undefined (w.r.t. I)
        // literals or literals whose atoms are must-be-true
        // (w.r.t. I).
        for (GCONJUNCTION::const_iterator l=body->neg_begin();
             l!=body->neg_end(); l++)
            {
            if( I.isUndefined(*l) )
                    uNegBody++;
            else if( I.isFalse(*l) || I.isMustBeTrue(*l) )
                // One false literal in the body or a negative body
                // literal whose atom is must-be-true both satisfy the
                // rule.
                uSatisfied = GRULE::SATISFIED_BY_BODY;
            }
        }


    assert( r.hasHead() );

    // Unless the body is false (iff uSatisfied is true at this
    // point), the rule may potentially support an atom in the head.
    if( ! uSatisfied )
        {
        for (GDISJUNCTION::const_iterator a=r.getHead().begin();
             a != r.getHead().end(); a++)
            {
            if( I.isTrue(*a) )
                {
                // If the rule does not potentially support an atom
                // yet, it potentially supports the current one.
                if( uPotentiallySupportedAtom == 0 )
                    uPotentiallySupportedAtom = a - r.getHead().begin() + 1;
                else
                    {
                    // If the rule already potentially supports an
                    // atom which is different from the current atom,
                    // the rule cannot support any atom at all.
                    if( *(r.getHead().begin() 
                          + uPotentiallySupportedAtom - 1) != *a )
                        {
                        uPotentiallySupportedAtom = 0;
                        break;
                        }
                    }
                }
            }
        }

    // If the head is not true yet, and if the body did not contain
    // any false literals, all head atoms are potentially supported
    // in this rule.

    // If the head is true, the rule is satisfied.  Satisfaction has
    // to be determined before increasing the GATOMs' potential support
    // counters.

    if (I.isTrue(r.getHead()))
        uSatisfied = GRULE::SATISFIED_BY_HEAD;

    // Check the rule's head for undefined (w.r.t. I) atoms.
    for (GDISJUNCTION::const_iterator a=r.getHead().begin();
         a != r.getHead().end(); a++)
        if( I.isUndefined(*a) || I.isMustBeTrue(*a) )
            uHead++;

    // Change the values of the potential support counters. This is
    // only done if check is false, otherwise the change flag is set
    // to true.

    assert( ! ( ! uSatisfied && uPotentiallySupportedAtom != 0 ) );
    assert( ! ( ! r.satisfied && r.potentiallySupportedAtom != 0 ) );

    if( ! r.satisfied && uSatisfied )
        {
        if( r.potentiallySupportedAtom == 0 && uPotentiallySupportedAtom == 0 )
            {
            // Previously, all head atoms were potentially supported
            // by the rule, now none of them is potentially supported.
            // Decrease the potential support counters of all head atoms.
            for( GDISJUNCTION::const_iterator a = r.getHead().begin();
                 a != r.getHead().end(); a++ )
                if( check )
                    change = true;
                else
                    decPotentialSupport(*a);
            }
        else if( r.potentiallySupportedAtom == 0
                 && uPotentiallySupportedAtom != 0 )
            {
            // Previously, all head atoms were potentially supported
            // by the rule, now only the supported atom is. Decrease
            // the potential support counters of all head atoms except
            // for the potentially supported atom.
            GDISJUNCTION::const_iterator suppAtom = 
                r.getHead().begin() + uPotentiallySupportedAtom - 1;
            for( GDISJUNCTION::const_iterator a = r.getHead().begin();
                 a != r.getHead().end(); a++ )
                if( *a != *suppAtom )
                    {
                    if( check )
                        change = true;
                    else
                        decPotentialSupport(*a);
                    }
            }
        else
            // All other cases should have been caught by one of the
            // two assertions above.
            assert(0);
        }
    else if( r.satisfied && ! uSatisfied )
        {
        if( r.potentiallySupportedAtom == 0 && uPotentiallySupportedAtom == 0 )
            {
            // Previously, no head atom was potentially supported by
            // the rule, now all of them are. Increase the potential
            // support counters of all head atoms.
            for( GDISJUNCTION::const_iterator a = r.getHead().begin();
                 a != r.getHead().end(); a++ )
                    if( check )
                        change = true;
                    else
                        incPotentialSupport(*a);
            }
        else if( r.potentiallySupportedAtom != 0
                 && uPotentiallySupportedAtom == 0 )
            {
            // Previously, only the supported head atom was
            // potentially supported by the rule, now all of them
            // are. Increase the potential support counters of all
            // head atoms except for the previously supported one.
            GDISJUNCTION::const_iterator suppAtom = 
                r.getHead().begin() + r.potentiallySupportedAtom - 1;
            for( GDISJUNCTION::const_iterator a = r.getHead().begin();
                 a != r.getHead().end(); a++ )
                if( *a != *suppAtom )
                    {
                    if( check )
                        change = true;
                    else
                        incPotentialSupport(*a);
                    }
            }
        else
            // All other cases should have been caught by one of the
            // two assertions above.
            assert(0);
        }
    else if( r.satisfied && uSatisfied )
        {
        if( r.potentiallySupportedAtom == 0 
            && uPotentiallySupportedAtom == 0 )
            {
            // Nothing needs to be done.
            }
        else if( r.potentiallySupportedAtom == 0 
                 && uPotentiallySupportedAtom != 0 )
            {
            // Previously, no head atom was potentially supported by
            // the rule, now exactly the supported atom is. Increase
            // the potential support counter of the potentially
            // supported atom (once for each occurrence).
            GDISJUNCTION::const_iterator suppAtom = 
                r.getHead().begin() + uPotentiallySupportedAtom - 1;
            for( GDISJUNCTION::const_iterator a = r.getHead().begin();
                 a != r.getHead().end(); a++ )
                if( *a == *suppAtom )
                    {
                    if( check )
                        change = true;
                    else
                        incPotentialSupport(*a);
                    }
            }
        else if( r.potentiallySupportedAtom != 0
                 && uPotentiallySupportedAtom == 0 )
            {
            // Previously, only the potentially supported head atom
            // was potentially supported by the rule, now no head atom
            // is potentially supported. Decrease the potential
            // support counter of the previously potentially supported
            // atom (once for each occurrence).
            GDISJUNCTION::const_iterator suppAtom = 
                r.getHead().begin() + r.potentiallySupportedAtom - 1;
            for( GDISJUNCTION::const_iterator a = r.getHead().begin();
                 a != r.getHead().end(); a++ )
                if( *a == *suppAtom )
                    {
                    if( check )
                        change = true;
                    else
                        decPotentialSupport(*a);
                    }
            }
        else if( r.potentiallySupportedAtom != 0 
                 && uPotentiallySupportedAtom != 0 )
            {
            if( r.potentiallySupportedAtom != uPotentiallySupportedAtom )
                {
                // The potentially supported atom changed. Decrease
                // the potential support counter of the previously
                // potentially supported atom and increase the
                // potential support counter of the newly potentially
                // supported atom (once for each occurrence).
                GDISJUNCTION::const_iterator prevSuppAtom = 
                    r.getHead().begin() + r.potentiallySupportedAtom - 1;
                GDISJUNCTION::const_iterator currSuppAtom = 
                    r.getHead().begin() + uPotentiallySupportedAtom - 1;
                for( GDISJUNCTION::const_iterator a = r.getHead().begin();
                     a != r.getHead().end(); a++ )
                    if( *a == *prevSuppAtom )
                        {
                        if( check )
                            change = true;
                        else
                            decPotentialSupport(*a);
                        }
                    else if( *a == *currSuppAtom )
                        {
                        if( check )
                            change = true;
                        else
                            incPotentialSupport(*a);
                        }
                }
            }
        else
            // All conditions have been checked.
            assert(0);
        }

    if (check)
        {
        // Check whether some counters have changed.
        // If a rule is satisfied, the counter values are undefined.
        // So if the rule was satisfied before and is also currently
        // satisfied, differing counter values are not considered as a
        // change.
        // This does not affect the potentiallySupportedAtom values.
        if ( ( ! ( uSatisfied && r.satisfied )
               && ( uSatisfied != r.satisfied 
                    || uHead != r.undefHead 
                    || uPosBody != r.undefPosBody
                    || uNegBody != r.undefNegBody ) )
             || uPotentiallySupportedAtom != r.potentiallySupportedAtom )
            {
            // If so, set the change flag
            change=true;
            // and output diagnostics if requested
            if (TraceLevel >= 3)
                {
                cdebug << "Counter change: " << r << "   "
                       << ( r.satisfied ? "sat/" : "psupport/" )
                       << r.undefHead << "/"
                       << r.undefPosBody << "/"
                       << r.undefNegBody << "/"
                       << r.potentiallySupportedAtom
                       << " -> "
                       << ( uSatisfied ? "sat/" : "psupport/" )
                       << uHead << "/"
                       << uPosBody << "/"
                       << uNegBody << "/"
                       << uPotentiallySupportedAtom
                       << endl;
                }
            }
        }
    else
        {
        // Do not update the counters if the rule has been satisfied
        // and is still satisfied, since counter values are
        // meaningless in this case.
        // Exception: the potentiallySupportedAtom value.
        if( ! ( uSatisfied && r.satisfied ) )
            {
            r.satisfied = uSatisfied;
            r.undefHead = uHead;
            r.undefPosBody = uPosBody;
            r.undefNegBody = uNegBody;
            }
        r.potentiallySupportedAtom = uPotentiallySupportedAtom;
        }

    return ( (!check) || change );
    }

//////////////////////////////////////////////////////////////////////////////
/// recomputes the undefined counters in a set of rules, optionally checking
/// whether changes occured
///
/// @param rules, the GRULES whose counters are to be recomputed
/// @param constraints, the GCONSTRAINTS related to these rules (These are
///        only needed for the checking feature.)
/// @param I, a GINTERPRET used to determine the undefinedness of atoms.
/// @param check, specifies whether a check on changes should be performed
///        defaults to false
///
/// @return if check was true, returns true if a change occured; if check was
///         false, true is returned in any case

bool MODEL_GENERATOR::RecomputeRuleCounters(
    const GINTERPRET&   I,
    bool                check ) 
    {
    bool change = false;

    vector<unsigned>* oldPotentiallySupportingRulesP = 0;

    if(check)
        oldPotentiallySupportingRulesP = 
            createPotentiallySupportingRulesCounters();

    // traverse all GRULEs in rules

    for (GRULES::iterator ri = rules.begin(); ri != rules.end(); ri++)
        {
        bool change1=RecomputeCounters(*ri, I, check);
        change = change || change1;
	}

    if(check)
        {
        bool change1 = 
            checkPotentiallySupportingRulesCounters(
                oldPotentiallySupportingRulesP);
        change = change || change1;
        }

    if( oldPotentiallySupportingRulesP )
        delete oldPotentiallySupportingRulesP;

    // if check was false, return true, else the change flag status
    return ( (!check) || change );
    }

//////////////////////////////////////////////////////////////////////////////
/// recomputes the undefined counters of a ground constraint, optionally
/// checking whether changes occured
///
/// @param c, the GCONSTRAINT whose counters are recomputed.
/// @param I, a GINTERPRET used to determine the undefinedness of atoms.
/// @param check, specifies whether a check on changes should be performed
///        defaults to false
///
/// @return if check was true, returns true if a change occured, if check was
///         false, true is returned in any case

static bool RecomputeCounters(GCONSTRAINT& c, const GINTERPRET& I, bool check) 
    {
    bool change = false;

    // If we have to check for changes, record current counter
    // values in local variables.
    unsigned int uPosBody = 0, uNegBody = 0;
    bool uSatisfied = false;

    // Check the positive body for undefined (w.r.t. I) literals.
    // Note that positive literals whose atoms are must-be-true are
    // treated as true in constraints.
    for (GCONJUNCTION::const_iterator l = c.pos_begin();
         l != c.pos_end(); l++)
        {
        if( I.isUndefined(*l) )
            uPosBody++;
        else if( I.isFalse(*l))
            uSatisfied = true;
        }

    // Check the negative body for undefined (w.r.t. I) literals.
    for (GCONJUNCTION::const_iterator l = c.neg_begin();
         l != c.neg_end(); l++)
        {
        if( I.isUndefined(*l) )
            uNegBody++;
        else if( I.isFalse(*l) || I.isMustBeTrue(*l) )
            // A negative literal whose atom is must-be-true or a
            // false literal in the body both satisfy the constraint.
            uSatisfied = true;
        }

    if (check)
        {
        // Check whether some counters have changed.
        // If a constraint is satisfied, the counter values are
        // undefined.  So if the constraint was satisfied before and
        // is also currently satisfied, differing counter values are
        // not considered as a change.
        if ( ! ( uSatisfied && c.satisfied )
             && ( uSatisfied != c.satisfied 
                  || uPosBody != c.undefPosBody
                  || uNegBody != c.undefNegBody ) )
            {
            // If so, set the change flag
            change=true;
            // and output diagnostics if requested
            if (TraceLevel >= 3)
                {
                cdebug << "Counter change: " << c << "   "
                       << ( c.satisfied ? "sat/" : "psupport/" )
                       << c.undefPosBody << "/"
                       << c.undefNegBody
                       << " -> "
                       << ( uSatisfied ? "sat/" : "psupport/" )
                       << uPosBody << "/"
                       << uNegBody
                       << endl;
                }
            }
        }
    else
        // Do not update the counters if the constraint has been
        // satisfied and is still satisfied, since counter values are
        // meaningless in this case.
        if( ! ( uSatisfied && c.satisfied ) )
            {
            c.satisfied = uSatisfied;
            c.undefPosBody = uPosBody;
            c.undefNegBody = uNegBody;
            }
            
    return ( (!check) || change );
    }

//////////////////////////////////////////////////////////////////////////////
/// recomputes the undefined counters of the GCONSTRAINTs in constraints
/// checking whether changes occured
///
/// @param constraints, a list of GCONSTRAINTs
/// @param I, a GINTERPRET used to determine the undefinedness of atoms.
/// @param check, specifies whether a check on changes should be performed
///        defaults to false
///
/// @return if check was true, returns true if a change occured, if check was
///         false, true is returned in any case

static bool RecomputeConstraintCounters(
          GCONSTRAINTS &constraints,
    const GINTERPRET   &I,
    bool                check ) 
    {
    bool change=false;

    for( GCONSTRAINTS::iterator ci = constraints.begin();
         ci != constraints.end();
         ci++ )
        {
        bool change1=RecomputeCounters(*ci, I, check);
        change = change || change1;
        }

    // if check was false, return true, else the change flag status
    return (!check) || change;
    }

//////////////////////////////////////////////////////////////////////////////
/// recomputes the undefined counters in the rules and constraints;
/// optionally checks whether changes occured.
///
/// @param I, a GINTERPRET used to determine the undefinedness of atoms.
/// @param check, specifies whether a check on changes should be performed
/// @return false if check is true and no change occured, true in all other
///         cases

bool MODEL_GENERATOR::RecomputeCounters(
    const GINTERPRET   &I,
    bool                check )
    {
    bool c1=RecomputeRuleCounters(I,check);
    bool c2=RecomputeConstraintCounters(constraints,I,check);

    if( check )
        return c1 || c2;
    else
        return true;
    }

//////////////////////////////////////////////////////////////////////////////

#define PUSH(x) { ShortStack.push(level,&x); }
#define UPUSH(x) { UnsignedStack.push(level,&x); }


// This function returns TRUE if two GATOMs belong to the same COMPONENT.
bool MODEL_GENERATOR::sameComponent(const GATOM &a1, const GATOM &a2) const
    {
    return ((*posGroundDG).getAtomComponent(a1) 
            == (*posGroundDG).getAtomComponent(a2));
    } // sameComponent()



// This function returns TRUE if a given GATOM belongs to a given
// COMPONENT.
// NOTE: It should work also with a GLITERAL, that inherits from GATOM.
bool MODEL_GENERATOR::belongToComponent(const COMPONENT_INDEX &C, const GATOM &a) const
    {
    return ((*posGroundDG).getAtomComponent(a) == C);
    } // belongToComponent()



// This function return a const_iterator to the first GATOM that is
// true in a given GDISJUNCTION (w.r.t. a given GINTERPRET I).  
// NOTE: returns D.end() if there is no one [i.e. the GDISJUNTION is
// NOT TRUE]. Remember that D.end() is not the last element, but
// last+1.
GDISJUNCTION::const_iterator MODEL_GENERATOR::findFirstTrue(const GDISJUNCTION &D, 
                                                            const GINTERPRET &I) const
    {
    GDISJUNCTION::const_iterator i;
    for (i=D.begin(); i != D.end() && (!I.isTrue(*i)); i++)
        {
        } // for()
    return i;
    } // findFirstTrue()



// This function returns a const_iterator to the first GATOM that
// belongs to the given COMPONENT in a given GDISJUNCTION.  
// NOTE: this function should be invoked only by the function
// propagateGUS() [see below].  So we are sure that in the CONJUNCTION
// (that should be the head of a rule) there is at least an element
// that belongs to the COMPONENT.  In the case of HCF components
// should be the only GATOM belonging to the COMPONENT.
GDISJUNCTION::const_iterator MODEL_GENERATOR::findHeadGatomInC( 
    const COMPONENT_INDEX &C, const GDISJUNCTION &D) const
    {
    GDISJUNCTION::const_iterator i;
    for (i=D.begin(); i != D.end(); i++)
        {
        if (belongToComponent(C,*i)) break;
        } // for()

    return i;
    } // findHeadGatomInC()



// The function checks if the head of a GRULE is true because of a
// GATOM that does not belong to a given COMPONENT C.  
// NOTE: if there are no true GATOMs in the head (i.e. it is false),
// the function returns false, of course. The function also returns
// false if the head of the rule is true, but only via GATOMs of
// COMPONENT C.
bool MODEL_GENERATOR::trueByExternal(const GRULE &r,
                                     const COMPONENT_INDEX &C, 
                                     const GINTERPRET &I) const 
    {
    assert ( r.hasHead() );
    
    // If the Head is false we don't need to perform the check
    if((r.satisfied & GRULE::SATISFIED_BY_HEAD)
       || (r.mbtHead > 0))
        { 
        const GDISJUNCTION &D = r.getHead();
        
        for (GDISJUNCTION::const_iterator i = D.begin(); i != D.end(); i++)
            {
            if ( (! belongToComponent(C,*i))
                 && ( (I.isTrue(*i)) || (I.isMustBeTrue(*i)) ) )
                return true;
            } // for()
        } // if()

    return false;  
    } // trueByExternal()



// This function checks if a rule has to be considered by the
// algorithm computeGUS. If it has the body false or there is in the
// head a GATOM TRUE that does not belong to the current component, we
// don't have to take care of it (so we return FALSE).  It implements
// the first part of the mask we have to apply to the program to
// filter rules during the computation of the GUS for a COMPONENT
// C. [see comments to the function computeGUS()].
// NOTE: Constraints are not considered useful.

bool MODEL_GENERATOR::usefulRule(
    const COMPONENT_INDEX &C,
    const GINTERPRET &I,
    const GRULE &r ) const
    {
    // A simple check to avoid null-pointers troubles
    if (r.hasBody())
        {
        if (r.satisfied & GRULE::SATISFIED_BY_BODY)
            return false;
        else
            return ((trueByExternal(r, C, I)) == false);
        } // if()
    else
        {
        assert( r.hasHead() );
        return trueByExternal(r, C, I) == false;
        }
    } // usefulRule()


//////////////////////////////////////////////////////////////////////////////
unsigned MODEL_GENERATOR::countPosBodyOccInC(
// Compute the number of positive ground literals  in the body of a rule r,
// discarding aggregates and literals that do not belong to the component C.
// This implements the second part of the mask we apply to the program to
// filter rules during the computation of the GUS for a component; cf. the
// comments for computeGUS().
//
    const COMPONENT_INDEX &C,
    const GRULE &r) const
    {
    unsigned int occ=0;
    // A simple check to avoid null-pointers troubles
    if (r.hasBody())
        {
        const GCONJUNCTION *body = r.getBody();
        for( GCONJUNCTION::const_iterator l = body->pos_begin(); 
             l != body->pos_end(); l++ )
            {
            if( belongToComponent(C, *l)  &&  ! l->isAggregate() )
                occ++;
            }
        }
    
    return occ;
    } // countPosBodyOccInC()


// This is the propagation algorithm. We rewrite it from scratch (for
// our purposes we need something far away less complex than the
// general one implemented in (generate.C)).
bool MODEL_GENERATOR::propagateGUS(const COMPONENT_INDEX &C, 
                                   GINTERPRET &I)
    {
    // This boolean is useful to check if the current computeGUS
    // execution derived something or nothing.
    bool somethingNew = false;

    // Simply, we have to propagate all truth-values that are in the queue
    while (!GUSqueue.empty())
        {        
        // We take the first atom from the queue
        const GATOM at(GUSqueue.front().second);

        GUSqueue.pop();

        if( TraceLevel >= 3 )
            cdebug << "GUS [comp. " << C+1
                   << " - propagating]: extracting " << at 
                   << "." << endl;
        
        // For each rule... containing this atom (as positive literal)
        // in the body (but only rules having in the head at least an
        // atom belonging to the same COMPONENT we are processing [see
        // the class GRULEPTRS in (generate.h)]).
        for (GRULEPTRS::const_iterator r
                 = (PosBodyOccurrences[at.getIndex()]).sameComponentBegin(); 
             r != (PosBodyOccurrences[at.getIndex()]).sameComponentEnd(); r++)
            {
            // Here we apply the mask on the program (see computeGUS()).
            if ((**r).isUsefulForGUS())
                {
                (**r).posBodyUndefInCDecr();

                // Just check if the rule is satisfied, and if it is
                // so set as TRUE the atom in the head .
                if ((**r).posBodyUndefInC() == 0)
                    {
                    // If we are in the body of the if() statement,
                    // the current rule is satisfied (through our masking,
                    // of course). So we can derive the truth-value of the 
                    // atom belonging to the current COMPONENT that is
                    // in the head.
                    // NOTE: We check if it was already found.
                    
                    assert( (**r).hasHead() );
                    assert( (findHeadGatomInC(C, (**r).getHead()))
                            != ((**r).getHead()).end() );
                    const GATOM atm = *(findHeadGatomInC(C, (**r).getHead())); 
                                        
                    if (! GUSInterpret.isTrue(atm) )
                        {                        
                        GUSInterpret.addTrueFromUndefined(atm);
                        
                        GUSqueue.push(TRUE_FROM_UNDEFINED_V, atm);

                        if( TraceLevel >= 3 )
                            cdebug << "GUS [comp. " << C+1
                                   << " - propagating]:   " << atm 
                                   << " 'non-unfounded' because of " 
                                   << **r << endl;
                        }
                    else
                        if( TraceLevel >= 3 )
                            cdebug << "GUS [comp. " << C+1
                                   << " - propagating]:   " << atm 
                                   << " 'non-unfounded' because of " 
                                   << **r << " (but was already reached)."
                                   << endl;
                    }// if()
                } // if()
            } // for()
        } // while()
    
    
    COMPONENT& component = (*posGroundDG).getComponents()[C];

    for (COMPONENT::const_iterator i = component.begin(); 
         i != component.end(); i++)
        {
        // NOTE: *i is an index, so we first create the corresponding
        // GATOM.
        const GATOM atm(*i);

        if ( GUSInterpret.isUndefined(atm) && !atm.isAggregate() )
            if (I.isTrue(atm) || I.isMustBeTrue(atm)) 
                {                
                somethingNew = true;
                
                if( TraceLevel >= 3 )
                    {
                    cdebug << "GUS [comp. " << C+1
                           << " - exiting]:   => assuming " << atm
                           << " FALSE - INCONSISTENCY on it."
                           << endl;
                    }

                return false;
                }
            else
                {
                // NOTE: we only set unfounded atoms to false if they are
                // neither false already nor aggregates.
                if( I.isUndefined(atm) )
                    {
                    somethingNew = true;

                    if( TraceLevel >= 3 )
                        {
                        cdebug << "GUS [comp. " << C+1
                               << " - exiting]:   => assuming " << atm 
                               << " FALSE (unfounded, belongs to the GUS)." 
                               << endl;
                        }

                    Propagate_Derive_False(atm, I);
                    }
                } // else                        
        } // for()
    
    if( TraceLevel >= 3 )
        {
        cdebug << "GUS: exiting... ";
        if (!somethingNew)
            cdebug << " and nothing new was derived.";
        cdebug << endl;
        }
    
    return true;   
    } // propagateGUS()



bool MODEL_GENERATOR::computeGUS(const COMPONENT_INDEX &C, GINTERPRET &I) 
// Compute the Greatest Unfounded Set (GUS) of a component C w.r.t. an
// interpretation I. Then sets to false (in the current interpretation)
// all atoms belonging to the GUS.  The boolean returned is useful to
// signal if there is inconsistency (i.e. we derive as FALSE an atom
// already set as true or as mbt in the current interpretation).
//
// NOTE: GUSqueue will contain as true all atoms reached during the
// propagation. In the end, all the other atoms belonging to C but set
// as undefined in GUSqueue are the GUS in C.
//
// NOTE: We have to ignore aggregate atoms, that is, skip them when counting
// positive body literals, and do not set them false upon exit.
    {
    // NOTE: at this moment, we process only HCF components.
    assert( posGroundDG->isHCF(C) && posGroundDG->isCyclic(C) );
        
    if( TraceLevel >= 3 )
        cdebug << "GUS: processing component " << C+1
               << "." << endl;
    
    // We need to create a mask on the program to see only the
    // subprogram useful to compute the GUS restricted to the
    // component.
    
    // Masking the program:
    // * we have to discard all rules having the body false or having
    // in the head some positive atoms not belonging to the current
    // COMPONENT [1st part of the masking].
    // * in the remaning rules we don't have to take care about
    // negative literals and about literals not belonging to the
    // current COMPONENT [2nd part of the masking].  

    // NOTE: the masking is currently obtained with the function
    // usefulRule(), but we use it to set a boolean into the class
    // grule to avoid recomputing.
    
    // We rewrite from scratch the propagation algorithm, because it
    // is more simple than in the general case, and it has to be as
    // quick as possible.
            
    // This is where we set the truth-values of the atoms belonging to
    // the COMPONENT.
    
    // NOTE: In the beginning we set them all as UNDEFINED. During the
    // propagation they can become true. Those left as UNDEFINED in the
    // end, are UNFOUNDED.

    
    COMPONENT& component = (*posGroundDG).getComponents()[C];   

    // If all the atoms of the component are already false it is
    // useless to perform the computation.
    bool allFalse = true;
    bool currentAtomFalse;

    // For all the atoms belonging to C...
    for (COMPONENT::const_iterator i = component.begin(); 
         i != component.end(); i++)
        {
        // NOTE: *i is an index, so we first create the corresponding
        // GATOM.
        const GATOM atm(*i);

        // First, we set all atoms belonging to C as undefined in
        // GUSInterpret. But if they are false in the extern
        // interpretation, we set them as false.
        //
        // NOTE: we need to set at the beginning all of them as
        // undefined (excepting those we can set as false at the
        // beginning, without checking), otherwise an assertion will
        // fire. So the following if() statement will check.

        if (I.isFalse(atm))
            {
            currentAtomFalse = true;

            // We can directly set this to false in our local interpretation.
            GUSInterpret.forceFalse(atm);
            } //if()
        else
            {
            // We need to set this atom as undefined, otherwise later
            // on an assertion will fire.
            GUSInterpret.addUndefined(atm);

            currentAtomFalse = false;

            allFalse = false;
            }

        if( TraceLevel >= 3 )
            {
            cdebug << "GUS [comp. " << C+1
                   << " - 'facts' searching]: considering "
                   << atm << "." << endl;
            }

        // ...Then we simply look for facts.

        // This flag says if the current atom was already reached
        // during the searching for "facts"...
        bool currentAtomReached = false;
        
        // For all the rules containing the current GATOM in the head...
        for (GRULEPTRS::const_iterator r=headRules_begin(atm); 
             r != headRules_end(atm); r++) 
            {
            // If the atom is already false it is useless to check for
            // it. Also if it was already reached during this "facts"
            // research...
            if ((!currentAtomFalse) && (!currentAtomReached))
                {          
                // The mask...
                if (usefulRule(C, I, (**r)))
                    {                                                       
                    if( TraceLevel >= 3 )
                        {
                        cdebug << "GUS [comp. " << C+1
                               << " - 'facts' searching]:"
                                  "   possibly supported by "
                               << **r << endl;
                        }

                    // This count all the atoms belonging to the
                    // current component that appear in the body of
                    // the rule. If the result is ZERO the current
                    // rule is a fact, for our current purposes...
                    (**r).posBodyUndefInCSet(countPosBodyOccInC(C, (**r)));
                    
                    if ((**r).posBodyUndefInC() == 0)
                        {        
                        // If we are in the body of the if()
                        // statement, the current rule is a fact. So
                        // we can derive the truth-value of the atom.
                        //
                        // NOTE: We check if it was already found.
                        if (!(GUSInterpret.isTrue(atm)))
                            {
                            // We remember that the atom was
                            // reached. So we can avoid to perform
                            // other checks over other rules
                            // containing it in the head.
                            currentAtomReached = true;

                            // The atom become "founded".
                            GUSInterpret.addTrueFromUndefined(atm);
                            
                            // It will generate propagation...
                            GUSqueue.push(TRUE_FROM_UNDEFINED_V, atm);
                            
                            if( TraceLevel >= 3 )
                                {
                                cdebug << "GUS [comp. " << C+1 
                                       << " - 'facts' searching]:"
                                          "     Yes! Atom " 
                                       << atm << " 'non-unfounded'."
                                       << endl;
                                }
                            }
                        else
                            if( TraceLevel >= 3 )
                                {
                                cdebug << "GUS [comp. " << C+1 
                                       << " - 'facts' searching]:"
                                          "     Yes! Atom " 
                                       << atm << " 'non-unfounded',"
                                          " (but was ALREADY reached)." 
                                       << endl;
                                }
                        
                        // We don't set as useful the rule from which
                        // we derive an atom as a "fact", because it
                        // won't allow us to derive something else in
                        // the future.
                        (**r).setUsefulForGUS(false);
                        } // if()
                    else
                        {
                        // Here we set the flag on the rule, so we can
                        // just check it, during the propagateGUS,
                        // instead of recall the method usefulRule().
                        // This flag allow us to skip all rule that
                        // have the body false or the head true
                        // because of a GATOM not belonging to the
                        // current component.
                        //
                        // NOTE: We don't need to do something else:
                        // during the propagateGUS we are sure that
                        // all the rules we have to check are properly
                        // set.
                        (**r).setUsefulForGUS(true);

                        if( TraceLevel >= 3 )
                            {
                            cdebug << "GUS [comp. " << C+1 
                                   << " - 'facts' searching]:"
                                      "     No." << endl;
                            }
                        } // else
                    } // if()
                else
                    // Rule not useful because usefulRule() returned
                    // false.
                    (**r).setUsefulForGUS(false);
                } // if()
            else
                // Rule not useful because the (only) [here there are
                // only HCF components!] atom belonging to the head is
                // already false -- Or this was just founded as a
                // "fact" and all its rules are useless to find
                // something else in this component (we are processing
                // only HCF components!).
                //
                // NOTE: at the moment we set as false only the rules
                // checked after the one that caused the "finding";
                // the previous ones will remain "useful" (since they
                // were set so), but no semanthic problems will arise.
                (**r).setUsefulForGUS(false);
            } // for()
        } // for()

    // If all the atoms belonging to the component are already false,
    // it is useless to go on.
    if (!allFalse)
        return propagateGUS(C, I);
    else
        {
        if( TraceLevel >= 3 )
            cdebug << "GUS [comp. " << C+1 
                   << "]: all atoms already false"
                      " in the extern interpretation --> exiting..."
                   << endl;
        
        return true;
        }
    } 


////////////////////////////////////////////////////////////////////////////
// Procedures that handle the componentQueues for computeGUS()
////////////////////////////////////////////////////////////////////////////


// This procedure simply pushes a component onto the appropriate
// queue, providing some debug outputs.
void MODEL_GENERATOR::pushComponent(const COMPONENT_INDEX &C, 
                                    const GRULE &r, const GATOM &a)
    {
    if ((*posGroundDG).isCyclic(C))
        {
        if ((*posGroundDG).isHCF(C))
            {
            // debug
            if( TraceLevel >= 3 )
                {
                cdebug << "GUS: Component " 
                       << C+1
                       << " added to HCFcomponentQueue because "
                       << a 
                       << " has lost supporting rule " 
                       << r
                       << endl;
                }
            HCFcomponentQueue.push(C);                      
            }
        else
            {
            // FIXME: This should be implemented in the future (here
            // we would like to process cyclic and non-HCF components
            // -- It should be possible to find a GUS inside a
            // component non-HCF w.r.t. a non-complete interpretation
            // that is unfounded-free [in polynomial time]).
            }
        }
    else
        {
        if( TraceLevel >= 3 )
            {
            cdebug << "GUS: Component " 
                   << C+1
                   << " won't be added to any queue. "
                   << a
                   << " lost supporting rule " 
                   << r << "  but is not cyclic."
                   << endl;
            }
        } // else
    } // pushComponent()

    
//////////////////////////////////////////////////////////////////////////////
// Derive a GATOM as true (from undefined). This involves setting 
// the corresponding TruthValue for the atom in the interpretation
// and inserting the appropriate pair into the global queue for further
// propagation.
//
// @param a,  the GATOM to be derived
// @param I, the current interpretation 
//
inline void MODEL_GENERATOR::Propagate_Derive_TrueFromUndefined( 
    const GATOM& a,
    GINTERPRET& I )
    {
    // Only atoms which are currently undefined should
    // be derived as true from undefined.
    assert( I.isUndefined(a) );

    // Update the interpretation and insert the appropriate pair into
    // the queue.
    I.addTrueFromUndefined(a);
    Queue.push(TRUE_FROM_UNDEFINED_V,a);

    HC.UndefinedEliminated++;
    }

//////////////////////////////////////////////////////////////////////////////
// Derive a GATOM as true (from must-be-true). This involves setting 
// the corresponding TruthValue for the atom in the interpretation
// and inserting the appropriate pair into the global queue for further
// propagation.
//
// @param a,  the GATOM to be derived
// @param I, the current interpretation 
//
inline void MODEL_GENERATOR::Propagate_Derive_TrueFromMustBeTrue( 
    const GATOM& a,
    GINTERPRET& I )
    {
    // Only atoms which are currently must-be-true should be derived
    // as true from must-be-true.
    assert( I.isMustBeTrue(a) );

    HC.MBTEliminated++;

    switch(getPotentialSupport(a))
        {
        case 2:
            HC.MBTEliminatedLevel2++;
            break;
        case 3:
            HC.MBTEliminatedLevel3++;
            break;
        }

    // Update the interpretation and insert the appropriate pair into
    // the queue.
    I.addTrueFromMustBeTrue(a);
    Queue.push(TRUE_FROM_MUST_BE_TRUE_V,a);
    }

//////////////////////////////////////////////////////////////////////////////
// Derive a GATOM as true. This involves setting the corresponding
// TruthValue for the atom in the interpretation and inserting the
// appropriate pair into the global queue for further propagation.
//
// @param a,  the GATOM to be derived
// @param I, the current interpretation 
//
inline void MODEL_GENERATOR::Propagate_Derive_True( 
    const GATOM& a,
    GINTERPRET& I )
    {
    // Only atoms which are currently undefined or must-be-true should
    // be derived as true.
    assert( I.isUndefined(a) || I.isMustBeTrue(a) );

    // Update the interpretation and insert the appropriate pair into
    // the queue.
    if( I.isUndefined(a) )
        Propagate_Derive_TrueFromUndefined(a,I);
    else
        Propagate_Derive_TrueFromMustBeTrue(a,I);
    }

//////////////////////////////////////////////////////////////////////////////
// Derive a GATOM as must-be-true. This involves setting the atom to
// must-be-true in the interpretation and inserting the pair into the
// global queue for further propagation.
//
// @param a,  the GATOM to be derived
// @param I, the current interpretation 
//
inline void MODEL_GENERATOR::Propagate_Derive_MustBeTrue( 
    const GATOM& a,
    GINTERPRET& I )
    {
    // Only atoms which are currently undefined should be derived as
    // must-be-true.
    assert( I.isUndefined(a) );

    HC.UndefinedEliminated++;
    HC.MBTIntroduced++;

    switch(getPotentialSupport(a))
        {
        case 2:
            HC.MBTIntroducedLevel2++;
            break;
        case 3:
            HC.MBTIntroducedLevel3++;
            break;
        }

    // Update the interpretation and insert the appropriate pair into
    // the queue.
    I.addMustBeTrue(a);
    Queue.push(MUST_BE_TRUE_V,a);
    }

//////////////////////////////////////////////////////////////////////////////
// Derive a GATOM as false. This involves setting the atom to false in
// the interpretation and inserting the pair into the global queue for
// further propagation.
//
// @param a, the GATOM to be derived
// @param I, the current interpretation 
//
inline void MODEL_GENERATOR::Propagate_Derive_False( 
    const GATOM& a,
    GINTERPRET& I )
    {
    // Only atoms which are currently undefined should be derived as
    // false.
    assert( I.isUndefined(a) );

    HC.UndefinedEliminated++;

    // Update the interpretation and insert the appropriate pair into
    // the queue.
    I.addFalse(a);
    Queue.push(FALSE_V,a);
    }

//////////////////////////////////////////////////////////////////////////////
// If the potential support counter of a true or must-be-true atom
// becomes one, the last potentially supporting rule must be the
// supporting rule for it in every stable model which extends
// I. Therefore, in this rule, all other head atoms must be false and
// the body becomes must-be-true.
//
// @param a, the true or must-be-true atom whose pot. support counter is one
// @param I, the current interpretation
//
// @return false upon inconsistency
//
inline bool MODEL_GENERATOR::Propagate_OnePotentiallySupportingRule(
//
    const GATOM& a,
    GINTERPRET&  I )
    {
    assert( I.isMustBeTrue(a) || I.isTrue(a) );
    assert( getPotentialSupport(a) == 1 );

    // Find a's last potentially supporting rule and derive all other
    // head atoms in it as false, and the body as must-be-true.

    GRULEPTRS::const_iterator rule;

    for( rule = headRules_begin(a);
         rule != headRules_end(a) && ! (**rule).potentiallySupports(a);
         rule++ )
        {}

    // If a does not have any potentially supporting rule, an
    // inconsistency has been detected.
    // This most probably happens only if the same atom occurs twice
    // even more often in the same rule head, which should no longer
    // be the case.
    assert( rule != headRules_end(a) );

#ifndef NDEBUG
    // No other rule with a in the head should potentially support a.
    GRULEPTRS::const_iterator rule1 = rule;
    for( rule1++;
         rule1 != headRules_end(a);
         rule1++ )
        assert( ! (**rule1).potentiallySupports(a) );
#endif

    const GCONJUNCTION *body = (**rule).getBody();
    const GDISJUNCTION *head = &(**rule).getHead();

    assert( head );

    // If there is more than one must-be-true atom in the head, or the
    // rule is satisfied because the body is false (but the flag has
    // not been set yet) the rule cannot support a, which is an
    // inconsistency.
    // Note that body may be NULL.
    // Also note that the rule can also be satisfied because the head
    // is true. But the head may also be true because a might have
    // been derived as true in the meantime - in this case there is no
    // inconsistency. We could test whether head - a is true, but it
    // is easier to defer this test.
    if( (**rule).mbtHead > 1 || ( body && I.isFalse(*body) ) )
        return false;

    // Derive all head atoms (apart from a) as false.
    // Attention: In the case where such a head atom also occurs in
    // the positive body, the body will become false, in which case
    // the rule cannot support a, which is an inconsistency. Since we
    // look at each atom of the positive body lateron, the check is
    // done there.
    for( GDISJUNCTION::const_iterator h = (*head).begin();
         h != (*head).end();
         h++ )
        {
        // Nothing has to be done for a or for atoms which are already
        // false.
        if( *h == a || I.isFalse(*h) ) continue;

        // If an atom (different from a) is must-be-true or true, the
        // rule cannot support a, which is an inconsistency.
        if( I.isMustBeTrue(*h) || I.isTrue(*h) )
            return false;

        if( TraceLevel >= 2 )
            cdebug << indent(level)
                   << "Deriving <FALSE," << *h << "> from " 
                   << **rule
                   << "  [in head of last pot. supporting rule for mbt/true " 
                   << a << "]" << endl;

        Propagate_Derive_False(*h,I);
        }

    if( body )
        {
        // Derive all atoms in the positive body as must-be-true.
        for( GCONJUNCTION::const_iterator l = (*body).pos_begin();
             l != (*body).pos_end();
             l++ )
            {
            // Nothing has to be done for atoms which are already true
            // or must-be-true.
            if( I.isTrue(*l) || I.isMustBeTrue(*l) ) continue;

            // If an atom occurs both in the head and in the positive
            // body of the rule, we could have derived this atom as
            // false before. In this case, the rule cannot support a,
            // which is an inconsistency.
            if( I.isFalse(*l) )
                return false;

            if( TraceLevel >= 2 )
                cdebug << indent(level) << "Deriving <MUST_BE_TRUE," 
                       << GATOM(*l) << ">" 
                       << "  [in pos. body of last pot. supporting rule for "
                          "mbt/true atom " << a << "]" << endl;

            Propagate_Derive_MustBeTrue(*l,I);
            }

        // Derive all atoms in the negative body as false.
        for( GCONJUNCTION::const_iterator l = (*body).neg_begin();
             l != (*body).neg_end();
             l++ )
            {
            // If a negative literal is true, then its atom is already
            // false.
            if( I.isTrue(*l) ) continue;

            // If an atom occurs both in the positive and in the
            // negative body of the rule, we have derived it as mbt
            // before (and the literal at hand is false now). This
            // case represents an inconsistency.
            if( I.isFalse(*l) )
                return false;

            if( TraceLevel >= 2 )
                cdebug << indent(level)
                       << "Deriving <FALSE," << GATOM(*l) << "> from "
                       << **rule
                       << "  [in neg. body of last pot. supporting rule for "
                          "mbt/true " << a << "]" << endl;

            Propagate_Derive_False(*l,I);
            }
        }

    return true;
    }

//////////////////////////////////////////////////////////////////////////////
// Adjust the potential support counters of head atoms occurring in a
// rule which becomes satisfied, set the potentially supported atom,
// if it exists, adjust heuristic values and make possible inferences.
//
// @param rule, the rule which becomes satisfied
// @param patom, a pointer to the satisfying head atom if it exists (else 0)
// @param I, the interpretation of the context
//
// @return false if an inconsistency has been detected, else true.
//
inline bool MODEL_GENERATOR::Propagate_RuleSatisfaction_AdjustPotentialSupport(
//
    GRULE &rule,
    const GATOM *patom,
    GINTERPRET &I)
    {
#ifndef NDEBUG
    // Only used for an assertion (no duplicate head atoms).
    bool fixedFirstOccurrence = false;
#endif

    // For each atom in the head, which is different from the one
    // which satisfies the rule in the case when the rule is satisfied
    // by its head, decrease its potential support counter.
    // Also set the rule's potentiallySupportedAtom to the position of
    // the first occurrence of the atom which satisfies the rule, if
    // it exists.
    for( GDISJUNCTION::const_iterator h=rule.getHead().begin();
         h != rule.getHead().end();
         h++ )
        {
        if( patom && *h == *patom ) 
            {
#ifndef NDEBUG
	    // No duplicate head atoms should exist.
            assert( ! fixedFirstOccurrence );
	    fixedFirstOccurrence = true;
#endif

	    // No particular head atom should have been supported before.
	    assert( rule.potentiallySupportedAtom == 0 );
	    PUSH( rule.potentiallySupportedAtom );
	    rule.potentiallySupportedAtom = h - rule.getHead().begin() + 1;
            }
        else
            {
            UPUSH( getPotentialSupportAsReference(*h) );
            decPotentialSupport(*h);

            // * component push * //
            if (haveToExecuteComputeGUS)
                {
                if (! I.isFalse(*h))
                    {
                    const COMPONENT_INDEX C 
                        = (*posGroundDG).getAtomComponent(*h);
                    
                    pushComponent(C, rule, (*h));
                    } // if()
                } // if()

            switch( getPotentialSupport(*h) )
                {
                case 0:
                    // If the rule was the last potentially supporting
                    // rule for an undefined atom, its negation should be
                    // assumed, unless the atom is already "must be true",
                    // in which case an inconsistency arises.
                    if( I.isUndefined(*h) )
                        {
                        if( TraceLevel >= 2 )
                            cdebug << indent(level)
                                   << "Deriving <FALSE," << *h << "> from "
                                   << rule
                                   << "  [single pot. supporting rule is dead]"
                                   << endl;

                        Propagate_Derive_False(*h,I);
                        }
                    else if( I.isMustBeTrue(*h) )
                        return false;
                    break;
                case 1:
                    if( I.isMustBeTrue(*h) || I.isTrue(*h) )
                        if( ! Propagate_OnePotentiallySupportingRule(*h,I) )
                            return false;
                    break;
                case 2:
                    if( I.isMustBeTrue(*h) )
                        {
                        HC.MBTIntroducedLevel2++;
                        HC.MBTEliminatedLevel3++;
                        }
                    break;
                case 3:
                    if( I.isMustBeTrue(*h) )
                        {
                        HC.MBTIntroducedLevel3++;
                        }
                    break;
                }
            }
        }

    return true;
    }

//////////////////////////////////////////////////////////////////////////////
// If a (previously unsatisfied) rule becomes satisfied because of a
// true head during propagation, this procedure is called. It may
// also be called if the rule has already been satisfied before.
//
// @param rule, the rule which becomes satisfied
// @param atom, the atom which has become true
// @param I, the current interpretation
//
// @return false upon inconsistency
//
inline bool MODEL_GENERATOR::Propagate_RuleBecomesSatisfiedByHead(
//
    GRULE&       rule,
    const GATOM& atom,
    GINTERPRET&  I )
    {
    // If the rule is already satisfied, true is returned, since no
    // inconsistency has arisen. If the rule potentially supports an
    // atom, then this atom is no longer potentially supported by the
    // rule (unless it is equal to the atom by which this rule becomes
    // satisfied).
    if( rule.satisfied )
        {
        if( rule.satisfied == GRULE::SATISFIED_BY_BODY )
            {
            // The rule cannot support an atom.
            assert( rule.potentiallySupportedAtom == 0 );
            PUSH( rule.satisfied );
            rule.satisfied = GRULE::SATISFIED_BY_HEAD_AND_BODY;
            }
        else if( rule.potentiallySupportedAtom != 0 )
            {
            assert( rule.satisfied == GRULE::SATISFIED_BY_HEAD );

            // The rule potentially supports an atom.
            assert( rule.getHead().end() - rule.getHead().begin() 
                    >= rule.potentiallySupportedAtom);

            // Set the iterator h to the potentially supported atom.
            GDISJUNCTION::const_iterator h = 
                rule.getHead().begin() + rule.potentiallySupportedAtom - 1;

            // If the atom by which this rule becomes satisfied is
            // equal to the potentially supported atom, this atom is
            // still potentially supported by the rule.
            // This should not occur because atoms occur at most once
            // per rule and should also derived at most once.
            assert( atom != *h );

            // Otherwise, the previously potentially supported atom is
            // no longer potentially supported. Therefore, the number
            // of rules which potentially support this atom is
            // decreased.
            UPUSH( getPotentialSupportAsReference(*h) );
            decPotentialSupport(*h);

            // * component push * //
            if (haveToExecuteComputeGUS)
                {
                // The supported atom should never be false.
                assert (! I.isFalse(*h));

                pushComponent((*posGroundDG).getAtomComponent(*h), rule, (*h));
                } // if()

            const GATOM& a = *h;

#ifndef NDEBUG
            // There should not occur any copies of the potentially
            // supported atom in the head.
            for( h++ ; h != rule.getHead().end() ; h++ )
                {
                assert( a != *h );
                }
#endif

            // If there is no potentially supporting rule left, the
            // atom has no potential support, although it is currently
            // true. This is an inconsistency.
            if( getPotentialSupport(a) == 0 )
                return false;

            PUSH( rule.potentiallySupportedAtom );
            rule.potentiallySupportedAtom = 0;
            }

        return true;
        }

    // At this point, the rule is not satisfied yet.

    if( rule.partiallyViolated() )
        {
        // A partially violated rule has been eliminated.
        HC.PVClauseEliminated++;

#ifdef WITH_PVCLEVELS
        switch( rule.partialViolationLevel() )
            {
            case 0:
                assert(0);
            case 2:
                HC.PVClauseEliminatedLevel2++;
                break;
            case 3:
                HC.PVClauseEliminatedLevel3++;
                break;
            default:
                // nothing
                break;
            }
#endif

        }

    // A clause has been eliminated.
    HC.ClauseEliminated++;

    // Mark rule as satisfied.
    PUSH( rule.satisfied );
    rule.satisfied = GRULE::SATISFIED_BY_HEAD;

    // Adjust the potential support counters.
    return Propagate_RuleSatisfaction_AdjustPotentialSupport(rule,&atom,I);
    }

//////////////////////////////////////////////////////////////////////////////
// If a (previously unsatisfied) rule becomes satisfied because of a
// false body during propagation, this procedure is called. It may
// also be called if the rule has already been satisfied before.
//
// @param rule, the rule which becomes satisfied
// @param I, the current interpretation
//
// @return false upon inconsistency
//
inline bool MODEL_GENERATOR::Propagate_RuleBecomesSatisfiedByBody(
//
    GRULE&      rule,
    GINTERPRET& I )
    {
    // If the rule is already satisfied, true is returned, since no
    // inconsistency has arisen.
    // If the rule potentially supports an atom, then this atom is no
    // longer supported by the rule.
    if( rule.satisfied )
        {
        if( rule.satisfied == GRULE::SATISFIED_BY_HEAD )
            {
            PUSH( rule.satisfied );
            rule.satisfied = GRULE::SATISFIED_BY_HEAD_AND_BODY;
            }

        if( rule.potentiallySupportedAtom != 0 )
            {
            // The rule potentially supports an atom.            

            assert( rule.getHead().end() - rule.getHead().begin() 
                    >= rule.potentiallySupportedAtom);

            // Set the iterator h to the potentially supported atom.
            GDISJUNCTION::const_iterator h = 
                rule.getHead().begin() + rule.potentiallySupportedAtom - 1;

            // The previously potentially supported atom is no longer
            // potentially supported. Therefore, the number of rules
            // which potentially support this atom is decreased.
            UPUSH( getPotentialSupportAsReference(*h) );
            decPotentialSupport(*h);

            // * component push * //
            if (haveToExecuteComputeGUS)
                {
                // The supported atom should never be false.
                assert(! I.isFalse(*h));
                    
                pushComponent((*posGroundDG).getAtomComponent(*h), rule, (*h));
                } // if()

            const GATOM& a = *h;

#ifndef NDEBUG
            // The previously potentially supported atom should not
            // occur several times in the head.
            for( h++ ; h != rule.getHead().end() ; h++ )
                {
                assert( a != *h );
                }
#endif

            // If there is no potentially supporting rule left, the
            // atom has no potential support, although it is currently
            // true. This is an inconsistency.
            if( getPotentialSupport(a) == 0 )
                return false;

            PUSH( rule.potentiallySupportedAtom );
            rule.potentiallySupportedAtom = 0;
            }

        return true;
        }

    // At this point, the rule is not satisfied yet.

    if( rule.partiallyViolated() )
        {
        // A partially violated rule has been eliminated.
        HC.PVClauseEliminated++;

#ifdef WITH_PVCLEVELS
        switch( rule.partialViolationLevel() )
            {
            case 0:
                assert(0);
            case 2:
                HC.PVClauseEliminatedLevel2++;
                break;
            case 3:
                HC.PVClauseEliminatedLevel3++;
                break;
            default:
                // nothing
                break;
            }
#endif

        }

    // A clause has been eliminated.
    HC.ClauseEliminated++;

    // Mark rule as satisfied.
    PUSH( rule.satisfied );
    rule.satisfied = GRULE::SATISFIED_BY_BODY;

    // Adjust the potential support counters.
    return Propagate_RuleSatisfaction_AdjustPotentialSupport(rule,0,I);
    }

//////////////////////////////////////////////////////////////////////////////
// This function is called with a GRULE rule, in which the body is true
// in the GINTERPRETation I, and whose undefined counter of the head is
// exactly one.
//
// @param rule, the rule which contains the single undefined head literal
// @param I, the current interpretation
//
inline void MODEL_GENERATOR::Propagate_DeriveSingleUndefinedHeadLiteral(
//
    const GRULE&      rule,
          GINTERPRET& I )
    {
    GDISJUNCTION::const_iterator h=rule.getHead().begin();
    GDISJUNCTION::const_iterator end=rule.getHead().end();

    // Find the first (and single) undefined or must-be-true head
    // atom, but do not try to search beyond the end (in case there
    // is no undefined head atom left).
    while( ! ( I.isUndefined(*h) || I.isMustBeTrue(*h) ) )
        {
        h++;
        if( h == end ) return;
        }

    // Derive the single undefined atom.
    if( TraceLevel >= 2 )
        cdebug << indent(level) 
               << "Deriving <TRUE," << *h << "> from "
               << rule
               << "  [single undefined head atom]"
               << endl;

    Propagate_Derive_True(*h,I);

#ifndef NDEBUG
    // No other atom may be undefined or must-be-true. Otherwise,
    // the counter values are incorrect.
    for( h++; h != end; h++ )
        assert( ! I.isUndefined(*h) && ! I.isMustBeTrue(*h) );
#endif
    }

//////////////////////////////////////////////////////////////////////////////
// This function is called with a GRULE rule, in which the head is
// false and the negative body is true in the GINTERPRETation I, and
// whose undefined counter of the positive body is exactly one.
//
// @param rule, the rule which contains the single undefined positive
//              body literal
// @param I, the current interpretation
//
inline void MODEL_GENERATOR::Propagate_DeriveSingleUndefinedPosBodyLiteral(
//
    const GRULE&        rule,
          GINTERPRET&   I )
    {
    GCONJUNCTION::const_iterator l=(*rule.getBody()).pos_begin();
    GCONJUNCTION::const_iterator end=(*rule.getBody()).pos_end();

    // Find the first (and single) undefined positive body literal,
    // but do not try to search beyond the end (in case there is no
    // undefined literal left).
    while( ! I.isUndefined(*l) )
        {
        // If the single undefined literal in the positive body is
        // must-be-true, the rule cannot be satisfied, which is an
        // inconsistency.
        // This would have been detected by the dispatching function,
        // unless the must-be-true value has not been propagated yet.
        // If this is the case, the inconsistency is detected when the
        // must-be-true value is propagated.

        l++;
        if( l == end ) return;
        }

    // Derive the negation of the single undefined literal.
    if( TraceLevel >= 2 )
        cdebug << indent(level)
               << "Deriving <FALSE," << *l << "> from "
               << rule
               << "  [single undefined positive body literal]"
               << endl;

    Propagate_Derive_False(*l,I);

#ifndef NDEBUG
    // No other literal may be undefined or its atom
    // must-be-true. Otherwise, the counter values are incorrect.
    for( l++; l != end; l++ )
        assert( ! I.isUndefined(*l) && ! I.isMustBeTrue(*l) );
#endif
    }

//////////////////////////////////////////////////////////////////////////////
// This function is called with a GRULE rule, in which the head is
// false and the positive body is true in the GINTERPRETation I, and
// whose undefined counter of the negative body is exactly one.
//
// @param rule, the rule which contains the single undefined body literal
// @param I, the current interpretation
//
inline void MODEL_GENERATOR::Propagate_DeriveSingleUndefinedNegBodyLiteral(
//
    const GRULE&        rule,
          GINTERPRET&   I )
    {
    GCONJUNCTION::const_iterator l=(*rule.getBody()).neg_begin();
    GCONJUNCTION::const_iterator end=(*rule.getBody()).neg_end();

    // Find the first (and single) undefined negative body literal,
    // but do not try to search beyond the end (in case there is no
    // undefined literal left).
    // Note that if the atom of the single undefined literal in the
    // negative body is must-be-true, the rule can already be
    // regarded as satisfied and nothing should be derived.
    while( ! I.isUndefined(*l) )
        {
        l++;
        if( l == end ) return;
        }

    // Otherwise set the single undefined literal to must-be-true.
    if( TraceLevel >= 2 )
        cdebug << indent(level) << "Deriving <MUST_BE_TRUE," 
               << GATOM(*l) 
               << "> [single undefined negative body literal]"
               << endl;

    Propagate_Derive_MustBeTrue(*l,I);

#ifndef NDEBUG
    // No other literal may be undefined or its atom
    // must-be-true. Otherwise, the counter values are incorrect.
    for( l++; l != end; l++ )
        assert( ! I.isUndefined(*l) && ! I.isMustBeTrue(*l) );
#endif
    }

//////////////////////////////////////////////////////////////////////////////
// This function is called with a GRULE rule, in which the body is
// must-be-true in the GINTERPRETation I, and whose undefined counter
// of the head is exactly one.
//
// @param rule, the rule which contains the single undefined head atom
// @param I, the current interpretation
//
inline void MODEL_GENERATOR::Propagate_MBTDeriveSingleUndefinedHeadLiteral(
//
    const GRULE&      rule,
          GINTERPRET& I )
    {
    GDISJUNCTION::const_iterator h=rule.getHead().begin();
    GDISJUNCTION::const_iterator end=rule.getHead().end();

    // Find the first (and single) undefined head atom, but do not
    // try to search beyond the end (in case there is no undefined
    // head atom left).
    // Note that if the single undefined atom in the head is already
    // must-be-true, nothing has to be derived.
    while( ! I.isUndefined(*h) )
        {
        h++;
        if( h == end ) return;
        }

    // Derive the single undefined atom.
    if( TraceLevel >= 2 )
        cdebug << indent(level) << "Deriving <MUST_BE_TRUE," << *h
               << "> [single undefined head atom and mbt body]"
               << endl;

    Propagate_Derive_MustBeTrue(*h,I);

#ifndef NDEBUG
    // No other atom may be undefined or must-be-true. Otherwise,
    // the counter values are incorrect.
    for( h++; h != end; h++ )
        assert( ! I.isUndefined(*h) && ! I.isMustBeTrue(*h) );
#endif
    }

//////////////////////////////////////////////////////////////////////////////
// This function is called with a GRULE rule, in which the head is
// false and the negative body is true in the GINTERPRETation I, and
// whose undefined counter of the positive body is exactly one more
// than its must-be-true counter.
//
// @param rule, the rule which contains the single undefined body literal
// @param I, the current interpretation
//
inline void MODEL_GENERATOR::Propagate_MBTDeriveSingleUndefinedPosBodyLiteral(
//
    const GRULE&        rule,
          GINTERPRET&   I )
    {
    GCONJUNCTION::const_iterator l=(*rule.getBody()).pos_begin();
    GCONJUNCTION::const_iterator end=(*rule.getBody()).pos_end();

    // Find the first (and single) undefined positive body literal,
    // but do not try to search beyond the end (in case there is no
    // undefined literal left).
    while( ! I.isUndefined(*l) )
        {
        l++;
        if( l == end ) return;
        }

    // Derive the negation of the single undefined literal.
    if( TraceLevel >= 2 )
        cdebug << indent(level)
               << "Deriving <FALSE," << *l << "> from "
               << rule
               << "  [single undefined and non-mbt positive body literal]"
               << endl;

    Propagate_Derive_False(*l,I);

#ifndef NDEBUG
    // No other literal may be undefined. Otherwise, the counter
    // values are incorrect.
    for( l++; l != end; l++ )
        assert( ! I.isUndefined(*l) );
#endif
    }

//////////////////////////////////////////////////////////////////////////////
// This function is called with a GRULE rule, in which the head is
// false and the positive body is must-be-true in the GINTERPRETation
// I, and whose undefined counter of the negative body is exactly one.
//
// @param rule, the rule which contains the single undefined body literal
// @param I, the current interpretation
//
inline void MODEL_GENERATOR::Propagate_MBTDeriveSingleUndefinedNegBodyLiteral(
//
    const GRULE&        rule,
          GINTERPRET&   I )
    {
    GCONJUNCTION::const_iterator l=(*rule.getBody()).neg_begin();
    GCONJUNCTION::const_iterator end=(*rule.getBody()).neg_end();

    // Find the first (and single) undefined negative body literal,
    // but do not try to search beyond the end (in case there is no
    // undefined literal left).
    // Note that if the single undefined literal in the negative body
    // is must-be-true, the rule can already be regarded as
    // satisfied and nothing should be derived.
    while( ! I.isUndefined(*l) )
        {
        l++;
        if( l == end ) return;
        }

    // Set the single undefined literal to must-be-true.
    if( TraceLevel >= 2 )
        cdebug << indent(level) << "Deriving <MUST_BE_TRUE," 
               << GATOM(*l) 
               << "> [single undefined neg. body literal "
                  "and mbt positive body]"
               << endl;

    Propagate_Derive_MustBeTrue(*l,I);

#ifndef NDEBUG
    // No other literal may be undefined or must-be-true. Otherwise,
    // the counter values are incorrect.
    for( l++; l != end; l++ )
        assert( ! I.isUndefined(*l) && ! I.isMustBeTrue(*l) );
#endif
    }

//////////////////////////////////////////////////////////////////////////////
inline void MODEL_GENERATOR::Propagate_HandlePosSingleUndefinedLiteral(
//
// Find the single undefined positive literal in c and falsify it.
//
// @param c, the constraint which contains the single undefined literal
// @param I, the current interpretation
//
    const GCONSTRAINT& c,
          GINTERPRET&  I )
    {
    GCONSTRAINT::const_iterator j;

    for( j=c.pos_begin(); j != c.pos_end(); j++ )
        { 
        if( I.isUndefined(*j) )
            {
            // Inference from constraint

            if( TraceLevel >= 2 )
                cdebug << indent(level)
                       << "Deriving <FALSE," << *j << "> from"
                       << c
                       << "  [inference from constraint]" << endl;

            Propagate_Derive_False(*j,I);
            break;
            }
        }

#ifndef NDEBUG
    if( j != c.pos_end() )
        j++;
    for( ; j != c.pos_end(); j++ )
        assert( ! I.isUndefined(*j) );
#endif
    }

//////////////////////////////////////////////////////////////////////////////
inline void MODEL_GENERATOR::Propagate_HandleNegSingleUndefinedLiteral(
//
// Find the single undefined negative literal in c and set it to
// must-be-true.
//
// @param c, the constraint which contains the single undefined literal
// @param I, the current interpretation
//
    const GCONSTRAINT& c,
          GINTERPRET&  I )
    {
    GCONSTRAINT::const_iterator j;

    for( j=c.neg_begin(); j != c.neg_end(); j++ )
        { 
        if( I.isUndefined(*j) )
            {
            // Inference from constraint

            if( TraceLevel >= 2 )
                cdebug << indent(level) << "Deriving <MUST_BE_TRUE," 
                       << GATOM(*j) << ">  [inference from constraint]" 
                       << endl;

            Propagate_Derive_MustBeTrue(*j,I);
            break;
            }
        }

#ifndef NDEBUG
    if( j != c.neg_end() )
        j++;
    for( ; j != c.neg_end(); j++ )
        assert( ! I.isUndefined(*j) );
#endif
    }

//////////////////////////////////////////////////////////////////////////////
// Derives a literal from a rule.
// This procedure is called if at most one undefined literal is left
// in the rule.
//
// @param rule, the rule from which the literal should be derived
// @param I, the current interpretation
//
// @return false if an inconsistency is detected, true else
//
inline bool MODEL_GENERATOR::Propagate_DeriveFromRule(
    const GRULE&      rule, 
          GINTERPRET& I )
    {
    assert( rule.undefHead + rule.undefPosBody - rule.mbtPosBody 
            + rule.undefNegBody <= 1 );

    // If rule.mbtPosBody == 0, then the body contains at most one
    // undefined literal (which is also not must-be-true). So in this
    // case the non-mustbetrue propagations should be applied.

    // If rule.mbtPosBody > 0, then the body is must-be-true.
    // Therefore we should apply the must-be-true propagations, except
    // for the case when rule.mbtPosBody == 1 and this must-be-true
    // literal is the single undefined literal in the rule, but see
    // below.

    // If rule.mbtPosBody == 1 and also rule.undefPosBody == 1 and all
    // other counters are 0, then in principle this literal should be
    // derived as false because it is a single undefined positive body
    // literal. But since it is must-be-true, deriving it as false is
    // an inconsistency! Also note that if this condition holds, no
    // condition of rule.undefHead == 1, rule.undefPosBody -
    // rule.mbtPosBody == 1, or rule.undefNegBody == 1 holds, so it is
    // safe to handle this in the mustbetrue propagation part instead
    // of in the non-mustbetrue propagation part, although it is no
    // real mustbetrue propagation.

    if( rule.mbtPosBody == 0 )
        {
        // There is exactly one undefined literal in the rule which
        // can be derived.
        if( rule.undefHead == 1 )
            Propagate_DeriveSingleUndefinedHeadLiteral(rule,I);
        else if( rule.undefPosBody == 1 )
            Propagate_DeriveSingleUndefinedPosBodyLiteral(rule,I);
        else if( rule.undefNegBody == 1 )
            Propagate_DeriveSingleUndefinedNegBodyLiteral(rule,I);
        else // all counters are zero
            {
            if( TraceLevel >= 2 )
                cdebug << "   Inconsistency from " << rule << endl;
            return false;
            }
        }
    else
        {
        // There is one or no undefined atom left (if must-be-true in the
        // positive body are considered as true).
        if( rule.undefHead == 1 )
            Propagate_MBTDeriveSingleUndefinedHeadLiteral(rule,I);
        else if( rule.undefPosBody - rule.mbtPosBody == 1 )
            Propagate_MBTDeriveSingleUndefinedPosBodyLiteral(rule,I);
        else if( rule.undefNegBody == 1 )
            Propagate_MBTDeriveSingleUndefinedNegBodyLiteral(rule,I);
        else // Body is must-be-true and head is false -> inconsistency.
            {
            if( TraceLevel >= 2 )
                cdebug << "   Inconsistency from " << rule << endl;
            return false;
            }
        }

    return true;
    }

//////////////////////////////////////////////////////////////////////////////
// Derives a literal in a rule from must-be-true to true.  This
// procedure is called if no undefined literal is left in the rule
// body and there is at least one must-be-true atom in the head.
//
// @param rule, the rule from which the literal should be derived
// @param I, the current interpretation
//
// @return false if an inconsistency is detected, true else
//
inline bool MODEL_GENERATOR::Propagate_DeriveMBTFromHead(
    const GRULE&      rule, 
          GINTERPRET& I )
    {
    assert( rule.mbtHead > 0 
            && rule.undefPosBody == 0 
            && rule.undefNegBody == 0 );

    GDISJUNCTION::const_iterator h = rule.getHead().begin();
    GDISJUNCTION::const_iterator end = rule.getHead().end();

    // Find the first must-be-true atom in the head. If there is none
    // left, do nothing [This means that the must-be-true atoms have
    // already been derived as true, but have not yet been
    // propagated].

    // Also do nothing if we find that one atom is true in the
    // head. This avoids making a redundant inference if two atoms
    // occurring in the same head would be derived in this way without
    // intermediate propagation.

    while( ! I.isMustBeTrue(*h) )
        {
        if( I.isTrue(*h) )
            return true;
        h++;
        if( h == end )
            return true;
        }

    if( getPotentialSupport(*h) == 1 && rule.mbtHead > 1 )
        return false;

    GDISJUNCTION::const_iterator h1 = h;
    for( h1++ ; h1 != end; h1++ )
        {
        if( I.isTrue(*h1) )
            return true;
        }

    if( TraceLevel >= 2 )
        cdebug << indent(level)
               << "Deriving <TRUE," << GATOM(*h) << "> from "
               << rule
               << "  [body is true, " << *h << " is mbt]" 
               << endl;

    Propagate_Derive_TrueFromMustBeTrue(*h,I);

    // The derived atom might be unsupported and is therefore treated
    // as a positive assumption.
    PositiveAssumptions->add(*h);
    PositiveAssumptionsStack.push_back(*h);

    return true;
    }

//////////////////////////////////////////////////////////////////////////////
// Derives a literal from a constraint.
// This procedure is called if at most one undefined literal is left
// in the constraint.
//
// @param constraint, the constraint from which the literal should be derived
// @param I, the current interpretation
//
// @return false if an inconsistency is detected, true else
//
inline bool MODEL_GENERATOR::Propagate_DeriveFromConstraint(
    const GCONSTRAINT& constraint, 
          GINTERPRET&  I )
    {
    assert( constraint.undefPosBody + constraint.undefNegBody <= 1 );

    if( constraint.undefPosBody == 1 )
        Propagate_HandlePosSingleUndefinedLiteral(constraint,I);
    else if( constraint.undefNegBody == 1 )
        Propagate_HandleNegSingleUndefinedLiteral(constraint,I);
    else // all counters are zero
        {
        if( TraceLevel >= 2 )
            cdebug << "  Inconsistency from " << constraint << endl;
        return false;
        }

    return true;
    }

//////////////////////////////////////////////////////////////////////////////
// Checks whether exactly one undefined literal is left in a constraint.
//
// @param constraint, the GCONSTRAINT to be checked
//
static inline bool exactlyOneDerivableLiteral( const GCONSTRAINT& constraint )
    {
    return ( constraint.undefPosBody + constraint.undefNegBody == 1 );
    }

//////////////////////////////////////////////////////////////////////////////
// Inference from weak constraint if both its violation would cause the
// current cost to exceed the current best cost and wc has only one undefined
// literal.
inline bool MODEL_GENERATOR::possibleInferenceFromWeakConstraint(
    const GWEAKCONSTRAINT& wc, 
    const COST& c) const
    {
    assert(bestCost);
    if ( exactlyOneDerivableLiteral(wc) )
        return ( isOptimalCostKnown  &&  c + wc.getWeights() > *bestCost ) ||
               ( !isOptimalCostKnown  &&  c + wc.getWeights() >= *bestCost );
    else
        return false;
    }

//////////////////////////////////////////////////////////////////////////////
// Check whether propagation from a weak constraint wc can be done.
// Inference is performed, if wc has only one undefined literal, and
// the violation of wc would increase the cost of the interpretation
// beyond the BestCost.
//
inline void MODEL_GENERATOR::Propagate_DeriveFromWeakConstraint(
    const GWEAKCONSTRAINT &wc,
    GINTERPRET& I )
//
    {
    if(TraceLevel >=2)
        cdebug << "  Inference from weak constraint " << wc
               << endl;

        #ifndef NDEBUG
        bool dummy =
        #endif

            Propagate_DeriveFromConstraint(wc,I);

        #ifndef NDEBUG
        assert(dummy);
        #endif
    }

//////////////////////////////////////////////////////////////////////////////
// Checks whether there is a literal in the rule which can be derived.
//
// @param rule, the GRULE to be checked
//
static inline bool atMostOneDerivableLiteral( const GRULE& rule )
    {
    return ( rule.undefHead + rule.undefPosBody - rule.mbtPosBody 
             + rule.undefNegBody <= 1 );
    }

//////////////////////////////////////////////////////////////////////////////
// Checks whether at least one must-be-true literal is left in the
// head of the rule and the body is true.
//
// @param rule, the GRULE to be checked
//
static inline bool atLeastOneDerivableMBTHeadAtom( const GRULE& rule )
    {
    return ( rule.mbtHead > 0 
             && rule.undefPosBody == 0 
             && rule.undefNegBody == 0 );
    }

//////////////////////////////////////////////////////////////////////////////
// Checks whether at most one undefined literal is left in the constraint.
//
// @param constraint, the GCONSTRAINT to be checked
//
static inline bool atMostOneDerivableLiteral( const GCONSTRAINT& constraint )
    {
    return ( constraint.undefPosBody + constraint.undefNegBody <= 1 );
    }


//////////////////////////////////////////////////////////////////////////////
// Checks whether no more undefined literal are left in the weak constraint.
//
// @param weak constraint, the GWEAKCONSTRAINT to be checked
//
static inline bool noMoreUndefinedLiteral( const GWEAKCONSTRAINT& wconstraint )
    {
    return ( wconstraint.undefPosBody + wconstraint.undefNegBody == 0 );
    }


//////////////////////////////////////////////////////////////////////////////
// Scan all weakconstraints and check whether inference can be done.
// Inference is done from c, if c has only one undefined literal, and
// the violation of c would increase the cost of the interpretation
// beyond the BestCost.
// Return false if an inconsistency arises, true else.
//
inline bool MODEL_GENERATOR::inferenceFromWeakConstraints( GINTERPRET& I )
//
    {
    for( GWEAKCONSTRAINTS::iterator i = wconstraints.begin();
         i != wconstraints.end();
         i++ )
        if( (*i).satisfied )
            continue;

        // Make inference from this weak constraint, if possible.
        else if( possibleInferenceFromWeakConstraint(*i,I.getCost()) )
            {
            if(TraceLevel >=2)
                cdebug << "  Inference from weak constraint " << *i
                       << endl;

            #ifndef NDEBUG
            bool dummy =
            #endif

                Propagate_DeriveFromConstraint(*i,I);

            #ifndef NDEBUG
            assert(dummy);
            #endif
            }
    return true;
    }

//////////////////////////////////////////////////////////////////////////////
// Derive all the undefined atoms appearing in the aggregate as MBT.
//
inline void MODEL_GENERATOR::Propagate_DeriveMustBeTrueFromAggregate(
    const GATOM &a,
    GINTERPRET  &I)
    {
    assert(a.isAggregate());
    assert(! I.isUndefined(a));

    const AGGREGATESET &aggrSet =
        a.getAggregate().getAggregateFunctionPtr()->getAggregateSet();

    for(AGGREGATESET::const_iterator i = aggrSet.begin();
        i != aggrSet.end(); i++)
        {
        if(I.isUndefined((*i).second))
            {
            if( TraceLevel >= 2 )
                cdebug << indent(level)
                       << "Deriving <MUST_BE_TRUE," << (*i).second
                       << ">  [inference from aggregate]"
                       << endl;
            Propagate_Derive_MustBeTrue((*i).second,I);
            }
        }

    }

//////////////////////////////////////////////////////////////////////////////
// Derive all the undefined atoms appearing in the aggregate as False.
//
inline void MODEL_GENERATOR::Propagate_DeriveFalseFromAggregate(
    const GATOM &a,
    GINTERPRET  &I)
    {
    assert(a.isAggregate());
    assert(! I.isUndefined(a));

    const AGGREGATESET &aggrSet =
        a.getAggregate().getAggregateFunctionPtr()->getAggregateSet();

    for(AGGREGATESET::const_iterator i = aggrSet.begin();
        i != aggrSet.end(); i++)
        {
        if(I.isUndefined((*i).second))
            {
            if( TraceLevel >= 2 )
                cdebug << indent(level)
                       << "Deriving <FALSE," << (*i).second
                       << ">  [inference from aggregate]"
                       << endl;

            Propagate_Derive_False((*i).second,I);
            }
        }
    }

//////////////////////////////////////////////////////////////////////////////
// Perform backward inference from the aggregate count if one of the 
// following conditions holds:
// - the lower guard is always satisfied, and there's only one way to
//   satisfy the upper guard, namely by setting all the undefined atoms
//   appearing in the aggregate to F;
// - the upper guard is always satisfied, and there's only one way to
//   satisfy the lower guard, namely by setting all the undefined atoms
//   appearing in the aggregate to MBT.
inline void MODEL_GENERATOR::PropagateMustBeTrueCount(
    const GATOM &ga,
    GINTERPRET  &I )
    {
    const AGGREGATEATOM &a = ga.getAggregate();

    if( a.getLowerBound() == a.getUpperGuardAsInt() )
        {
        // The lower guard is always satisfied, and there's
        // only one way to satisfy the upper guard.

        Propagate_DeriveFalseFromAggregate(ga,I);
        }
    else if( a.getUpperBound() == a.getLowerGuardAsInt() )
        {
        // The upper guard is always satisfied, and there's
        // only one way to satisfy the lower guard.

        Propagate_DeriveMustBeTrueFromAggregate(ga,I);
        }

    }

//////////////////////////////////////////////////////////////////////////////
// Perform backward inference from the aggregate count if one of the 
// following conditions holds:
// - the lower guard is always satisfied, and there's only one way to
//   violate the upper guard, namely by setting all the undefined atoms
//   appearing in the aggregate to MBT;
// - the upper guard is always satisfied, and there's only one way to
//   violate the lower guard, namely by setting all the undefined atoms
//   appearing in the aggregate to F.
inline void MODEL_GENERATOR::PropagateFalseCount(
    const GATOM &ga,
    GINTERPRET  &I )
    {
    const AGGREGATEATOM &a = ga.getAggregate();

    if( ( a.getLowerGuardAsInt() <= a.getLowerBound() )
       && ( a.getUpperBound() -1 == a.getUpperGuardAsInt() ) )
        {
        // The lower guard is always satisfied, and there's
        // only one way to violate the upper guard.

        Propagate_DeriveMustBeTrueFromAggregate(ga,I);
        }
    else if( ( a.getUpperBound() <= a.getUpperGuardAsInt() )
            && ( a.getLowerGuardAsInt() -1 == a.getLowerBound() ) )
        {
        // The upper guard is always satisfied, and there's
        // only one way to violate the lower guard.

        Propagate_DeriveFalseFromAggregate(ga,I);
        }

    }

///////////////////////////////////////////////////////////////////////////
// Return true if at least one GATOM in the aggregateSet has already
// been derived true w.r.t. the current interpretation I; false otherwise.
//
inline bool MODEL_GENERATOR::atLeastOneTrueAtom(
    const AGGREGATESET &aggregateSet,
    const GINTERPRET   &I ) const
    {
    for( AGGREGATESET::const_iterator i = aggregateSet.begin();
         i != aggregateSet.end();
         i++ )
        if( I.isTrue( (*i).second ) )
            return true;

    return false;
    }

//////////////////////////////////////////////////////////////////////////
// Return true if, w.r.t. the current interpretation I, there are no true 
// GATOMs and only one undefined GATOM in the aggregateSet of the aggregate 
// atom a, having projected term greater than the upper guard;
// false otherwise.
// The aggregateSet has ascending ordering, so we can start from the last
// element of the set to search for elements greater than the upper guard.
//
inline bool MODEL_GENERATOR::onlyOneUndefAndNoTrueAtomsGreaterThanUpperGuard(
    const AGGREGATEATOM &a,
    const GINTERPRET    &I ) const
    {
    bool found = false;

    for( AGGREGATESET::const_reverse_iterator 
         i = a.getAggregateSet().rbegin();
         i != a.getAggregateSet().rend();
         i++ )
        {
        // Check whether the current GATOM in the aggregateSet, with 
        // projected term greater than the upper guard, is true
        // w.r.t. to the current interpretation. 
        if( I.isTrue( (*i).second )
            && *(*i).first.begin() > a.getUpperGuardAsVar() )
            return false;
	
        if( *(*i).first.begin() <= a.getUpperGuardAsVar() )
            break;

        if( I.isUndefined( (*i).second )
            && *(*i).first.begin() > a.getUpperGuardAsVar() )
            {
            if( found )
                return false;
            else
                found = true;
            }
        }

    return found;
    }

//////////////////////////////////////////////////////////////////////////
// Return true if there are no GATOMs in the aggregateSet of the aggregate
// atom a that are true w.r.t. the current interpretation I and whose 
// projected term is in the range [lowerGuard, upperGuard]; false otherwise.
//
inline bool MODEL_GENERATOR::noTrueAtomsBetweenGuards(
    const AGGREGATEATOM &a,
    const GINTERPRET    &I ) const
    {
    for( AGGREGATESET::const_iterator i = a.getAggregateSet().begin();
         i != a.getAggregateSet().end();
         i++ )
        {
        if( I.isTrue( (*i).second )
            && *(*i).first.begin() >= a.getLowerGuardAsVar()
            && *(*i).first.begin() <= a.getUpperGuardAsVar() )
            return false;
        }

    return true;
    }

//////////////////////////////////////////////////////////////////////////
// Return true if there are no GATOMs in the aggregateSet of the aggregate
// atom a that are true or undefined w.r.t. the current interpretation I
// and whose projected term is in the range [lowerGuard, upperGuard];
// false otherwise.
//
inline bool MODEL_GENERATOR::noTrueOrUndefAtomsBetweenGuards(
    const AGGREGATEATOM &a,
    const GINTERPRET    &I ) const
    {
    for( AGGREGATESET::const_iterator i = a.getAggregateSet().begin();
         i != a.getAggregateSet().end();
         i++ )
        {
        if( ( I.isUndefined( (*i).second ) || I.isTrue( (*i).second ) )
            && *(*i).first.begin() >= a.getLowerGuardAsVar()
            && *(*i).first.begin() <= a.getUpperGuardAsVar() )
            return false;
        }

    return true;
    }

//////////////////////////////////////////////////////////////////////////
// Return true if there is only one GATOM in the aggregateSet of the
// aggregate atom a that is true or undefined w.r.t. the current
// interpretation I and that has projected term in the range
// [lowerGuard, upperGuard]; false otherwise.
//
inline bool MODEL_GENERATOR::onlyOneTrueOrUndefAtomBetweenGuards(
    const AGGREGATEATOM &a,
    const GINTERPRET    &I ) const
    {
    bool found = false;

    for( AGGREGATESET::const_iterator i = a.getAggregateSet().begin();
         i != a.getAggregateSet().end();
         i++ )
        {
        if( ( I.isUndefined( (*i).second ) || I.isTrue( (*i).second ) )
            && *(*i).first.begin() >= a.getLowerGuardAsVar()
            && *(*i).first.begin() <= a.getUpperGuardAsVar() )
            if( found )
                return false;
            else
                found = true;
        }

    return found;
    }

///////////////////////////////////////////////////////////////////////////
// Return true if, w.r.t. the current interpretation I, there are no true
// GATOMs and only one undefined GATOM in the aggregateSet of the aggregate
// atom a, having projected term less than the lower guard;
// false otherwise.
//
inline bool MODEL_GENERATOR::onlyOneUndefAndNoTrueAtomsLessThanLowerGuard(
    const AGGREGATEATOM &a,
    const GINTERPRET    &I ) const
    {
    bool found = false;

    for( AGGREGATESET::const_iterator i = a.getAggregateSet().begin();
         i != a.getAggregateSet().end();
         i++ )
        {
        // Check whether the current GATOM in the aggregateSet, with
        // projected term less than the lower guard, is true w.r.t. 
        // to the current interpretation.
        if( I.isTrue( (*i).second )
            && *(*i).first.begin() < a.getLowerGuardAsVar() )
            return false;
	
        if( I.isUndefined( (*i).second )
            && *(*i).first.begin() < a.getLowerGuardAsVar() )
            if( found )
                return false;
            else
                found = true;
        }

    return found;
    }

////////////////////////////////////////////////////////////////////////////
// Perform backward inference from a "max" aggregate if it has been derived 
// must-be-true:
//     - derive false all the GATOMs in the aggregateSet that can violate 
//       the aggregate, i.e. all the GATOMs with projected term greater than
//       the upper guard;
//     - derive must-be-true a GATOM g if it is the only one that can satisfy 
//       the aggregate, i.e. if it is the only one (among the true and 
//       undefined GATOMs in the aggregateSet) whose projected term is in the 
//       range [lowerGuard, upperGuard].
//
inline void MODEL_GENERATOR::PropagateMustBeTrueMax(
    const GATOM      &ga,
          GINTERPRET &I )
    {
    assert(I.isMustBeTrue(ga));

    const AGGREGATEATOM &a = ga.getAggregate();

    // Start from the last element in the ordered set.
    for( AGGREGATESET::const_reverse_iterator
         i = a.getAggregateSet().rbegin();
         i != a.getAggregateSet().rend();
         i++ )
        {
        if( I.isUndefined( (*i).second ) )
            {
            // Derive false the current GATOM in the aggregateSet
            // if its projected term is greater than the upper guard.
            if( *(*i).first.begin() > a.getUpperGuardAsVar() )
                {
                if( TraceLevel >= 2 )
                    cdebug << indent(level)
                           << "Deriving <FALSE," << (*i)
                           << ">  [to avoid that the aggregate is"
                              " unconditionally violated]"
                           << endl;
                Propagate_Derive_False( (*i).second,I );
                }
            // Derive must-be-true the current GATOM in the aggregateSet
            // if it is the only undefined one whose projected term is
            // in the range [lowerGuard, upperGuard].
            else if( *(*i).first.begin() >= a.getLowerGuardAsVar()
                     && onlyOneTrueOrUndefAtomBetweenGuards(a,I) )
                {
                if( TraceLevel >= 2 )
                    cdebug << indent(level)
                           << "Deriving <MUSTBETRUE," << (*i)
                           << ">  [to avoid that the aggregate is"
                              " unconditionally violated]"
                           << endl;
                Propagate_Derive_MustBeTrue( (*i).second,I );

                return;
                }
            }
        }
    }

////////////////////////////////////////////////////////////////////////////
// Perform backward inference from a "max" aggregate if it has been derived
// false.
//
// If no true GATOM occurs in the aggregateSet there are two ways in which
// it's possible to avoid the satisfaction of the aggregate max:
//   - to derive the truth value of one or more GATOMs in the aggregateSet,
//     such that guards are violated;
//   - to derive false all the undefined GATOMs in the aggregateSet, such
//     that the aggregateSet is empty.
// Since there is not a unique way to avoid the satisfaction of the 
// aggregate, no inference can be made in that case.
// On the contrary, if at least one GATOM in the aggregateSet is already 
// true, there are no chances to make the aggregateSet empty and so there 
// is only one way to avoid the satisfaction of the aggregate.
// In this case, backward inference can be performed only if exactly one
// of the following condition holds:
//   1) The lower guard is satisfied:
//        - derive must-be-true a GATOM g if it is the only that can violate 
//          the aggregate, i.e. the only undefined GATOM with projected 
//          term greater than the upper guard;
//   2) The upper guard is satisfied:
//       - derive false all the undefined GATOMs in the aggregateSet that 
//         can satisfy the aggregate, i.e. all the GATOMs with projected 
//         term in the range [lowerGuard, upperGuard];
//       - derive must-be-true a GATOM g if it is the only one that can 
//         violate the aggregate, i.e. if it's the only undefined GATOM
//         with projected term less than the lower guard. 
//
inline void MODEL_GENERATOR::PropagateFalseMax(
    const GATOM      &ga,
          GINTERPRET &I )
    {
    assert(I.isFalse(ga));

    const AGGREGATEATOM &a = ga.getAggregate();

    // If there is no true GATOM in the aggregateSet no inferences can be
    // performed, because there isn't a unique way to avoid the satisfaction 
    // of the aggregate. 
    if( ! atLeastOneTrueAtom( a.getAggregateSet(),I ) )
        return;

    // Check whether the lower guard is already satisfied 
    // and there is only one way to violate the aggregate.
    if( a.getLowerGuardAsInt() <= a.getLowerBound()
        && onlyOneUndefAndNoTrueAtomsGreaterThanUpperGuard(a,I) )
        {
        for( AGGREGATESET::const_reverse_iterator
             i = a.getAggregateSet().rbegin();
             i != a.getAggregateSet().rend();
             i++ )
            {
            if( I.isUndefined( (*i).second ) )
                {
                // Derive must-be-true the current GATOM in the
                // aggregateSet if it is the only undefined one having
                // projected term s greater than the upper guard. 
                if( *(*i).first.begin() > a.getUpperGuardAsVar() )
                    {
                    if( TraceLevel >= 2 )
                        cdebug << indent(level)
                               << "Deriving <MUSTBETRUE," << (*i)
                               << ">  [to avoid that the aggregate is"
                                  " unconditionally satisfied]"
                               << endl;
                    Propagate_Derive_MustBeTrue( (*i).second,I );

                    return;
                    }
                }
            }
        }
   // Check whether the upper guard is already satisfied and there
   // is only one way to violate the lower guard.
   else if( a.getUpperGuardAsInt() >= a.getUpperBound() )
        {
        // Start from the last element in the ordered set.
        for( AGGREGATESET::const_reverse_iterator
             i = a.getAggregateSet().rbegin();
             i != a.getAggregateSet().rend();
             i++ )
            {
            if( I.isUndefined( (*i).second ) )
                {
                // Derive false the current GATOM in the
                // aggregateSet if its projected term is 
                // in the range [lowerGuard, upperGuard].
                if( *(*i).first.begin() >= a.getLowerGuardAsVar()
                    && *(*i).first.begin() <= a.getUpperGuardAsVar() )
                    {
                    if( TraceLevel >= 2 )
                        cdebug << indent(level)
                               << "Deriving <FALSE," << (*i)
                               << ">  [to avoid that the aggregate is"
                                  " unconditionally satisfied]"
                               << endl;
                    Propagate_Derive_False( (*i).second,I );
                    }
                // Derive must-be-true the current GATOM in the aggregateSet
                // if, among all GATOMs with projected term less than the  
                // lower guard, it is the only undefined one and none of 
                // these GATOMs is true.
                else if( a.getLowerGuardAsVar() > *(*i).first.begin()
                         && onlyOneUndefAndNoTrueAtomsLessThanLowerGuard(a,I) )
                    {
                    if( TraceLevel >= 2 )
                        cdebug << indent(level)
                               << "Deriving <MUSTBETRUE," << (*i)
                               << ">  [to avoid that the aggregate is"
                                  " unconditionally satisfied]"
                               << endl;
                    Propagate_Derive_MustBeTrue( (*i).second,I );

                    return;
                    }
                }
            }
        }
    }

////////////////////////////////////////////////////////////////////////////
// Perform backward inferences from a "min" aggregate if it has been derived 
// must-be-true:
//     - derive false all the GATOMs in the aggregateSet that can violate
//       the aggregate, i.e. all the GATOMs with projected term less than 
//       the lower guard;
//     - derive must-be-true a GATOM g if it is the only one that can 
//       satisfy the aggregate, i.e. if it is the only one (among the true 
//       and undefined GATOMs in the aggregateSet) whose projected term is 
//       in the range [lowerGuard, upperGuard].
//       
inline void MODEL_GENERATOR::PropagateMustBeTrueMin(
    const GATOM      &ga,
          GINTERPRET &I )
    {
    assert(I.isMustBeTrue(ga));

    const AGGREGATEATOM &a = ga.getAggregate();

    for( AGGREGATESET::const_iterator
         i = a.getAggregateSet().begin();
         i != a.getAggregateSet().end();
         i++ )
        {
        if( I.isUndefined( (*i).second ) )
            {
            // Derive false the current GATOM in the aggregateSet
            // if its projected term is less than the lower guard.
            if( *(*i).first.begin() < a.getLowerGuardAsVar() )
                {
                if( TraceLevel >= 2 )
                    cdebug << indent(level)
                           << "Deriving <FALSE," << (*i)
                           << ">  [to avoid that the aggregate is"
                              " unconditionally violated]"
                           << endl;
                Propagate_Derive_False( (*i).second,I );
                }
            // Derive must-be-true the current GATOM in the aggregateSet
            // if it is the only undefined one whose projected term 
            // in the range [lowerGuard, upperGuard].
            else if( *(*i).first.begin() <= a.getUpperGuardAsVar() 
                     && onlyOneTrueOrUndefAtomBetweenGuards(a,I) )
                {
                if( TraceLevel >= 2 )
                    cdebug << indent(level)
	                   << "Deriving <MUSTBETRUE," << (*i)
	                   << ">  [to avoid that the aggregate is"
	                      " unconditionally violated]"
	                   << endl;
                    Propagate_Derive_MustBeTrue( (*i).second,I );

                    return;
                }
            // Nothing else can be derived in the aggregateSet.
            else
                return;    
            }
        }
    }

////////////////////////////////////////////////////////////////////////////
// Perform backward inference from a "min" aggregate if it has been derived
// false.
// 
// If no true GATOM occurs in the aggregateSet there are two ways in which
// it's possible to avoid the satisfaction of the aggregate min:
//   - to derive the truth value of one or more GATOMs in the aggregateSet,
//     such that guards are violated;
//   - to derive false all the Undefined GATOMs in the aggregateSet, such
//     that the aggregateSet is empty.
// Since there is not a unique way for satisfaction, no inference can be 
// made in that case.
// On the contrary, if at least one GATOM in the aggregateSet is already 
// true, there are no chances to make the aggregateSet empty and so there 
// is only one way to avoid the satisfaction of the aggregate.
// In this case, backward inference can be performed only if exactly one
// of the following condition holds:
//   1) The lower guard is satisfied:
//     - derive false all the undefined GATOMs in the aggregateSet that can 
//       satisfy the aggregate, i.e. all the GATOMs with projected term in 
//       the range [lowerGuard, upperGuard];
//     - derive must-be-true a GATOM g if it is the only one that can 
//       violate the aggregate, i.e. if it's the only undefined GATOM with 
//       projected term greater than the upper guard and there are no true 
//       GATOMs in the range [lowerGuard, upperGuard];
//   2) The upper guard is satisfied:
//     - derive must-be-true a GATOM g if it is the only one that can 
//       violate the aggregate, i.e. the only undefined GATOM with projected 
//       term less than the lower guard. 
//
inline void MODEL_GENERATOR::PropagateFalseMin(
    const GATOM      &ga,
          GINTERPRET &I )
    {
    assert(I.isFalse(ga));

    const AGGREGATEATOM &a = ga.getAggregate();

    // If there is no true GATOM in the aggregateSet no inferences can be
    // performed, because there isn't a unique way to avoid the satisfaction
    // of the aggregate.
    if( ! atLeastOneTrueAtom( a.getAggregateSet(),I ) )
        return;

    // Check whether the lower guard is already satisfied.
    if( a.getLowerGuardAsInt() <= a.getLowerBound() )
        {
        bool trueAtomsBetweenGuards = false; 
        unsigned undefAtomGreaterThanUpperGuard = 0;
        GATOM undefAtom;

        for( AGGREGATESET::const_iterator
             i = a.getAggregateSet().begin();
             i != a.getAggregateSet().end();
             i++ )
            {
            if( *(*i).first.begin() >= a.getLowerGuardAsVar()
                && *(*i).first.begin() <= a.getUpperGuardAsVar() )
                {
                if( I.isTrue( (*i).second ) ) 
                    trueAtomsBetweenGuards = true;

                // Derive false the current undefined GATOM in the 
                // aggregateSet if its projected term is in the range 
                // [lowerGuard, upperGuard].
                else if( I.isUndefined( (*i).second ) ) 
                    {			
                    if( TraceLevel >= 2 )
                        cdebug << indent(level)
                               << "Deriving <FALSE," << (*i)
                               << ">  [to avoid that the aggregate is"
                                  " unconditionally satisfied]"
                               << endl;
                    Propagate_Derive_False( (*i).second,I );
                    }
                }
            else if( *(*i).first.begin() > a.getUpperGuardAsVar() )
                {
                if( I.isUndefined( (*i).second ) )
                    {
                    if( undefAtomGreaterThanUpperGuard )
                        // Nothing else can be derived in the aggregateSet,
                        // because there is more than one undefined GATOM 
                        // greater than the upperGuard, and by virtue of 
                        // the ascending order of this set, all undefined 
                        // GATOMs with projected term between guards have 
                        // already been derived false.
                        break;
                    else
                        {
                        undefAtomGreaterThanUpperGuard++;
                        undefAtom = (*i).second;                        
                        }
                    }
                }
            else 
                return;
            }
        // Derive must-be-true the only undefined GATOM in the
        // aggregateSet whose projected term is greater than the upper 
        // guard if there aren't true GATOMs with projected term in
        // the range [lowerGuard, upperGuard].
        if( ! trueAtomsBetweenGuards && undefAtomGreaterThanUpperGuard ) 
            {
            if( TraceLevel >= 2 )
                cdebug << indent(level)
                       << "Deriving <MUSTBETRUE," << undefAtom
                       << ">  [to avoid that the aggregate is"
                          " unconditionally satisfied]"
                       << endl;
            Propagate_Derive_MustBeTrue(undefAtom,I);
            }
        }
    // Check whether the upper guard is already satisfied and there
    // is only one way to violate the lower guard.
    else if( a.getUpperGuardAsInt() >= a.getUpperBound()
             && onlyOneUndefAndNoTrueAtomsLessThanLowerGuard(a,I) )
        {
        // Find the only GATOM in the aggregateSet with projected term
        // less than the lower bound and derive it must-be-true.
        for( AGGREGATESET::const_iterator
             i = a.getAggregateSet().begin();
             i != a.getAggregateSet().end();
             i++ )
            {
            if( I.isUndefined( (*i).second ) )
                {
                if( a.getLowerGuardAsVar() > *(*i).first.begin() )
                    {
                    if( TraceLevel >= 2 )
                        cdebug << indent(level)
                               << "Deriving <MUSTBETRUE," << (*i)
                               << ">  [to avoid that the aggregate is"
                                  " unconditionally satisfied]"
                               << endl;
                    Propagate_Derive_MustBeTrue( (*i).second,I );

                    return;
                    }
                }
            }
        }
    }

//////////////////////////////////////////////////////////////////////////////
// Perform backward inference from the aggregate sum:
// - derive MustBeTrue each GATOM ga in the AggregateSet such that its 
//   falsity makes the aggregate unconditionally false, i.e. 
//       upperBound - value(ga) < lowerGuard
// - derive False each GATOM ga in the AggregateSet such that its truth makes 
//   the aggregate unconditionally false, i.e.
//       lowerBound + value(ga) > upperGuard
// where value(ga) is the value of the first free variable of the GATOM ga.
// aggregateSet is sorted according to the (projected) term, so we can scan 
// it in an ordered way (starting from the GATOM with maximum term), and we 
// can derive a truth value for all the GATOMS examined, stopping at the 
// first one that can't  make the aggregate false.
// All the following GATOMS in aggregateSet can't make the aggregate 
// false, either, having a term with smaller value.
inline void MODEL_GENERATOR::PropagateMustBeTrueSum(
    const GATOM &ga,
    GINTERPRET  &I )
    {
    const AGGREGATEATOM &a = ga.getAggregate();

    for( AGGREGATESET::const_reverse_iterator 
        i = a.getAggregateSet().rbegin();
        i != a.getAggregateSet().rend();
        i++ )
        {
        if( I.isUndefined( (*i).second ) )
            {
            if( a.getLowerBound() + ( *(*i).first.begin() ).getInt()
                > a.getUpperGuardAsInt() )
                {
                if( TraceLevel >= 2 )
                    cdebug << indent(level)
                           << "Deriving <FALSE," << (*i)
                           << ">  [to avoid that the aggregate is"
                              " unconditionally false]"
                           << endl;
                Propagate_Derive_False( (*i).second,I );
                }
            else if( a.getUpperBound() - ( *(*i).first.begin() ).getInt()
                     < a.getLowerGuardAsInt() )
                {
                if( TraceLevel >= 2 )
                    cdebug << indent(level)
                           << "Deriving <MUSTBETRUE," << (*i)
                           << ">  [to avoid that the aggregate is"
                              " unconditionally false]"
                           << endl;

                Propagate_Derive_MustBeTrue( (*i).second,I );
                }
            else
                // The current term can't make the aggregate false.
                return;
            }
        }
    }

//////////////////////////////////////////////////////////////////////////////
// Perform backward inference from the aggregate sum:
// - derive MustBeTrue each GATOM ga in the AggregateSet such that if a is 
//   False then the aggregate becomes unconditionally satisfied, i.e.
//       lowerGuard <= lowerBound && upperBound - value(ga) <= upperGuard
// - derive False each GATOM a in the aggregateSet such that if a is True
//   then the aggregate becomes unconditionally satisfied, i.e.
//       upperBound <= upperGuard && lowerBound + value(ga) >= lowerGuard
// where value(ga) is the value of the first free variable of the GATOM ga.
// aggregateSet is sorted according to the (projected) term, so we can scan
// it in an ordered way (starting from the GATOM with maximum term), and we 
// can derive a truth value for all the GATOMS examined, stopping at the    
// first one that can't  make the aggregate satisfied.
// All the following GATOMS in aggregateSet can't make the aggregate 
// satisfied, either ,having a term with smaller value.
inline void MODEL_GENERATOR::PropagateFalseSum(
    const GATOM &ga,
    GINTERPRET  &I )
    {
    const AGGREGATEATOM &a = ga.getAggregate();

    for( AGGREGATESET::const_reverse_iterator 
        i = a.getAggregateSet().rbegin();
        i != a.getAggregateSet().rend();
        i++ )
        {
        if( I.isUndefined( (*i).second ) )
            {
            if( ( a.getUpperGuardAsInt() >= a.getUpperBound() )
               && ( a.getLowerBound() + ( *(*i).first.begin() ).getInt()
                    >= a.getLowerGuardAsInt() ) )
                {
                if( TraceLevel >= 2 )
                    cdebug << indent(level)
                           << "Deriving <FALSE," << (*i)
                           << ">  [to avoid that the aggregate is"
                              " unconditionally satisfied]"
                           << endl;
                Propagate_Derive_False( (*i).second,I );
                }
            else if( ( a.getLowerGuardAsInt() <= a.getLowerBound() )
                    && ( a.getUpperBound() - ( *(*i).first.begin() ).getInt()
                         <= a.getUpperGuardAsInt() ) )
                {
                if( TraceLevel >= 2 )
                    cdebug << indent(level)
                           << "Deriving <MUSTBETRUE," << (*i)
                           << ">  [to avoid that the aggregate is"
                              " unconditionally satisfied]"
                           << endl;
                Propagate_Derive_MustBeTrue( (*i).second,I );
                }
            else
                // The current term can't make the aggregate satisfied.
                return;
            }
        }
    }

//////////////////////////////////////////////////////////////////////////////
// Perform backward inference from the aggregate times:
// - derive MustBeTrue each GATOM ga in the AggregateSet such that its
//   falsity makes the aggregate unconditionally false, i.e.
//       upperBound / value(ga) < lowerGuard
//   with value(ga) > 0;
// - derive False each GATOM ga in the AggregateSet such that its truth makes
//   the aggregate unconditionally false, i.e.
//       lowerBound * value(ga) > upperGuard
// where value(ga) is the value of the first free variable of the GATOM ga.
// aggregateSet is sorted according to the (projected) term, so we can scan
// it in an ordered way (starting from the GATOM with maximum term), and we 
// can derive a truth value for all the GATOMS examined, stopping at the    
// first one that can't  make the aggregate false.
// All the following GATOMS in aggregateSet can't make the aggregate 
// false, either, having a term with smaller value.
inline void MODEL_GENERATOR::PropagateMustBeTrueTimes(
    const GATOM &ga,
    GINTERPRET  &I )
    {
    const AGGREGATEATOM &a = ga.getAggregate();

    for( AGGREGATESET::const_reverse_iterator
        i = a.getAggregateSet().rbegin();
        i != a.getAggregateSet().rend();
        i++ )
        {
        if( I.isUndefined( (*i).second ) )
            {
            if( a.getLowerBound() * ( *(*i).first.begin() ).getInt()
                > a.getUpperGuardAsInt() )
                {
                if( TraceLevel >= 2 )
                    cdebug << indent(level)
                           << "Deriving <FALSE," << (*i)
                           << ">  [to avoid that the aggregate is"
                              " unconditionally false]"
                           << endl;
                Propagate_Derive_False( (*i).second,I );
                }
            else if( ( *(*i).first.begin() ).getInt() > 0 
                    && a.getUpperBound() / ( *(*i).first.begin() ).getInt()
                       < a.getLowerGuardAsInt() )
                {
                if( TraceLevel >= 2 )
                    cdebug << indent(level)
                           << "Deriving <MUSTBETRUE," << (*i)
                           << ">  [to avoid that the aggregate is"
                              " unconditionally false]"
                           << endl;

                Propagate_Derive_MustBeTrue( (*i).second,I );
                }
            else
                // The current term can't make the aggregate false.
                return;
            }
        }
    }

//////////////////////////////////////////////////////////////////////////////
// Perform backward inference from the aggregate times:
// - derive MustBeTrue each GATOM ga in the AggregateSet such that if a is
//   False then the aggregate becomes unconditionally satisfied, i.e.
//       lowerGuard <= lowerBound && upperBound / value(ga) <= upperGuard
//   with value(ga) > 0;
// - derive False each GATOM a in the aggregateSet such that if a is True
//   then the aggregate becomes unconditionally satisfied, i.e.
//       upperBound <= upperGuard && lowerBound * value(ga) >= lowerGuard
// where value(ga) is the value of the first free variable of the GATOM ga.
// aggregateSet is sorted according to the (projected) term, so we can scan
// it in an ordered way (starting from the GATOM with maximum term), and we 
// can derive a truth value for all the GATOMS examined, stopping at the    
// first one that can't  make the aggregate satisfied.
// All the following GATOMS in aggregateSet can't make the aggregate 
// satisfied, either, having a term with smaller value.
inline void MODEL_GENERATOR::PropagateFalseTimes(
    const GATOM &ga,
    GINTERPRET  &I )
    {
    const AGGREGATEATOM &a = ga.getAggregate();

    for( AGGREGATESET::const_reverse_iterator
        i = a.getAggregateSet().rbegin();
        i != a.getAggregateSet().rend();
        i++ )
        {
        if( I.isUndefined( (*i).second ) )
            {
            if( ( a.getUpperGuardAsInt() >= a.getUpperBound() )
               && ( a.getLowerBound() * ( *(*i).first.begin() ).getInt()
                    >= a.getLowerGuardAsInt() ) )
                {
                if( TraceLevel >= 2 )
                    cdebug << indent(level)
                           << "Deriving <FALSE," << (*i)
                           << ">  [to avoid that the aggregate is"
                              " unconditionally satisfied]"
                           << endl;
                Propagate_Derive_False( (*i).second,I );
                }
            else if( ( *(*i).first.begin() ).getInt() > 0 
                    && ( a.getLowerGuardAsInt() <= a.getLowerBound() )
                    && ( a.getUpperBound() / ( *(*i).first.begin() ).getInt()
                         <= a.getUpperGuardAsInt() ) )
                {
                if( TraceLevel >= 2 )
                    cdebug << indent(level)
                           << "Deriving <MUSTBETRUE," << (*i)
                           << ">  [to avoid that the aggregate is"
                              " unconditionally satisfied]"
                           << endl;
                Propagate_Derive_MustBeTrue( (*i).second,I );
                }
            else
                // The current term can't make the aggregate satisfied.
                return;
            }
        }
    }

//////////////////////////////////////////////////////////////////////////////
// Perform backward inference from the aggregate GATOM a.
inline void MODEL_GENERATOR::PropagateMustBeTrueAggregate(
    const GATOM &a,
    GINTERPRET  &I )
    {
    assert(a.isAggregate());
    assert(I.isMustBeTrue(a));
   
    switch ( a.getAggregate().getAggregateType().getType() )
        {
        case AGGREGATE_TYPE::Count:
            PropagateMustBeTrueCount(a,I);
            break;
        case AGGREGATE_TYPE::Max:
            PropagateMustBeTrueMax(a,I);
            break;
        case AGGREGATE_TYPE::Min:
            break;
        case AGGREGATE_TYPE::Sum:
            PropagateMustBeTrueSum(a,I);
            break;
        case AGGREGATE_TYPE::Times:
            PropagateMustBeTrueTimes(a,I);
            break;
        default:
            assert(0);
        }
    }

//////////////////////////////////////////////////////////////////////////////
// Perform backward inference from the aggregate GATOM a.
inline void MODEL_GENERATOR::PropagateFalseAggregate(
    const GATOM &a,
    GINTERPRET  &I )
    {
    assert(a.isAggregate());
    assert(I.isFalse(a));
    
    switch ( a.getAggregate().getAggregateType().getType() )
        {
        case AGGREGATE_TYPE::Count:
            PropagateFalseCount(a,I);
            break;
        case AGGREGATE_TYPE::Max:
            PropagateFalseMax(a,I);
            break;
        case AGGREGATE_TYPE::Min:
            break;
        case AGGREGATE_TYPE::Sum:
            PropagateFalseSum(a,I);
            break;
        case AGGREGATE_TYPE::Times:
            PropagateFalseTimes(a,I);
            break;
        default:
            assert(0);
        }
    }

//////////////////////////////////////////////////////////////////////////////
// Each time one of the two bounds of an aggregateFunction with index i is 
// updated, scan all the aggregates which are based on this aggregateFunction
// and make possible derivations.
//
// Returns: false if an inconsistency has arisen, true else.
//
inline bool MODEL_GENERATOR::PropagateAggregateBoundUpdate(
//
    const unsigned i,
          GINTERPRET &I )
    { 
    for( vector<GATOM>::const_iterator
         j = occursInGatom[i].begin();
         j != occursInGatom[i].end();
         j++ )
        {
        if ( (*j).getAggregate().isViolated() )
            {
            assert( ! I.isTrue(*j) );
            if( I.isMustBeTrue(*j) )
                {
                return false;
                }
            else if( I.isFalse(*j) )
                {
                // Do nothing.
                }
            else
                {
                if( TraceLevel >= 2 )
                   cdebug << indent(level)
                          << "Deriving <FALSE," << (*j)
                          << ">  [aggregate violated]"
                          << endl;

                Propagate_Derive_False(*j,I);
                }
            }
        else if ( (*j).getAggregate().isSatisfied() )
            {
            if( I.isTrue(*j) )
                {
                // Do nothing.
                }
            else if( I.isFalse(*j) )
                {
                return false;
                }
            else
                {
                if( TraceLevel >= 2 )
                    cdebug << indent(level)
                           << "Deriving <TRUE," << (*j)
                           << ">  [aggregate satisfied]"
                           << endl;

                Propagate_Derive_True(*j,I);
                }
            }
        if( I.isMustBeTrue(*j) )
            PropagateMustBeTrueAggregate(*j,I);
        else if ( I.isFalse(*j) )
            PropagateFalseAggregate(*j,I);
        } 

    return true;
    }

//////////////////////////////////////////////////////////////////////////
// Find the new maximum values among the projected terms of the undefined
// and true/mbt auxiliary atoms appearing in an aggregateSet.
// Whenever there is no undefined or true/mbt value, the respective
// value is INT_MIN.
//
void MODEL_GENERATOR::findMaxUndefTrueInAggregateSet(
    AGGREGATEATOM &a,
    const GINTERPRET &I) const
    {
    // If no undefined (resp., true/mbt) atom exists in a, the respective
    // value will be INT_MIN.

    // The maximum among true atoms is monotonically
    // increasing. Moreover, it may be due to an atom which became
    // true before the creation of the model checker, and which is
    // therefore not present any longer. Therefore we should only look
    // for an atom with more than the current value.
    // The maximum among undefined atoms is, however decreasing and
    // may also go back to INT_MIN. It is therefore necessary to start
    // from the lowest possible value (INT_MIN).
    a.setMaxUndef(INT_MIN);

    bool foundMaxUndef = false;
    bool foundMaxTrue = false;

    assert( a.getAggregateType().getType() == AGGREGATE_TYPE::Max );

    int& maxUndef = (static_cast<MAXFUNCTION*>(a.getAggregateFunctionPtr())
                     ->getMaxUndefAsReference());
    int& maxTrue = (static_cast<MAXFUNCTION*>(a.getAggregateFunctionPtr())
                    ->getMaxTrueAsReference());

    // The aggregateSet has ascending ordering, so the last
    // undefined auxiliary atom provides the new maxUndef.
    for( AGGREGATESET::const_reverse_iterator
        j = a.getAggregateSet().rbegin();
        j != a.getAggregateSet().rend();
        j++ )
        {
        if( ! foundMaxUndef && I.isUndefined( (*j).second ) )
            {
            // New undefined maximum found.
            assert( ( *(*j).first.begin() ).isInt() );
            // The new value is definitely greater (better) or the
            // same as the current maximum. In the worst case we make
            // a useless assignment if they are equal.
            assert( maxUndef <= ( *(*j).first.begin() ).getInt() );
            maxUndef = ( *(*j).first.begin() ).getInt();
            foundMaxUndef = true;
            // If also a new true/mbt maximum has been found already,
            // we can stop searching.
            if( foundMaxTrue )
                break;
            }
        else if( ! foundMaxTrue 
                 && ( I.isTrue( (*j).second ) 
                      || I.isMustBeTrue( (*j).second ) ) )
            {
            // New true/mbt maximum found.
            assert( ( *(*j).first.begin() ).isInt() );
            // The new value might be smaller (worse) than the current
            // maximum. This can happen if the current maximum is due
            // to an atom which is no longer present (became true
            // early and was optimized away). Since we make a
            // comparison, make the assignment also only if the new
            // maximum is really an improvement.
            if( maxTrue < ( *(*j).first.begin() ).getInt() )
                maxTrue = ( *(*j).first.begin() ).getInt();
            foundMaxTrue = true;
            // If also a new undefined maximum has been found already,
            // we can stop searching.
            if( foundMaxUndef )
                break;
            }
        }
    }

//////////////////////////////////////////////////////////////////////////
// Find the new minimum values among the projected terms of the undefined
// and true/mbt auxiliary atoms appearing in an aggregateSet.
// Whenever there is no undefined or true/mbt value, the respective
// value is INT_MAX.
//
void MODEL_GENERATOR::findMinUndefTrueInAggregateSet(
    AGGREGATEATOM &a,
    const GINTERPRET &I) const
    {

    assert( a.getAggregateType().getType() == AGGREGATE_TYPE::Min );

    // If no undefined (true/mbt) atom exists in a, the respective
    // value will be INT_MAX.

    // The minimum among true atoms is monotonically
    // decreasing. Moreover, it may be due to an atom which became
    // true before the creation of the model checker, and which is
    // therefore not present any longer. Therefore we should only look
    // for an atom with less than the current value.
    // The minimum among undefined atoms is, however increasing and
    // may also go to INT_MAX. It is therefore necessary to start from
    // the largest possible value (INT_MAX).
    a.setMinUndef(INT_MAX);

    bool foundMinUndef = false;
    bool foundMinTrue = false;

    int& minUndef = (static_cast<MINFUNCTION*>(a.getAggregateFunctionPtr())
                     ->getMinUndefAsReference());
    int& minTrue = (static_cast<MINFUNCTION*>(a.getAggregateFunctionPtr())
                    ->getMinTrueAsReference());

    // The aggregateSet has ascending ordering, so the first
    // undefined (true/mbt) auxiliary atom provides a new minimum.
    for( AGGREGATESET::const_iterator
         j = a.getAggregateSet().begin();
         j != a.getAggregateSet().end();
         j++ )
        {
        if( ! foundMinUndef && I.isUndefined( (*j).second ) )
            {
            // New undefined minimum found.
            assert( ( *(*j).first.begin() ).isInt() );
            // The new value is definitely smaller (better) or the
            // same as the current minimum. In the worst case we make
            // a useless assignment if they are equal.
            assert( minUndef >= ( *(*j).first.begin() ).getInt() );
            minUndef = ( *(*j).first.begin() ).getInt();
            foundMinUndef = true;
            // If also a new true/mbt minimum has been found already,
            // we can stop searching.
            if( foundMinTrue )
                break;
            }
        else if( ! foundMinTrue
                 && ( I.isTrue( (*j).second )
                      || I.isMustBeTrue( (*j).second ) ) )
            {
            // New true/mbt minimum found.
            assert( ( *(*j).first.begin() ).isInt() );
            // The new value might be larger (worse) than the current
            // minimum. This can happen if the current minimum is due
            // to an atom which is no longer present (became true
            // early and was optimized away). Since we make a
            // comparison, make the assignment also only if the new
            // minimum is really an improvement.
            if( minTrue > ( *(*j).first.begin() ).getInt() )
                minTrue = ( *(*j).first.begin() ).getInt();
            foundMinTrue = true;
            // If also a new undefined minimum has been found already,
            // we can stop searching.
            if( foundMinUndef )
                break;
            }
        }
    }

//////////////////////////////////////////////////////////////////////////
// If a new truth value has been found for a GATOM appearing in the  
// aggregateSet of a min (max) aggregate, and its projected term is equal to 
// the current minimum (maximum) among the projected terms of the undefined 
// GATOMs of the same aggregateSet, search for the new minimum (maximum).
//
void MODEL_GENERATOR::updateAggregate(
    const pair<unsigned,TERMS> &p,
    const GINTERPRET &I )
    {
    AGGREGATEATOM &a = ( *occursInGatom[p.first].begin() ).getAggregate();

    switch( a.getAggregateType().getType() )
        {
        case AGGREGATE_TYPE::Count:
            break;
        case AGGREGATE_TYPE::Max:
            if( ( *p.second.begin() ).getInt() == a.getMaxUndef() )
                {
                findMaxUndefTrueInAggregateSet(a,I);
                }
            break;
        case AGGREGATE_TYPE::Min:
            if( ( *p.second.begin() ).getInt() == a.getMinUndef() )
                {
                findMinUndefTrueInAggregateSet(a,I);
                }
            break;
        case AGGREGATE_TYPE::Sum:
            break;
        case AGGREGATE_TYPE::Times:
            break;
        default:
            assert(0);
        }
    }

// Mark aggregate counters, pushing them in the stack INTEGERSTACK.
inline void MODEL_GENERATOR::pushAggregateCounters(
    AGGREGATEFUNCTION &a )
    {
    switch( a.getAggregateType().getType() )
        {
        case AGGREGATE_TYPE::Count:
            break;
        case AGGREGATE_TYPE::Max:
            // Mark the current value of maxUndef.
            IntegerStack.push( level, &( ( static_cast<MAXFUNCTION*>(&a) )->
                               getMaxUndefAsReference() ) );
            // Mark the current value of maxTrue.
            IntegerStack.push( level, &( ( static_cast<MAXFUNCTION*>(&a) )->
                               getMaxTrueAsReference() ) );
            break;
        case AGGREGATE_TYPE::Min:
            // Mark the current value of minUndef.
            IntegerStack.push( level, &( ( static_cast<MINFUNCTION*>(&a) )->
                               getMinUndefAsReference() ) );
            // Mark the current value of minTrue.
            IntegerStack.push( level, &( ( static_cast<MINFUNCTION*>(&a) )->
                               getMinTrueAsReference() ) );
            break;
        case AGGREGATE_TYPE::Sum:
            break;
        case AGGREGATE_TYPE::Times:
            // Mark currentValue.
            IntegerStack.push( level, &( ( static_cast<TIMESFUNCTION*>(&a) )->
                               getCurrentValueAsReference() ) );
            break;
        default:
            assert(0);
        }
    }

//////////////////////////////////////////////////////////////////////////////
// Scan all the aggregates in which the GATOM ga appears inside an 
// aggregateSet, update bounds and make possible derivations.
//
// Returns: false if an inconsistency has arisen, true else.
inline bool MODEL_GENERATOR::PropagateInAggregate(
//
    const GATOM      &ga,
          GINTERPRET &I,
          bool        isTrue )
    {
    for( vector< pair<unsigned,TERMS> >::const_iterator
        i = occursInAggregateFunction[ga.getIndex()].begin();
        i != occursInAggregateFunction[ga.getIndex()].end();
        i++ )
        {
        // The aggregate function must appear at least in one GATOM.
        assert ( occursInGatom[(*i).first].begin() 
                 != occursInGatom[(*i).first].end() );

        AGGREGATEATOM &a = occursInGatom[(*i).first].begin()->getAggregate();

        // If a true GATOM has been found in the aggregateSet, mark the 
        // current value of the lower bound ...
        if( isTrue )
            IntegerStack.push( level, &( a.getAggregateFunctionPtr()
                               ->getLowerBoundAsReference() ) );
        // ... otherwise, if a false GATOM has been found, mark the current
        // value of the upperBound.
        else
            IntegerStack.push( level, &( a.getAggregateFunctionPtr()
                               ->getUpperBoundAsReference() ) );

        // Mark the current value of aggregate counters.
        pushAggregateCounters( *a.getAggregateFunctionPtr() );

        updateAggregate((*i),I);

        // Update bounds with the value assumed by the first free variable 
        // in the GATOM at hand. Note that an AGGREGATEFUNCTION can occur 
        // in several AGGREGATEATOMs, and the bounds are only updated for 
        // the first such occurrence.
        a.updateBounds( *(*i).second.begin(), isTrue );

        if( ! PropagateAggregateBoundUpdate( (*i).first, I ) )
            return false;
        }

    return true;    
    }

//////////////////////////////////////////////////////////////////////////////
bool MODEL_GENERATOR::PropagatePositiveFromUndefined(
//
// Returns: false if an inconsistency has arisen, true else.

    const GATOM&      a,
          GINTERPRET& I )
    {
    // Determine satisfaction of rules and constraints.

    // Perform inferences in aggregates.
    if ( ! PropagateInAggregate(a,I,true) )
        return false;

    // If the atom of the propagated positive literal occurs in the
    // negative body of a rule, the body becomes false and therefore
    // the rule becomes satisfied.
    for( GRULEPTRS::const_iterator i = negBodyRules_begin(a);
         i != negBodyRules_end(a);
         i++ )
        {
        if( ! Propagate_RuleBecomesSatisfiedByBody(**i,I) )
            return false;
        }

    // If the atom of the propagated positive literal occurs in the
    // head of a rule, the head becomes true and therefore the rule
    // becomes satisfied.
    for( GRULEPTRS::const_iterator i = headRules_begin(a);
         i != headRules_end(a);
         i++ )
        {
        if( ! Propagate_RuleBecomesSatisfiedByHead(**i,a,I) )
            return false;
        }

    // If the atom of the propagated positive literal occurs in the
    // negative part of a constraint, it becomes satisfied.
    for( GCONSTRAINTPTRS::const_iterator i = negConstraints_begin(a);
         i != negConstraints_end(a);
         i++ )
        {
        if( (**i).satisfied ) continue;

        PUSH( (**i).satisfied );
        (**i).satisfied=true;

        // A constraint (and thus also a partially violated clause)
        // has been eliminated.
        HC.ClauseEliminated++;
        HC.PVClauseEliminated++;

#ifdef WITH_PVCLEVELS
        switch( (**i).undefPosBody + (**i).undefNegBody )
            {
            case 0:
                assert(0);
            case 2:
                HC.PVClauseEliminatedLevel2++;
                break;
            case 3:
                HC.PVClauseEliminatedLevel3++;
                break;
            default:
                // nothing
                break;
            }
#endif

        }
    // If the atom of the propagated positive literal occurs in the
    // negative part of a weak constraint, it becomes satisfied.
    for( GWEAKCONSTRAINTPTRS::const_iterator i = negWeakConstraints_begin(a);
         i != negWeakConstraints_end(a);
         i++ )
        {
        if( (**i).satisfied ) continue;

        PUSH( (**i).satisfied );
        (**i).satisfied=true;

        // We have eliminated a weak constraint.
        if(HC.EliminatedWCCost)
            {
            // Calculate the level and the increased value
            unsigned int costLevel = (**i).getLevel();
            unsigned int incrValue =  (*HC.EliminatedWCCost).getCost(costLevel)
                                      + (**i).getWeight();
            // Store it in the proper level.
            (*HC.EliminatedWCCost).setCost(costLevel, incrValue);
            }
        }

    // Adjust counters and make possible inferences.

    if( getPotentialSupport(a) == 1 )
        if( ! Propagate_OnePotentiallySupportingRule(a,I) )
            return false;

    // Decrement the undefPosBody counters of those unsatisfied rules
    // in which the atom of the propagated literal occurs in the
    // positive body, and make inferences, if possible
    for( GRULEPTRS::const_iterator i = posBodyRules_begin(a);
         i != posBodyRules_end(a);
         i++ )
        {
        if( (**i).satisfied ) continue;

        assert( (**i).undefPosBody > 0 );
        PUSH( (**i).undefPosBody );
        (**i).undefPosBody--;

        if( atMostOneDerivableLiteral(**i) )
            {
            if( ! Propagate_DeriveFromRule(**i,I) )
                // An inconsistency has been detected.
                return false;
            }
        else if( atLeastOneDerivableMBTHeadAtom(**i) )
            {
            if( ! Propagate_DeriveMBTFromHead(**i,I) )
                // An inconsistency has been detected.
                return false;
            }

        // A partially violated clause could have been created (if the
        // body became true/mbt).
        if( (**i).partiallyViolated() )
            {
            // If the clause is violated by its body, a new partially
            // violated clause has just been created.
            if( (**i).partiallyViolatedByBody() )
                {
                HC.PVClauseIntroduced++;

#ifdef WITH_PVCLEVELS
                switch( (**i).partialViolationLevel() )
                    {
                    case 0:
                        assert(0);
                    case 2:
                        HC.PVClauseIntroducedLevel2++;
                        break;
                    case 3:
                        HC.PVClauseIntroducedLevel3++;
                        break;
                    default:
                        // nothing
                        break;
                    }
#endif

                }

#ifdef WITH_PVCLEVELS
            else
                {
                // The rule is partially violated by its head. The
                // number of undefined atoms in the body has been
                // reduced. A partially violated clause of a
                // particular level could have been created.
                switch( (**i).partialViolationLevel() )
                    {
                    case 0:
                        assert(0);
                    case 1:
                        // A partially violated clause of level 2 has
                        // been eliminated.
                        HC.PVClauseEliminatedLevel2++;
                        break;
                    case 2:
                        // A partially violated clause of level 3 has
                        // been eliminated and one of level 2 has been
                        // introduced.
                        HC.PVClauseIntroducedLevel2++;
                        HC.PVClauseEliminatedLevel3++;
                        break;
                    case 3:
                        // A partially violated clause of level 3 has
                        // been introduced.
                        HC.PVClauseIntroducedLevel3++;
                        break;
                    default:
                        // nothing
                        break;
                    }
                }
#endif

            }
        }

    // Decrement the undefPosBody counters of those unsatisfied
    // constraints in which the atom of the propagated literal occurs
    // in the positive body, and make inferences, if possible.
    for( GCONSTRAINTPTRS::const_iterator i = posConstraints_begin(a);
         i != posConstraints_end(a);
         i++ )
        {
        if( (**i).satisfied ) continue;

        assert( (**i).undefPosBody > 0 );
        PUSH( (**i).undefPosBody );
        (**i).undefPosBody--;

        if( atMostOneDerivableLiteral(**i) )
            if( ! Propagate_DeriveFromConstraint(**i,I) )
                // An inconsistency has been detected.
                return false;

#ifdef WITH_PVCLEVELS
        switch( (**i).undefPosBody + (**i).undefNegBody )
            {
            case 0:
                assert(0);
            case 1:
                // A partially violated clause of level 2 has been
                // eliminated.
                HC.PVClauseEliminatedLevel2++;
                break;
            case 2:
                // A partially violated clause of level 3 has been
                // eliminated and one of level 2 has been introduced.
                HC.PVClauseEliminatedLevel3++;
                HC.PVClauseIntroducedLevel2++;
                break;
            case 3:
                // A partially violated clause of level 3 has been
                // introduced.
                HC.PVClauseIntroducedLevel3++;
                break;
            default:
                // nothing
                break;
            }
#endif

        }

    // Decrement the undefPosBody counters of those unsatisfied weak
    // constraints in which the atom of the propagated literal occurs
    // in the positive body.
    // Try to make some inference from weak constraint, then check
    // whether the weak constraint is violated.
    for( GWEAKCONSTRAINTPTRS::const_iterator i = posWeakConstraints_begin(a);
         i != posWeakConstraints_end(a);
         i++ )
        {
        if( (**i).satisfied ) continue;

        assert( (**i).undefPosBody > 0 );
        PUSH( (**i).undefPosBody );
        (**i).undefPosBody--;
        
        // Try to derive something from this weak constraint.
        if( possibleInferenceFromWeakConstraint(**i,I.getCost()) )
            Propagate_DeriveFromWeakConstraint(**i,I);

        else if( noMoreUndefinedLiteral(**i) )
            // A violation has been detected.
            {
            I.increaseCost((**i).getWeights());
            if((isOptimalCostKnown && (I.getCost() > *bestCost)) ||
               (!isOptimalCostKnown && (I.getCost() >= *bestCost))) 
                {            
                // Throw inconsistency as we won't find a better model.
                if(TraceLevel >=3)
                    {
                    cdebug << indent(level)
                           << "Violated WeakConstraint " << (**i) 
                           << " throws inconsistency."
                           << endl << indent(level)
                           << "Current cost of the inconsistent"
                              " interpretation ([Weight:Level]): "
                           << endl;
                    I.printCost(cdebug);
                    }
                return false;
                }
            increasedCost = true;
            }
        }
    return true;
    }

//////////////////////////////////////////////////////////////////////////////
bool MODEL_GENERATOR::PropagatePositiveFromMustBeTrue(
//
// Returns: false if an inconsistency has arisen, true else.

    const GATOM&      a,
          GINTERPRET& I )
    {
    // Determine satisfaction of rules (constraints, and aggregates
    // have already been dealt with when MustBeTrue was propagated
    // since must-be-true atoms are treated as true atoms in constraints).

#ifndef NDEBUG
    // If the atom of the propagated positive literal occurs in the
    // negative body of a rule, the body becomes false and therefore
    // the rule becomes satisfied.
    for( GRULEPTRS::const_iterator i = negBodyRules_begin(a);
         i != negBodyRules_end(a);
         i++ )
        {
        assert( Propagate_RuleBecomesSatisfiedByBody(**i,I) );
        }
#endif

    // If the atom of the propagated positive literal occurs in the
    // head of a rule, the head becomes true and therefore the rule
    // becomes satisfied.
    for( GRULEPTRS::const_iterator i = headRules_begin(a);
         i != headRules_end(a);
         i++ )
        {
        if( ! Propagate_RuleBecomesSatisfiedByHead(**i,a,I) )
            return false;
        }

    // Adjust counters and make possible inferences.

    if( getPotentialSupport(a) == 1 )
        if( ! Propagate_OnePotentiallySupportingRule(a,I) )
            return false;

    // Decrement the undefPosBody counters of those unsatisfied rules
    // in which the atom of the propagated literal occurs in the
    // positive body, and make inferences, if possible
    for( GRULEPTRS::const_iterator i = posBodyRules_begin(a);
         i != posBodyRules_end(a);
         i++ )
        {
        if( (**i).satisfied ) continue;

        assert( (**i).undefPosBody > 0 );
        PUSH( (**i).undefPosBody );
        (**i).undefPosBody--;

        assert( (**i).mbtPosBody > 0 );
        PUSH( (**i).mbtPosBody );
        (**i).mbtPosBody--;

        if( atMostOneDerivableLiteral(**i) )
            {
            if( ! Propagate_DeriveFromRule(**i,I) )
                // An inconsistency has been detected.
                return false;
            }
        else if( atLeastOneDerivableMBTHeadAtom(**i) )
            {
            if( ! Propagate_DeriveMBTFromHead(**i,I) )
                // An inconsistency has been detected.
                return false;
            }
        }

    return true;
    }

//////////////////////////////////////////////////////////////////////////////
bool MODEL_GENERATOR::PropagateNegative(
//
// Returns: false if an inconsistency has arisen, true else.

    const GLITERAL&   a,
          GINTERPRET& I )
    {
    // Determine satisfaction of rules and constraints.
 
    if(a.isAggregate())
        // Perform backward inferences from aggregate.
        PropagateFalseAggregate(a,I);
    else if ( ! PropagateInAggregate(a,I,false) )
        return false;

    // If the atom of the propagated negative literal occurs in the
    // positive body of a rule, the body becomes false and therefore
    // the rule becomes satisfied.
    for( GRULEPTRS::const_iterator i = posBodyRules_begin(a);
         i != posBodyRules_end(a);
         i++ )
        {
        if( ! Propagate_RuleBecomesSatisfiedByBody(**i,I) )
            return false;
	}

    // If the atom of the propagated negative literal occurs in the
    // positive part of a constraint, the constraint becomes satisfied.
    for( GCONSTRAINTPTRS::const_iterator i = posConstraints_begin(a);
         i != posConstraints_end(a);
         i++ )
        {
        if( (**i).satisfied ) continue;

        PUSH( (**i).satisfied );
        (**i).satisfied=true;

        // A constraint (and thus also a partially violated clause)
        // has been eliminated.
        HC.ClauseEliminated++;
        HC.PVClauseEliminated++;

#ifdef WITH_PVCLEVELS
        switch( (**i).undefPosBody + (**i).undefNegBody )
            {
            case 0:
                assert(0);
            case 2:
                HC.PVClauseEliminatedLevel2++;
                break;
            case 3:
                HC.PVClauseEliminatedLevel3++;
                HC.PVClauseEliminated++;
                break;
            default:
                // nothing
                break;
            }
#endif

        }

    // If the atom of the propagated negative literal occurs in the positive
    // part of a weak constraint,the WeakConstraint becomes satisfied.
    for( GWEAKCONSTRAINTPTRS::const_iterator i = posWeakConstraints_begin(a);
         i != posWeakConstraints_end(a);
         i++ )
        {
        if( (**i).satisfied ) continue;

        PUSH( (**i).satisfied );
        (**i).satisfied=true;

        // We have eliminated a weak constraint.
        if(HC.EliminatedWCCost)
            {
            // Calculate the level and the increased value.
            unsigned int costLevel = (**i).getLevel();
            unsigned int incrValue = (*HC.EliminatedWCCost).getCost(costLevel) 
                                     + (**i).getWeight();

            // Store it in the proper level hc counter.
            (*HC.EliminatedWCCost).setCost(costLevel, incrValue);
            }
        }

    // Adjust counters and make possible inferences.

    // If the atom of the propagated negative literal occurs in the
    // negative body of an unsatisfied rule, the corresponding counter
    // is decreased, and possible inferences are made.
    for( GRULEPTRS::const_iterator i = negBodyRules_begin(a);
         i != negBodyRules_end(a);
         i++ )
        {
        if( (**i).satisfied ) continue;

        assert( (**i).undefNegBody > 0 );
        PUSH( (**i).undefNegBody );
        (**i).undefNegBody--;

        if( atMostOneDerivableLiteral(**i) )
            {
            if( ! Propagate_DeriveFromRule(**i,I) )
                // An inconsistency has been detected.
                return false;
            }
        else if( atLeastOneDerivableMBTHeadAtom(**i) )
            {
            if( ! Propagate_DeriveMBTFromHead(**i,I) )
                // An inconsistency has been detected.
                return false;
            }

        // A partially violated clause could have been created (if the
        // body became true/mbt).
        if( (**i).partiallyViolated() )
            {
            // If the clause is violated by its body, a new partially
            // violated clause has just been created.
            if( (**i).partiallyViolatedByBody() )
                {
                HC.PVClauseIntroduced++;

#ifdef WITH_PVCLEVELS
                switch( (**i).partialViolationLevel() )
                    {
                    case 0:
                        assert(0);
                    case 2:
                        HC.PVClauseIntroducedLevel2++;
                        break;
                    case 3:
                        HC.PVClauseIntroducedLevel3++;
                        break;
                    default:
                        // nothing
                        break;
                    }
#endif

                }

#ifdef WITH_PVCLEVELS
            else
                {
                // The rule is partially violated by its head. The
                // number of undefined atoms in the body has been
                // reduced. A partially violated clause of a
                // particular level could have been created.
                switch( (**i).partialViolationLevel() )
                    {
                    case 0:
                        assert(0);
                    case 1:
                        // A partially violated clause of level 2 has
                        // been eliminated.
                        HC.PVClauseEliminatedLevel2++;
                        break;
                    case 2:
                        // A partially violated clause of level 3 has
                        // been eliminated and one of level 2 has been
                        // introduced.
                        HC.PVClauseIntroducedLevel2++;
                        HC.PVClauseEliminatedLevel3++;
                        break;
                    case 3:
                        // A partially violated clause of level 3 has
                        // been introduced.
                        HC.PVClauseIntroducedLevel3++;
                        break;
                    default:
                        // nothing
                        break;
                    }
                }
#endif

            }
        }

    // If the atom of the propagated negative literal occurs in the
    // head of an unsatisfied rule, the corresponding counter is
    // decreased, and possible inferences are made.
    for( GRULEPTRS::const_iterator i = headRules_begin(a);
         i != headRules_end(a);
         i++ )
        {
        if( (**i).satisfied ) continue;

        assert( (**i).undefHead > 0 );
        PUSH( (**i).undefHead );
        (**i).undefHead--;

        if( atMostOneDerivableLiteral(**i) )
            if( ! Propagate_DeriveFromRule(**i,I) )
                // An inconsistency has been detected.
                return false;
        // There is no need to check the "true body, at least one
        // must-be-true atoms in the head" condition here.

        // A partially violated clause could have been created (if the
        // head became false).
        if( (**i).partiallyViolated() )
            {
            // If the clause is violated by its head, a new partially
            // violated clause has just been created.
            if( (**i).partiallyViolatedByHead() )
                {
                HC.PVClauseIntroduced++;

#ifdef WITH_PVCLEVELS
                switch( (**i).partialViolationLevel() )
                    {
                    case 0:
                        assert(0);
                    case 2:
                        HC.PVClauseIntroducedLevel2++;
                        break;
                    case 3:
                        HC.PVClauseIntroducedLevel3++;
                        break;
                    default:
                        // nothing
                        break;
                    }
#endif

                }

#ifdef WITH_PVCLEVELS
            else
                {
                // The rule is partially violated by its body. The
                // number of undefined atoms in the head has been
                // reduced. A partially violated clause of a
                // particular level could have been created.
                switch( (**i).partialViolationLevel() )
                    {
                    case 0:
                        assert(0);
                    case 1:
                        // A partially violated clause of level 2 has
                        // been eliminated.
                        HC.PVClauseEliminatedLevel2++;
                        break;
                    case 2:
                        // A partially violated clause of level 3 has
                        // been eliminated and one of level 2 has been
                        // introduced.
                        HC.PVClauseIntroducedLevel2++;
                        HC.PVClauseEliminatedLevel3++;
                        break;
                    case 3:
                        // A partially violated clause of level 3 has
                        // been introduced.
                        HC.PVClauseIntroducedLevel3++;
                        break;
                    default:
                        // nothing
                        break;
                    }
                }
#endif

            }
        }

    // If the atom of the propagated negative literal occurs in the
    // negative part of an unsatisfied constraint, the corresponding
    // counter is decreased, and possible inferences are made.
    for( GCONSTRAINTPTRS::const_iterator i = negConstraints_begin(a);
         i != negConstraints_end(a);
         i++ )
        {
        if( (**i).satisfied ) continue;

        assert( (**i).undefNegBody > 0 );
        PUSH( (**i).undefNegBody );
        (**i).undefNegBody--;

        if( atMostOneDerivableLiteral(**i) )
            if( ! Propagate_DeriveFromConstraint(**i,I) )
                // An inconsistency has been detected.
                return false;

#ifdef WITH_PVCLEVELS
        switch( (**i).undefPosBody + (**i).undefNegBody )
            {
            case 0:
                assert(0);
            case 1:
                // A partially violated clause of level 2 has been
                // eliminated.
                HC.PVClauseEliminatedLevel2++;
                break;
            case 2:
                // A partially violated clause of level 3 has been
                // eliminated and one of level 2 has been introduced.
                HC.PVClauseEliminatedLevel3++;
                HC.PVClauseIntroducedLevel2++;
                break;
            case 3:
                // A partially violated clause of level 3 has been
                // introduced.
                HC.PVClauseIntroducedLevel3++;
                break;
            default:
                // nothing
                break;
            }
#endif

        }
    // If the atom of the propagated negative literal occurs in the negative
    // part of an unsatisfied weak constraint, the corresponding counter is
    // decreased, possible inference is made, and we check whether the weak
    // constraint is violated.
    for( GWEAKCONSTRAINTPTRS::const_iterator i = negWeakConstraints_begin(a);
         i != negWeakConstraints_end(a);
         i++ )
        {
        if( (**i).satisfied ) continue;

        assert( (**i).undefNegBody > 0 );
        PUSH( (**i).undefNegBody );
        (**i).undefNegBody--;

        // Try to derive something from this weak constraint.
        if( possibleInferenceFromWeakConstraint(**i,I.getCost()) )
            Propagate_DeriveFromWeakConstraint(**i,I);

        else if( noMoreUndefinedLiteral(**i))
            // A violation has been detected.
            {
            I.increaseCost((**i).getWeights());
            if((isOptimalCostKnown &&( I.getCost() > *bestCost)) ||
               (!isOptimalCostKnown && ( I.getCost() >= *bestCost)) )
                {
                // Throw inconsistency as we won't find a better model.
                if(TraceLevel >=3)
                    {
                    cdebug << indent(level) 
                           << "Violated WeakConstraint " << (**i) 
                           << " throws inconsistency."
                           << endl << indent(level)
                           << "Current cost of the inconsistent"
                              " interpretation ([Weight:Level]): "
                           << endl;
                    I.printCost(cdebug);
                    }
                return false;
                }
            increasedCost = true;
            }
        }

    return true;
    }


//////////////////////////////////////////////////////////////////////////////
bool MODEL_GENERATOR::PropagateMustBeTrue(
//
// Returns: false if an inconsistency has arisen, true else.

    const GATOM &a,
          GINTERPRET &I )
    {
    // Determine satisfaction of rules and constraints.

    if(a.isAggregate())
        // Perform backward inferences from aggregate.
        PropagateMustBeTrueAggregate(a,I);
    else if ( ! PropagateInAggregate(a,I,true) )
        return false;

    // If the atom occurs in the negative body of a rule, the body can
    // never become true later on. The rule should therefore be
    // considered satisfied.
    for( GRULEPTRS::const_iterator i=negBodyRules_begin(a);
         i != negBodyRules_end(a);
         i++ )
        {
        if( ! Propagate_RuleBecomesSatisfiedByBody(**i,I) )
            return false;
        }

    // If the atom occurs in the negative part of a constraint, the
    // constraint can never be violated. Therefore we consider the
    // constraint satisfied.
    for( GCONSTRAINTPTRS::const_iterator i=negConstraints_begin(a);
         i != negConstraints_end(a);
         i++ )
        {
        if( (**i).satisfied ) continue;

        PUSH( (**i).satisfied );
        (**i).satisfied=true;

        // A constraint (and thus also a partially violated clause)
        // has been eliminated.
        HC.ClauseEliminated++;
        HC.PVClauseEliminated++;

#ifdef WITH_PVCLEVELS
        switch( (**i).undefPosBody + (**i).undefNegBody )
            {
            case 0:
                assert(0);
            case 2:
                HC.PVClauseEliminatedLevel2++;
                break;
            case 3:
                HC.PVClauseEliminatedLevel3++;
                break;
            default:
                // nothing
                break;
            }
#endif

        }
    // If the atom occurs in the negative part of a weak constraint, the
    // weak constraint can never be violated. Therefore we consider the
    // weak constraint satisfied.
    for( GWEAKCONSTRAINTPTRS::const_iterator i=negWeakConstraints_begin(a);
         i != negWeakConstraints_end(a);
         i++ )
        {
        if( (**i).satisfied ) continue;

        PUSH( (**i).satisfied );
        (**i).satisfied=true;

        // We have eliminated a weak constraint.
        // Calculate the level and the increased value
        if(HC.EliminatedWCCost)
            {
            unsigned int costLevel = (**i).getLevel();
            unsigned int incrValue =  (*HC.EliminatedWCCost).getCost(costLevel)
                                    + (**i).getWeight();
            // Store it in the proper level
            (*HC.EliminatedWCCost).setCost(costLevel, incrValue);
            }       
        }

    // Adjust counters and make possible inferences.

    if( getPotentialSupport(a) == 1 )
        if( ! Propagate_OnePotentiallySupportingRule(a,I) )
            return false;

    // If the propagated must-be-true atom occurs in the head of an
    // unsatisfied rule, the corresponding counter is increased, and
    // possible inferences are made.
    for( GRULEPTRS::const_iterator i = headRules_begin(a);
         i != headRules_end(a);
         i++ )
        {
        if( (**i).satisfied ) continue;

        assert( (**i).undefHead > 0 );
        assert( (**i).undefHead > (**i).mbtHead );
        PUSH( (**i).mbtHead );
        (**i).mbtHead++;

        // We do not have to check the "single undefined literal"
        // condition here.
        if( atLeastOneDerivableMBTHeadAtom(**i) )
            {
            if( ! Propagate_DeriveMBTFromHead(**i,I) )
                // An inconsistency has been detected.
                return false;
            }
        }

    // If the propagated must-be-true atom occurs in the positive body
    // of an unsatisfied rule, the corresponding counter is increased,
    // and possible inferences are made.
    for( GRULEPTRS::const_iterator i = posBodyRules_begin(a);
         i != posBodyRules_end(a);
         i++ )
        {
        if( (**i).satisfied ) continue;

        assert( (**i).undefPosBody > 0 );
        assert( (**i).undefPosBody > (**i).mbtPosBody );
        PUSH( (**i).mbtPosBody );
        (**i).mbtPosBody++;

        if( atMostOneDerivableLiteral(**i) )
            if( ! Propagate_DeriveFromRule(**i,I) )
                // An inconsistency has been detected.
                return false;
        // There is no need to check the "true body, at least one
        // must-be-true atoms in the head" condition here.

        // A partially violated clause could have been created (if the
        // body became true/mbt).
        if( (**i).partiallyViolated() )
            {
            // If the clause is violated by its body, a new partially
            // violated clause has just been created.
            if( (**i).partiallyViolatedByBody() )
                {
                HC.PVClauseIntroduced++;

#ifdef WITH_PVCLEVELS
                switch( (**i).partialViolationLevel() )
                    {
                    case 0:
                        assert(0);
                    case 2:
                        HC.PVClauseIntroducedLevel2++;
                        break;
                    case 3:
                        HC.PVClauseIntroducedLevel3++;
                        break;
                    default:
                        // nothing
                        break;
                    }
#endif

                }

#ifdef WITH_PVCLEVELS
            else
                {
                // The rule is partially violated by its head. The
                // number of undefined atoms in the body has been
                // reduced. A partially violated clause of a
                // particular level could have been created.
                switch( (**i).partialViolationLevel() )
                    {
                    case 0:
                        assert(0);
                    case 1:
                        // A partially violated clause of level 2 has
                        // been eliminated.
                        HC.PVClauseEliminatedLevel2++;
                        break;
                    case 2:
                        // A partially violated clause of level 3 has
                        // been eliminated and one of level 2 has been
                        // introduced.
                        HC.PVClauseIntroducedLevel2++;
                        HC.PVClauseEliminatedLevel3++;
                        break;
                    case 3:
                        // A partially violated clause of level 3 has
                        // been introduced.
                        HC.PVClauseIntroducedLevel3++;
                        break;
                    default:
                        // nothing
                        break;
                    }
                }
#endif

            }
        }

    // Unlike in rules, must-be-true atoms should be treated exactly
    // as "true" atoms in constraints, since no unsupportedness can
    // arise from constraints.

    // If the atom occurs in the positive part of a constraint, the
    // undefined counter is decreased, and possible inferences are made.
    for( GCONSTRAINTPTRS::const_iterator i = posConstraints_begin(a);
         i != posConstraints_end(a);
         i++ )
        {
        if( (**i).satisfied ) continue;

        assert( (**i).undefPosBody > 0 );
        PUSH( (**i).undefPosBody );
        (**i).undefPosBody--;

        if( atMostOneDerivableLiteral(**i) )
            if( ! Propagate_DeriveFromConstraint(**i,I) )
                // An inconsistency has been detected.
                return false;
        }

    // If the atom occurs in the positive part of a constraint, the
    // undefined counter is decreased, and possible inferences are made.
    for( GWEAKCONSTRAINTPTRS::const_iterator i = posWeakConstraints_begin(a);
         i != posWeakConstraints_end(a);
         i++ )
        {
        if( (**i).satisfied ) continue;

        assert( (**i).undefPosBody > 0 );
        PUSH( (**i).undefPosBody );
        (**i).undefPosBody--;

        // Try to derive something from this weak constraint.
        if( possibleInferenceFromWeakConstraint(**i,I.getCost()) )
            Propagate_DeriveFromWeakConstraint(**i,I);

        else if( noMoreUndefinedLiteral(**i) )
            // A violation has been detected.
            {
            I.increaseCost((**i).getWeights());
            if((isOptimalCostKnown &&( I.getCost() > *bestCost)) ||
               (!isOptimalCostKnown && ( I.getCost() >= *bestCost)) )
                {
                // Throw an inconsistency as we won't find a better model.
                if(TraceLevel >=3)
                    {
                    cdebug << indent(level)
                           << "Violated WeakConstraint " << (**i) 
                           << " throws inconsistency." 
                           << endl << indent(level)
                           << "Current cost of the inconsistent"
                              " interpretation ([Weight:Level]): "
                           << endl;
                    I.printCost(cdebug);
                    }
                return false;
                }
            increasedCost = true;
            }
        }

    return true;
    }

//////////////////////////////////////////////////////////////////////////////
bool MODEL_GENERATOR::Propagate(
//
// Returns: false if an inconsistency has arisen, true else.
//
// Queue is always emtpy upon return, even if inconsistency has been detected.
//
    GINTERPRET &I )
    {
    do 
        {
        // Check whether an inference from weak 
        // constraints has to be computed.
        increasedCost = false;

        while( ! Queue.empty() )
            {
            const GATOM a(Queue.front().second);
            const TruthValue val=Queue.front().first;
            
            if( TraceLevel >= 2 )
                cdebug << indent(level) << "Dequeuing and propagating " 
                       << a << " <" << val << ">"
                       << endl;

            Queue.pop();
            
            switch( val )
                {
                case TRUE_FROM_UNDEFINED_V:
                    assert( I.isTrue(a) );
                    if( ! PropagatePositiveFromUndefined(a,I) )
                        {
                        Queue.clear();
                        if (haveToExecuteComputeGUS)                        
                            HCFcomponentQueue.clear();
                        return false;
                        }
                    break;
                case TRUE_FROM_MUST_BE_TRUE_V:
                    assert( I.isTrue(a) );
                    if( ! PropagatePositiveFromMustBeTrue(a,I) )
                        {
                        Queue.clear();
                        if (haveToExecuteComputeGUS)
                            HCFcomponentQueue.clear();
                        return false;
                        }
                    break;
                case FALSE_V:
                    assert( I.isFalse(a) );
                    if( ! PropagateNegative(a,I) )
                        {
                        Queue.clear();
                        if (haveToExecuteComputeGUS)
                            HCFcomponentQueue.clear();
                        return false;
                        }
                    break;
                case MUST_BE_TRUE_V:
                    assert( I.isMustBeTrue(a) || I.isTrue(a) );
                    if( ! PropagateMustBeTrue(a,I) )
                        {
                        Queue.clear();
                        if (haveToExecuteComputeGUS)
                            HCFcomponentQueue.clear();
                        return false;
                        }
                    break;
                default:
                    assert(0);
                }
            }

        // Here we call the computeGUS algorithm.
        while (haveToExecuteComputeGUS && !HCFcomponentQueue.empty())
            {
            COMPONENT_INDEX C = HCFcomponentQueue.front();
            
            bool inconsist = ! (computeGUS(C, I));

            HCFcomponentQueue.pop();
            
            if (inconsist)
                {
                Queue.clear();
                HCFcomponentQueue.clear();
                return false;
                } // if()
            } // while()

        // Check whether the cost has been increased, if so make inference from
        // weak constraints.
        if(increasedCost)
            if(!inferenceFromWeakConstraints(I))
                {
                Queue.clear();
                if (haveToExecuteComputeGUS)                        
                    HCFcomponentQueue.clear();
                return false;
                }
        } // do_while()
    while (!Queue.empty());
    
    return true;
    }

//////////////////////////////////////////////////////////////////////////////
bool MODEL_GENERATOR::LookaheadPropagate(
//
// This procedure handles the propagation after some lookahead has
// lead to an inconsistency.
//
// @param I, the current interpretation
//
// @return false on inconsistency
//
// Queue is always emtpy upon return, even if inconsistency has been detected.
//
    GINTERPRET &I )
    {
    while( ! LookaheadQueue.empty() )
        {
        const GATOM a(LookaheadQueue.front().second);
        const TruthValue val=LookaheadQueue.front().first;

        if( TraceLevel >= 2 )
            cdebug << indent(level) << "Dequeuing and propagating " 
                   << a << " <" << val
                   << ">" << endl;

        LookaheadQueue.pop();

        switch( val )
            {
            case FALSE_V:
                if( I.isUndefined(a) )
                    {
                    Propagate_Derive_False(a,I);
                    }
                else if( ! I.isFalse(a) )
                    {
                    // Inconsistency
                    LookaheadQueue.clear();
                    return false;
                    }
                // If a is already false in I, it has already been
                // derived as false.
                break;
            case MUST_BE_TRUE_V:
                if( I.isUndefined(a) )
                    {
                    Propagate_Derive_MustBeTrue(a,I);
                    }
                else if( I.isFalse(a) )
                    {
                    // Inconsistency
                    LookaheadQueue.clear();
                    return false;
                    }
                // If a is already must-be-true or true in I, nothing
                // needs to be done.
                break;
            default:
                assert(0);
            }
        }

    return Propagate(I);
    }

//////////////////////////////////////////////////////////////////////////////
// Perform the lookahead for a given literal.
// In the course of this the heuristic counters are set.
//
// @param a, the GATOM of the literal
// @param neg, true (false) if the literal is negative (positive)
// @param supported, true if the literal should be assumed to be supported
// @param I, the current interpretation
//
// @return false on inconsistency
//
inline bool MODEL_GENERATOR::Lookahead(
    const GATOM& a,
    const bool neg,
    const bool supported,
    GINTERPRET& I)
    {
    // A negative atom should never be supported.
    assert( !(supported && neg) );

    // Save the interpretation state.
    I.setLoggingMarker();
    
    // The effects of this propagation should be redone
    // after returning from the recursion. Therefore they
    // are defined to belong to the next recursion level.
    level++;

    // Reset the heuristic counters.
    HC.reset();

    // propagate the literal
    if( neg )
        {
        if( TraceLevel >= 2 )
            cdebug << indent(level)
                   << ">>>>>> Start Lookahead: <FALSE," << a 
                   << ">" << endl;

        Propagate_Derive_False(a,I);
        }
    else
        {
        if( supported )
            {
            if( TraceLevel >= 2 )
                cdebug << indent(level)
                       << ">>>>>> Start Lookahead: <TRUE," << a
                       << ">" << endl;

            Propagate_Derive_TrueFromUndefined(a,I);
            }
        else
            {
            if( TraceLevel >= 2 )
                cdebug << indent(level)
                       << ">>>>>> Start Lookahead: <MUST_BE_TRUE," << a
                       << ">" << endl;

            Propagate_Derive_MustBeTrue(a,I);
            }
        }

#ifdef TIMERS
    timerPropagate.start();
#endif

    bool inconsistent = ! Propagate(I);

    if(HC.HeuristicCost)
        (*HC.HeuristicCost) = I.getCost();

    if( TraceLevel >= 2 )
        cdebug << indent(level)
               << "<<<<<< Stop Lookahead" << endl;

#ifdef TIMERS
    timerPropagate.stop();
#endif

    if( ! inconsistent && TraceLevel >= 3 )
        cdebug << HC << endl;

    // Undo propagation side effects.

    level--;
    I.unrollUntilLoggingMarker();
    ShortStack.unroll(level);
    UnsignedStack.unroll(level);
    if( ! IntegerStack.empty() )
        IntegerStack.unroll(level);

    // If the lookahead did create inconsistency, queue the "negation"
    // (pos->neg, neg->mbt) for subsequent propagation.
    if( inconsistent )
        {
        if( neg )
            {
            LookaheadQueue.push(MUST_BE_TRUE_V,a);
            }
        else
            {
            LookaheadQueue.push(FALSE_V,a);
            }
        }

    return inconsistent;
}


//////////////////////////////////////////////////////////////////////////////
// Calculate the heuristic values for a given PT.
//
// @param a, the GATOM of the PT literal
// @param neg, true (false) if the PT literal is negative (positive)
// @param I, the current interpretation
//
// @return false on inconsistency
//
bool MODEL_GENERATOR::CalculateHeuristicValue(
    const GATOM &a,
    const bool neg,
    GINTERPRET &I,
    HeuristicCounters *hcptr,
    HeuristicCounters *hccomplptr)
    {
    // Save PositiveAssumptions' size.
    const vector<GATOM>::size_type positiveAssumptionsSize = PositiveAssumptionsStack.size();

    bool inconsistent = false;

    if( hcptr )
        {
        // If the PT is positive, it should be derived as
        // (potentially) supportedly true. In other words, if neg is
        // false, we want to give support, if neg is true, we
        // don't. So the truthvalue of support is exactly !neg.
        statLookahead++;
        inconsistent = Lookahead(a,neg,!neg,I);

        if( inconsistent )
            {
            statInconsistentLookahead++;
            }
        else
            {
            // Save the heuristic values.
            *hcptr = HC;
            }
        }

    if( hccomplptr && ! inconsistent )
        {
        // Now compute the heuristic values of the complementary literal...
        statLookahead++;
        inconsistent = Lookahead(a,!neg,false,I);
        // ...and save them
        if( ! inconsistent )
            {
            *hccomplptr = HC;
            }
        else
            {
            statInconsistentLookahead++;
            }
        }

    // Restore PositiveAssumptions.
    while( PositiveAssumptionsStack.size() > positiveAssumptionsSize )
        {
        PositiveAssumptions->erase(PositiveAssumptionsStack.back());
        PositiveAssumptionsStack.pop_back();
        }

    return ! inconsistent;
    }

//////////////////////////////////////////////////////////////////////////////
// All rules, constraints and atoms are checked if something can be
// unconditionally derived. If so, the atoms and their truthvalue in
// question will be queued. Afterwards, the queued atoms and their
// associated truthvalues will be propagated.
//
// IMPORTANT: The linear time structures must have been initialised
// before the call to this function.
//
// @param I, the initial interpretation to be modified. This means
//           that all atoms are undefined upon the call to this
//           function.
//
// @return false if an inconsistency has been detected, true else
//
bool MODEL_GENERATOR::ComputeWellFounded( 
//
    GINTERPRET   &I )
    {

    // Initialise bestCost.
    if( ! isOptimalCostKnown)
        {
        bestCost = new COST(*costBound);

        if(optCostBound)
            isOptimalCostKnown = true;
        }

    // Traverse all GATOMs and derive the falsity of rules whose
    // potentially supporting rule counter is 0.
    for( unsigned i = 0; i != GATOMSET::getSize(); i++)
        {
        GATOM a(i);

        if( ( getPotentialSupport(a) == 0 ) && ! a.isAggregate() )
            {
            // An atom which does not occur in any rule head may
            // not have become true.
            assert( ! I.isTrue(a) );

            // There is no way how an atom may have become
            // "must-be-true" yet.
            assert( ! I.isMustBeTrue(a) );

            // Actually, the atom should be undefined.
            assert( I.isUndefined(a) );

            // If this is a strange GATOM, which does not occur
            // anywhere, just set it to true.
            // FIXME: Such cases should not be produced by the
            // grounding...
            if( posBodyRules_begin(a) == posBodyRules_end(a)
                && negBodyRules_begin(a) == negBodyRules_end(a)
                && posConstraints_begin(a) == posConstraints_end(a)
                && negConstraints_begin(a) == negConstraints_end(a)
                && posWeakConstraints_begin(a) == posWeakConstraints_end(a)
                && negWeakConstraints_begin(a) == posWeakConstraints_end(a) )
                {
                // The atom should not occur in any head, otherwise
                // the potential support counter should not be zero
                // (nothing has been propagated yet).
                assert( headRules_begin(a) == headRules_end(a) );

                if( TraceLevel >= 2 )
                    cdebug << "Deriving <FALSE," << a
                           << "> (without propagation) [" << a
                           << " does not occur anywhere]" << endl;

                I.addFalse(a);
                }
            else
                {
                // Otherwise use the usual propagation.
                if( TraceLevel >= 2 )
                    cdebug << "Deriving <FALSE," << a << ">  [" << a
                           << " does not occur in any head]" << endl;

                Propagate_Derive_False(a,I);
                }
            }
        }

    // Traverse all rules and check whether something can be derived.
    for( GRULES::iterator ri = rules.begin();
         ri != rules.end();
         ri++ )
        {
        // A rule should not be satisfied here, since all atoms are
        // undefined, and propagation did not happen yet.
        if( (*ri).satisfied )
            assert( 0 );

        // If there is a single undefined literal, derive it.
        if( atMostOneDerivableLiteral(*ri) )
            if( ! Propagate_DeriveFromRule(*ri,I) )
                // An inconsistency has been detected.
                return false;
        }

    // Traverse all constraints and check whether something can be
    // derived.
    for( GCONSTRAINTS::iterator ci = constraints.begin();
         ci != constraints.end(); 
         ci++ )
	{
        // A constraint should not be satisfied here, since all atoms
        // are undefined, and propagation did not happen yet.
        if( (*ci).satisfied )
            assert( 0 );
        
        // If there is a single undefined literal, derive it.
        if( atMostOneDerivableLiteral(*ci) )
            if( ! Propagate_DeriveFromConstraint(*ci,I) )
                // An inconsistency has been detected.
                return false;
        }

    // Traverse all weak constraints and check whether a inference can be made.
    for( GWEAKCONSTRAINTS::iterator ci = wconstraints.begin();
         ci != wconstraints.end(); 
         ci++ )
	{
        // A weak constraint should not be satisfied here, since all atoms
        // are undefined, and propagation did not happen yet.
        if( (*ci).satisfied )
            assert( 0 );

        // If there is a single undefined literal, try to derive it from
        // this weak constraint.
        if( possibleInferenceFromWeakConstraint(*ci,I.getCost()) )
            Propagate_DeriveFromWeakConstraint(*ci,I);     
        }

    // Propagate any queued atoms and their associated truthvalues and
    // return the result (success or inconsistency).
    return Propagate(I);
    }

//////////////////////////////////////////////////////////////////////////////
// If it looks like a model has been found, assume the remaining
// undefined atoms as false to discover potential unsatisfied rules
// and constraints.
//
// @param I, the model candidate
//
// @return false if an inconsistency has been detected, true else.
//
bool MODEL_GENERATOR::Totalise( GINTERPRET& I )
    {
    // for every atom in the HB:
    for( unsigned i = 0; i < GATOMSET::getSize(); i++ )
        {
        GATOM a(i);

        if( I.isUndefined(a) )
            {
            Propagate_Derive_False(a,I);
            if( ! Propagate(I) )
                return false;
            }
        else if( I.isMustBeTrue(a) )
            {
            // An atom can be mbt at this point only if it has been
            // derived mbt during the totalisation phase. In this
            // case, it should be false and true at the same time,
            // which is an inconsistency.
            return false;
            }
        }
    return true;
    }

//////////////////////////////////////////////////////////////////////////////
void MODEL_GENERATOR::foundStableModel(const GINTERPRET &I)
//
    {

    if( !isOptimalCostKnown )
        {
        assert(bestCost);
        *bestCost = I.getCost();
        }

    if (bestModel)
        {
        *bestModel = I;
        }
    else
        bestModel = new GINTERPRET(I);

    // Check whether the cost of the non-deterministic propagation is zero.
    // In this case the model is optimal for sure.
    assert(wfCost);
    if( !isOptimalCostKnown )
        if( (I.getCost() - *wfCost).isZero() )
            isOptimalCostKnown = true;

    if( isOptimalCostKnown )
        statModelsFound++;

    // In some cases we may be interested in the existence of a model only,
    // and thus have no callback function available.
    if( callback )
        {
        if(groundedWC)
            {
            if( isOptimalCostKnown
                || optTraceWC
                || optCostBound )
                {
                assert(bestModel);
                timerCallback.start();
                if ( ! callback(this,globalI,bestModel) )
                    mode=MODE_ABORT;
                timerCallback.stop();
                }
            else if ( TraceLevel >= 3 )
                {
                assert(bestCost);
 
                cdebug << endl << "Current best model:"
                       << *bestModel << endl
                       << "Cost ([Weight:Level]): "
                       << *bestCost << endl;
                }
            }
        else
            {
            timerCallback.start();
            if ( ! callback(this,globalI,&I) )
                mode=MODE_ABORT;
            timerCallback.stop();
            }
        }
    }

//////////////////////////////////////////////////////////////////////////////
// @return whether the (partial) model is stable.
//
bool MODEL_GENERATOR::checkModel(
//
    GINTERPRET &I,
    GATOMSET *Y,
    bool *Yinitialized )
    {
    // Create the dependency graph exactly once and on demand, as
    // building it might be expensive.
    if( ! posGroundDG  &&  ! ( flags & KNOWN_HCF) )
        {
        timerDepgraph.start();
        posGroundDG = new DEPGRAPH<GATOM,GCONSTRAINTS,GPROGRAM> 
            (rules,GCONSTRAINTS(),0);
        timerDepgraph.stop();
        }

    // Note: We also have to invoke the Model Checker for
    // non-disjunctive input due to the way we choose (positive) PTs
    // and assume positive literals in the case of alternate branches
    // during Model Generation and inferences from constraints.

#ifndef NDEBUG
    size_t oldQueueSize=Queue.size();
    size_t oldShortStackSize=ShortStack.size();
    size_t oldUnsignedStackSize=UnsignedStack.size();
    size_t oldIntegerStackSize=IntegerStack.size();
#endif

    I.enableLogging(false);

    const bool stable=IsStable( I,rules,
                                flags & KNOWN_HCF,
                                posGroundDG,*PositiveAssumptions,
                                Y, Yinitialized );

    I.enableLogging(true);

    assert( Queue.size() == oldQueueSize );
    assert( ShortStack.size() == oldShortStackSize );
    assert( UnsignedStack.size() == oldUnsignedStackSize );
    assert( IntegerStack.size() == oldIntegerStackSize );
    
    return stable;
    }

//////////////////////////////////////////////////////////////////////////////
void MODEL_GENERATOR::foundModelCandidate(
//
    GINTERPRET& I )
    {
    statModelCandidates++;
    turnPointHigh=level;

    // The check for must-be-true freeness is cheaper, so we do it first.
    if( I.isMustBeTrueFree()
        && Totalise(I) )
        {
#ifdef NEVER
#ifndef NDEBUG
        if( flags & PERFORM_GUS )
            {
            assert( posGroundDG );

            // Here we check if all the HCF and cyclic components are
            // "stable".
            for (COMPONENT_INDEX C = 0; 
                 C < (((*posGroundDG).getComponents()).size()); C++)
                {
                if ((*posGroundDG).isCyclic(C) && (*posGroundDG).isHCF(C))
                    {
                    assert( IsComponentStable( posGroundDG, I,
                                               *PositiveAssumptions,
                                               C, MCM_CALLDAVISPUTNAM ) );

                    if (TraceLevel >= 3)
                        cdebug << "GUS: correctness checking: component"
                               << C + 1 << " (cyclic and HCF) is 'stable'."
                               << endl;
                    }
                }
            }
#endif
#endif

        timerTotalChecks.start();

        if( ( flags & PERFORM_CHECK )
            &&  PositiveAssumptionsStack.size() > 0 )
            {
            statTotalChecks++;

            if( ! checkModel(I,&ufset,&ufsetInitialized) )
                {
                assert( ! ufsetInitialized
                        || IsUnfoundedSet(ufset,I,rules,0) );

                mode=MODE_CHECK_FAILED;
                ufsetRules.clear();
                }

            if( TraceLevel >= 1 )
                cdebug << indent(level)
                       << "Model check: "
                       << ( mode==MODE_CHECK_FAILED ? "failed."
                                                    : "succeeded." )
                       << endl;
            }

        timerTotalChecks.stop();
    
        if( callbackCheck && mode != MODE_CHECK_FAILED )
            {
            timerCallback.start();

            if ( ! callbackCheck(this,globalI,&I) )
                {
                mode=MODE_CHECK_FAILED;
                ufsetInitialized=false;
                }

            timerCallback.stop();
            }

        if( mode != MODE_CHECK_FAILED )
            {
            foundStableModel(I);
            // Note that this may set mode == MODE_ABORT!
            ufsetInitialized=false;
            }
        }   
    else
        {
        if( TraceLevel >= 1 )
            cdebug << indent(level) 
                   << "Model check: "
                      "Total(ized) Interpretation violates constraints "
                      "or must-be-true atoms would become false." 
                   << endl;
        ufsetInitialized=false;
        }
    }

//////////////////////////////////////////////////////////////////////////////
// computes the deterministic extension of I in which ptl is true (if
// it is a negative literal) or false (if ptl is a positive
// literal). To this end ptl is set to mbt (if ptl is negative) or
// false (if ptl is positive) and propagation is performed.
//
// @param reallyPT, a flag indicating whether a PT or its complement
//                  is to be assumed
// @param ptl, a literal
// @param I, an interpretation
// @return false if there is no consistent extension, true else
//
bool MODEL_GENERATOR::assumeComplementaryPTLiteral(
//
          bool        reallyPT,
    const GLITERAL&   ptl, 
          GINTERPRET& I    )
    {
    if( (ptl.isNegative() && reallyPT) || (!ptl.isNegative() && !reallyPT) )
        {
        // If the current PT literal is negative, set it to
        // must-be-true and propagate it. It is not set to true since
        // "true" implies supportedness which is not guaranteed in
        // this case.
        if( TraceLevel >= 2 )
            {
            cdebug << indent(level)
                   << "Deriving <MUST_BE_TRUE," 
                   << GATOM(ptl);
            if( ptl.isNegative() && reallyPT )
                cdebug << ">  [fully explored negative PT literal]"
                       << endl;
            else
                cdebug << ">  [fully explored complement "
                          "of a positive PT literal]" << endl;
            }

        Propagate_Derive_MustBeTrue(ptl,I);
        }
    else
        {
        // If the current PT literal is positive, set it to false and
        // propagate it, since no more model which includes it should
        // be found.  If it is already must-be-true, the subtree
        // should be abandoned immediately.
        if( I.isMustBeTrue(ptl) )
            {
            assert(0);
            if( TraceLevel >= 1 )
                cdebug << indent(level)
                       << "Propagation leads to inconsistency. "
                          "Aborting entire subtree."
                       << endl;

            return false;
            }
        else
            {
            if( TraceLevel >= 2 )
                {
                cdebug << indent(level)
                       << "Deriving <FALSE," << ptl;
                if( !ptl.isNegative() && reallyPT )
                    cdebug << ">  [fully explored positive PT literal]"
                           << endl;
                else
                    cdebug << ">  [fully explored complement "
                              "of a negative PT literal]" << endl;
                }

            Propagate_Derive_False(ptl,I);
            }
        }

#ifdef TIMERS
    timerPropagate.start();
#endif

    bool inconsistent = ! Propagate(I);

#ifdef TIMERS
    timerPropagate.stop();
#endif

    if( inconsistent )
        {
        if( TraceLevel >= 1 )
            cdebug << indent(level)
                   << "Propagation leads to inconsistency. "
                      "Aborting entire subtree."
                   << endl;

        if( TraceLevel >= 3 )
            RecomputeCounters(I,true);

        return false;
        }

    return true;
    }


//////////////////////////////////////////////////////////////////////////////
void MODEL_GENERATOR::assumePTLiteral(
//
          bool&       isChoicePointWithRealBranches,
          bool        reallyPT,
    const GLITERAL&   ptl, 
          GINTERPRET& I    )
    {
    I.setLoggingMarker();

    // Atoms may be added to PositiveAssumptions when a positive
    // PT literal is picked and also during propagations. To undo
    // these additions, the size of PositiveAssumptions is saved
    // before picking a PT literal, and after propagation (and
    // possibly recursion) atoms are popped from its back until
    // the structure has shrunk to the saved size again.
    const vector<GATOM>::size_type positiveAssumptionsSize
        = PositiveAssumptionsStack.size();

    // Propagate the current possibly true.
    //
    // The effects of this propagation should be redone
    // after returning from the recursion. Therefore they
    // are defined to belong to the next recursion level.

    level++;

    if( (ptl.isNegative() && reallyPT) || (!ptl.isNegative() && !reallyPT ) )
        {
        if( TraceLevel >= 2 )
            {
            cdebug << indent(level) 
                   << "Deriving <FALSE," << GATOM(ptl);
            if( ptl.isNegative() && reallyPT )
                cdebug << ">  [negative PT literal]" << endl;
            else
                cdebug << ">  [promising complement of a positive PT literal]"
                       << endl;
            }

        Propagate_Derive_False(ptl,I);
        }
    else
        {
        if( !ptl.isNegative() && reallyPT )
            {
            if( TraceLevel >= 2 )
                cdebug << indent(level)
                       << "Deriving <TRUE," << ptl
                       << ">  [positive PT literal]" << endl;

            // There should be no PT literals which are
            // must-be-true because if must-be-true atoms
            // satisfy the PT criterion, they should be
            // derived as true deterministically.
            Propagate_Derive_TrueFromUndefined(ptl,I);

            PositiveAssumptions->add(ptl);
            PositiveAssumptionsStack.push_back(ptl);
            }
        else
            {
            assert(ptl.isNegative() && !reallyPT);

            if( TraceLevel >= 2 )
                cdebug << indent(level) 
                       << "Deriving <MUST_BE_TRUE," << GATOM(ptl) 
                       << ">  [negative PT literal]" << endl;

            Propagate_Derive_MustBeTrue(ptl,I);
            }
        }

#ifdef TIMERS
    timerPropagate.start();
#endif

    bool inconsistent= ! Propagate(I);

#ifdef TIMERS
    timerPropagate.stop();
#endif

    // If propagation causes inconsistency...
    if( inconsistent )
        {
        statInconsistentChoices++;

        if( TraceLevel >= 1 )
            cdebug << indent(level)
                   << "Propagation generates inconsistency"
                   << endl;
        }
    else
        {
        assert( Queue.empty() );

        if( TraceLevel >= 1 )
            cdebug << indent(level)
                   << "Propagation generates " << I << endl;
        }

    if( TraceLevel >= 3 )
        RecomputeCounters(I,true);

    // ...we abandon the current subtree, else we continue
    // processing this very tree.
    if( ! inconsistent )
        {
        if( ! isChoicePointWithRealBranches )
            {
            isChoicePointWithRealBranches = true;
            statChoicePointsWithRealBranches++;
            }

        findModelsContaining(I);

        if( mode == MODE_ABORT )
            return;
        }

    // The propagations which might follow should not be
    // redone in this recursion level.

    level--;

#ifdef TIMERS
    timerRecomputeCounters.start();
#endif

    I.unrollUntilLoggingMarker();
    ShortStack.unroll(level);
    UnsignedStack.unroll(level);
    if( ! IntegerStack.empty() )
        IntegerStack.unroll(level);

#ifdef TIMERS
    timerRecomputeCounters.stop();
#endif
                
    assert( ! RecomputeCounters(I,true) );

    // Restore PositiveAssumptions.
    while( PositiveAssumptionsStack.size() 
           > positiveAssumptionsSize )
        {
        PositiveAssumptions->erase(PositiveAssumptionsStack.back());
        PositiveAssumptionsStack.pop_back();
        }

    // This assertion triggers if PositiveAssumptions has
    // shrunk below its previous size during propagation
    // and recursion.
    assert( PositiveAssumptionsStack.size()
            == positiveAssumptionsSize );
    }

//////////////////////////////////////////////////////////////////////////////
void MODEL_GENERATOR::findModelsContaining(
//
    GINTERPRET& I )
    {
    // lastBestCost is used to check whether a inference from weak constraints
    //  is possible.
    COST* lastBestCost = 0;

    // Level 1 is the computation tree's root, therefore it
    //   is always reached exactly once
    // Level 2's counter is statRecursion[1] etc.
    if( level > statMaxRecursion )
        {
        assert( level == statMaxRecursion+1 );

        statMaxRecursion=level;
        // The statRecursion counters are only needed for stat level 2 and up.
        if( OptionStatsLevel > 1 )
            {
	    statRecursion.push_back(1);
	    assert( statRecursion.size() == level );
            }
        }
    else
        {
        // The statRecursion counters are only needed for stat level 2 and up.
        if( OptionStatsLevel > 1 )
            {
            statRecursion[level-1]++;
            // Note: Levels start with 1, while C++ arrays start with index 0.
            }
        }

    // Perform a (partial) model check while descending the search tree
    // processed by the Model Generator.
    //
    // This has to come after updating the statistics, to avoid skewing these.

    const unsigned triggerLevel=turnPointLow+(turnPointHigh-turnPointLow)*1/3;
    if( ( flags & PERFORM_CHECK )
        && OptionPartialChecksForwards
        && PositiveAssumptionsStack.size() > 0
        && ! posGroundDG->isHCF()
        && turnPointLow
        && triggerLevel > turnPointLow
        && level == triggerLevel )
        {
        statPartialChecksForwards++;

        timerPartialChecks.start();
        const bool ok=checkModel(I,&ufset,&ufsetInitialized);
        timerPartialChecks.stop();

        if( ! ok )
            {
            statPartialChecksForwardsFailed++;

            mode=MODE_CHECK_FAILED;
            ufsetInitialized=true;
            ufsetRules.clear();

            if( TraceLevel >= 1 )
                cdebug << indent(level)
                       << "Partial Model Check: failed: "
                       << I << endl;

            assert( ! lastBestCost );

            return;
            }
        else
            {
            if( TraceLevel >= 1 )
                cdebug << indent(level)
                       << "Partial Model Check: ok" << endl;
            }
        }

    timerPT.start();

    PT pt(*this,I);

    HeuristicPT* PPTs = new HeuristicPT(*this,I,OptionPTCacheSize);
    HeuristicPT& PTs = *PPTs;

    if( OptionUseHeuristics )
        {
        if( ! PTs.init() )
            {
            statPruningLookahead++;
            delete PPTs;
            assert(!lastBestCost);
            timerPT.stop();
            return;
            }
        }
    else
        pt.init(rules);

    timerPT.stop();

    if( OptionUseHeuristics ? PTs.atEnd() : pt.atEnd() )
        foundModelCandidate(I);
    else
        {
        bool backtrack=false;
        // A computation node is a choice point if at least one
        // consistent choice is taken.
        bool isChoicePointWithRealBranches=false;
        do
            {
            if( ! OptionUseHeuristics )
                assert( pt.isReallyPossible() );

            bool reallyPT = true;

            const GLITERAL& ptl = OptionUseHeuristics
                ? PTs.getLiteral(reallyPT) : pt.getLiteral();

            // Store current cost, so that we can check, after assumePTLiteral,
            // whether we need to make an inference from weak constraints.
            // This is done only during the first step of the computation.
            if( !isOptimalCostKnown )
                {
                lastBestCost = new COST(I.getCost());
                }

            assumePTLiteral(isChoicePointWithRealBranches,reallyPT,ptl,I);

            // (Optionally) perform partial model checks and backtrack as 
            // long as these checks fail, i.e., until we encounter a founded
            // partial model.
            if( mode == MODE_CHECK_FAILED )
                {
                if( ( flags & PERFORM_CHECK )
                    && PositiveAssumptionsStack.size() > 0 )
                    {
                    statPartialChecksBacktrack++;
                    timerPartialChecks.start();

                    bool known_uf=false;              // Known to be unfounded?
#if CHECK_OPT >= 1
                    if( ufsetInitialized )
                        {
                        timerPartialChecksQuick.start();

                        if( TraceLevel >= 1 )
                            cdebug << indent(level) 
                                   << "Using cached unfounded set "
                                   << ufset << endl;

                        // If this is our first time here for this unfounded
                        // set, we fill ufsetRules with those rules relevant
                        // for the unfoundedness condition.  Else we can use
                        // ufsetRules as initialized previously.
                        if( ufsetRules.size() == 0 )
                            known_uf=IsUnfoundedSet(ufset,I,rules,&ufsetRules);
                        else
                            known_uf=IsUnfoundedSet(ufset,I,ufsetRules);

                        timerPartialChecksQuick.stop();

                        if( TraceLevel >= 1 )
                            cdebug << indent(level)
                                   << "Quick partial model check: "
                                   << ( known_uf ? "failed" : "ok" ) << endl;

                        if( known_uf )
                            statPartialChecksBacktrackQuick++;
                        }
#endif

                    // If the quick check did not determine that we have an
                    // unfounded set (or we didn't perform such a quick check)
                    // we now have to perform a full check.
                    if( ! known_uf )
                        {
#if CHECK_OPT >= 2
                        known_uf = ! checkModel(I,&ufset,&ufsetInitialized);
#else
                        known_uf = ! checkModel(I,0,0);
#endif
                        ufsetRules.clear();
                        }
                    
                    timerPartialChecks.stop();

                    if( known_uf )
                        {
                        if( TraceLevel >= 1 )
                            cdebug << indent(level)
                                   << "Partial Model Check: failed: "
                                   << I << endl;

                        backtrack=true;
                        continue;
                        }
                    else
                        {
                        if( TraceLevel >= 1 )
                            cdebug << indent(level)
                                   << "Partial Model Check: ok" << endl;

                        turnPointLow=level;

                        mode=MODE_REGULAR;
                        }
                    }
                else
                    mode=MODE_REGULAR;
                }

            // If we are in Abort mode or a best model already exists,
            // we can return.
            if( mode == MODE_ABORT
                || ( !isOptimalCostKnown && I.getCost() == *bestCost ) )
                {
                backtrack=true;
                continue;
                }
            // If a lower best cost is found we need to make an inference from
            // weak constraints. This is done only if we're in the first step
            // of the computation.
            else if( !isOptimalCostKnown )
                {
                assert(bestCost && lastBestCost);
                if( *lastBestCost > *bestCost )
                    {
                    if(TraceLevel >=1)
                        cdebug << " inference from weakconstraints after "
                                  "backtracking" << endl;
                    if(!inferenceFromWeakConstraints(I))
                        {
                        backtrack=true;               // Exit the main loop...
                        continue;                     // ...right away.
                        }
                    }
                }

            if( assumeComplementaryPTLiteral(reallyPT,ptl,I) )
                {
                timerPT.start();

                if( OptionUseHeuristics )
                    {
                    if( ! PTs.next() )
                        {
                        statPruningLookahead++;
                        backtrack=true;
                        }
                    }
                else
                    pt++;

                timerPT.stop();

                if( ! backtrack
                    && ( OptionUseHeuristics ? PTs.atEnd() : pt.atEnd() ) )
                    {
                    foundModelCandidate(I);
                    }
                }
            else
                backtrack=true;
            }
        while( ! backtrack
               && ! ( OptionUseHeuristics ? PTs.atEnd() : pt.atEnd() ) );
        }

    delete PPTs;
    if( lastBestCost )
        delete lastBestCost;
    }


//////////////////////////////////////////////////////////////////////////////
void MODEL_GENERATOR::printStatistics(ostream &out) const
//
    {
    // statRecursion[0] is the counter for the root node level
    // (level 1).  This counter is always 1.  
    // statRecursion[1] is the counter for level 2 etc.  
    //
    // For the user output we use a different counting scheme: level 1
    // is not displayed (since it is always 1).  Display begins with
    // "level 1" which corresponds to level 2 in the implementation
    // terminology.
    //
    // Output is therefore: "Level 1 =" << statRecursion[1]
    //                      "Level 2 =" << statRecursion[2] etc.

    out << "Maximum recursion level   : ";

    if( inconsistency )
        out << "n/a (inconsistency in well-founded computation)" << endl;
    else if( statMaxRecursion > 0 )
        {
        // statMaxRecursion can only be 0 if an inconsistency arose in
        // the well-founded computation or a partial model check
        // before selection of the first PT literal failed. Even if
        // the (single) model is computed in the well-founded stage,
        // it must be checked whether there are any PT literals, which
        // already belongs to the first recursion step.

        out << statMaxRecursion-1 << endl;

        if( OptionStatsLevel > 1 )
            {
            for(unsigned i=1; i < statMaxRecursion; i++)
                out << "         Reached level " << setw(3) << i 
                    << ": " << statRecursion[i] << " times" << endl;

            const unsigned choices=accumulate( &statRecursion[1],
                                               &statRecursion[statMaxRecursion],
                                               0 );

            out << "=> Consistent choices     : " << choices 
                                                  << endl
                << "Inconsistent choices      : " << statInconsistentChoices
                                                  << endl;
            }

        out << "Choice Points             : " << statChoicePoints << endl;
        }
    else
        {
        assert( statPartialChecksForwards > 0 );
        out << "n/a (initial partial model check failed)" << endl;
        }

    if( OptionStatsLevel > 1 )
        {    
        out << "Lookaheads performed      : " << statLookahead << endl
            << " Inconsistent lookaheads  : " << statInconsistentLookahead << endl
            << " Pruning lookaheads       : " << statPruningLookahead << endl
            << "Choice Points (internal)  : " << statChoicePointsWithRealBranches << endl
            << "Partial checks forwards   : " << statPartialChecksForwards 
                                              << endl
            << "         forwards (failed): " << statPartialChecksForwardsFailed
                                              << endl
            << "Partial checks backtracking: " << statPartialChecksBacktrack
                                              << endl
            << "      backtracking (quick): " << statPartialChecksBacktrackQuick
                                              << endl
            << "Answer set candidates     : " << statModelCandidates << endl
            << "Total model checks        : " << statTotalChecks << endl
            << "Answer sets computed      : " << getNumberOfModelsComputed() << endl
            << endl;
        }

    out << "Answer sets printed       : " << statModelsPrinted << endl; 
    }

//////////////////////////////////////////////////////////////////////////////
void MODEL_GENERATOR::printTimers(ostream &out) const
//
    {
    out << "Model Generator time      : " << timerTotal
                                             - timerTotalChecks 
                                             - timerPartialChecks
                                             - timerCallback
                                          << endl;

    if( OptionStatsLevel > 1 )
        {
        out << " DS Initialisation        : " << timerStructInit << endl
            << " Depgraph                 : " << timerDepgraph << endl
            << " Well-founded computation : " << timerWellFounded << endl
            << " PT computation           : " << timerPT << endl
            << " Frontend callbacks       : " << timerCallback << endl
#ifdef TIMERS
            << " Propagation              : " << timerPropagate << endl
            << " Counter Recomputation    : " << timerRecomputeCounters
            << endl
#endif
            ;
         }

    out  << "Model Checker time        : " << timerTotalChecks
                                              + timerPartialChecks
                                           << endl; 

    if( flags & PERFORM_CHECK )
        {
        out << " Total Model Check time   : " << timerTotalChecks
                                              << endl
            << " Partial Model Check time : " << timerPartialChecks
                                              << endl;

        if( OptionStatsLevel > 1 ) 
            {
            out << "  Quick Partial Check time: " << timerPartialChecksQuick
                                                  << endl;
            ::ModelCheckerStats();
            }
        }
    }


//////////////////////////////////////////////////////////////////////////////
MODEL_GENERATOR::MODEL_GENERATOR(
//
        GRULES            &rules2,
        GCONSTRAINTS      &constraints2,
        GWEAKCONSTRAINTS  &wconstraints2,
        const CONJUNCTION *originalQuery2,
        const VATOMSET    *globalEDB2,
        const INTERPRET   *globalI2,
        const unsigned    &optionModels2,
        const COST        *costBound2,
        const FLAGS       flags2,
        bool              (*callback2)(MODEL_GENERATOR*, 
                                       const INTERPRET*, 
                                       const GINTERPRET*),
        bool              (*callbackCheck2)(MODEL_GENERATOR*, 
                                             const INTERPRET*,
                                             const GINTERPRET*),
        bool              recursiveAggregatesExist2,
        const COST        *wfCost2,
        bool              groundedWC2,
        bool              parsedWC2,
        bool              optTraceWC2,
        bool              optCostBound2,
        const unsigned    maxWeakConstraintLevel2,
        const unsigned    aggregateFunctionsCount2,
        const CriteriaSequence heuristicSequence2,
        const CriteriaSequence heuristicCombinationSequence2
    )
    : rules(rules2),
      constraints(constraints2),
      wconstraints(wconstraints2),
      originalQuery(originalQuery2),
      globalEDB(globalEDB2),
      globalI(globalI2),

      optionModels(optionModels2),
      flags(flags2),
      callback(callback2),
      callbackCheck(callbackCheck2),

      posGroundDG(0),

      haveToExecuteComputeGUS(false),
      recursiveAggregatesExist(recursiveAggregatesExist2),

      mode(MODE_REGULAR),
      
      statMaxRecursion(0),
      statRecursion(),
      statInconsistentChoices(0),
      statChoicePoints(0),
      statChoicePointsWithRealBranches(0),
      statModelCandidates(0),
      statTotalChecks(0),
      statPartialChecksForwards(0),
      statPartialChecksForwardsFailed(0),
      statPartialChecksBacktrack(0),
      statPartialChecksBacktrackQuick(0),
      statModelsFound(0),
      statModelsPrinted(0),
      statLookahead(0),
      statInconsistentLookahead(0),
      statPruningLookahead(0),
      
      PositiveAssumptions(0),

      ufset(), ufsetInitialized(false), ufsetRules(),
      
      inconsistency(false),

      bestCost(0),
      bestModel(0),

      groundedWC(groundedWC2),
      parsedWC(parsedWC2),
      optTraceWC(optTraceWC2),
      optCostBound(optCostBound2),

      increasedCost(false),
      maxWeakConstraintLevel(maxWeakConstraintLevel2),
      aggregateFunctionsCount(aggregateFunctionsCount2),

      HC(maxWeakConstraintLevel),
      hcbuffer(maxWeakConstraintLevel),
      hccomplbuffer(maxWeakConstraintLevel),

      heuristicSequence(heuristicSequence2),
      heuristicCombinationSequence(heuristicCombinationSequence2),

      GUSInterpret(maxWeakConstraintLevel2)

    {
    // If maxWeakConstraintLevel is zero then the optimal cost is known to
    // be equal to zero since there are not weak constraints. 
    if (maxWeakConstraintLevel2)
        isOptimalCostKnown = false;
    else
        isOptimalCostKnown = true;

    if (costBound2)
        costBound = new COST(*costBound2);
    else
        // If costBound is the NULL pointer, initialize it with the minimal
        // (zero) cost. As of now (March 2001), this happens only when MG is 
        // called by plan.C and check.C.
        costBound = new COST();

    if (wfCost2)
        wfCost = new COST(*wfCost2);
    else
        // If wfCost is the NULL pointer, initialize it with the minimal
        // (zero) cost. As of now (March 2001), this happens only when MG is 
        // called by plan.C and check.C.
        wfCost = new COST();
    
#ifndef NDEBUG
    if( flags & KNOWN_HCF )
        {
        assert( ! posGroundDG );

        posGroundDG = new DEPGRAPH<GATOM,GCONSTRAINTS,GPROGRAM> 
            (rules,GCONSTRAINTS(),0);
        assert( posGroundDG->isHCF() );
        delete posGroundDG;
        posGroundDG = 0;
        }
#endif        

    // In either of the following cases, we need the positive dependency graph.
    if( (flags & PERFORM_GUS)
        || OptionPartialChecksForwards
        || recursiveAggregatesExist )
        {
        assert( ! posGroundDG );

        // Here we must create an instance of the positive ground depgraph.
        timerDepgraph.start();
        posGroundDG = new DEPGRAPH<GATOM,GCONSTRAINTS,GPROGRAM> 
            (rules, GCONSTRAINTS(), 0);
        timerDepgraph.stop();
        }

    if( flags & PERFORM_GUS )
        {
        // The computeGUS algorithm has to run only if the graph
        // contains at least one component which is cyclic AND HCF.
        
        // If there are no cyclic components it is not useful to check
        // further.
        if ((*posGroundDG).isCyclic())
            {
            const unsigned componentsCount = 
                (*posGroundDG).getComponents().size();
            
            // Check if there is at least one cyclic and HCF component.
            // If it is so, the computeGUS algorithm will be enabled.
            for (unsigned  C = 0; C < componentsCount; C++)
                {
                if ( (*posGroundDG).isCyclic(C)
                     && (*posGroundDG).isHCF(C) )
                    {
                    if (! haveToExecuteComputeGUS)
                        {
                        haveToExecuteComputeGUS = true;
                        
                        if ( TraceLevel >= 3 )
                            cdebug << "ComputeGUS is ACTIVE." << endl;
                        
                        // We initialise the queue that stores the
                        // components on which we have to execute the
                        // computeGUS algorithm. Actually, we should
                        // push onto the queue a component if at least
                        // one atom belonging to it becomes TRUE
                        // during the propagation (FIXME: or MBT).
                        // NOTE: it is used only by the computeGUS
                        // algorithm, so we need to initialise it only
                        // if it can be invoked.
                        const unsigned componentsCount = 
                            (*posGroundDG).getComponents().size();
                        HCFcomponentQueue.init(componentsCount);
                        } // if()

                    HCFcomponentQueue.push(C);
                    } // if()         
                } // for()
            } // if()
        
#ifdef NDEBUG
        // If we don't have to perform any GUS or model check computation
        // and do not need it for debugging, we can remove the dependency
        // graph.
        if( ! haveToExecuteComputeGUS && ! (flags & PERFORM_CHECK) )
            {
            delete posGroundDG;
            posGroundDG = 0;
            }
#endif

        if ( TraceLevel >= 3 )
            if (!haveToExecuteComputeGUS)
                cdebug << "ComputeGUS is NOT ACTIVE." << endl;
        }

    } // MODEL_GENERATOR()

//////////////////////////////////////////////////////////////////////////////
void MODEL_GENERATOR::init(
//
    GINTERPRET &I )
    {
    level = 0;

    assert( ShortStack.empty() );
    assert( UnsignedStack.empty() );
    assert( IntegerStack.empty() );
    assert( Queue.empty() );

    assert( PositiveAssumptions == 0 );
    PositiveAssumptions=new GATOMSET;
    assert( PositiveAssumptionsStack.size() == 0 );

    // Intialize counters and linear-time data structures.

    assert( HeadOccurrences.empty() );
    assert( PosBodyOccurrences.empty() );
    assert( NegBodyOccurrences.empty() );
    assert( PosConstraintOccurrences.empty() );
    assert( NegConstraintOccurrences.empty() );
    assert( potentiallySupportingRules.empty() );
    assert( occursInAggregateFunction.empty() );
    assert( occursInGatom.empty() );

#ifdef TIMERS
    timerStructInit.start();
#endif

    // InitialiseLinearDataStructures should return true, as we always
    // use "fresh" GINTERPRETs.

#ifndef NDEBUG
    inconsistency = ! 
#endif
        initialiseLinearDataStructures(I);

    assert( ! inconsistency );

#ifdef TIMERS
    timerStructInit.stop();
#endif

    if( TraceLevel >= 3 )
        {
        PrintGatomRulesConstraints(cdebug);
        PrintGruleCounters(cdebug);
        PrintGconstraintCounters(cdebug);
        }
    }

//////////////////////////////////////////////////////////////////////////////
void MODEL_GENERATOR::freeLinearDataStructures()
//
    {
    HeadOccurrences.clear();
    PosBodyOccurrences.clear();
    NegBodyOccurrences.clear();
    PosConstraintOccurrences.clear();
    NegConstraintOccurrences.clear();
    PosWConstraintOccurrences.clear();
    NegWConstraintOccurrences.clear();
    potentiallySupportingRules.clear();
    occursInAggregateFunction.clear();
    occursInGatom.clear();
    }

//////////////////////////////////////////////////////////////////////////////
void MODEL_GENERATOR::done()
//
    {
    freeLinearDataStructures();

    delete PositiveAssumptions;
    PositiveAssumptions=0;
    PositiveAssumptionsStack.clear();

    alreadyConsideredAsPosPT.clear();
    alreadyConsideredAsNegPT.clear();

    binaryClauseOccurrencesPos.clear();
    binaryClauseOccurrencesNeg.clear();

    ShortStack.clear();
    UnsignedStack.clear();
    IntegerStack.clear();
    Queue.clear();
    if (haveToExecuteComputeGUS)
        HCFcomponentQueue.clear();

    if(isOptimalCostKnown)
        {
        // Free costBound.
        assert(costBound);
        delete costBound;
        }
    }

//////////////////////////////////////////////////////////////////////////////
void MODEL_GENERATOR::detcons()
//
    {
    timerTotal.start();
    timerFirst.start();

    GINTERPRET J(maxWeakConstraintLevel);

    init(J);

#ifdef TIMERS
    timerWellFounded.start();
#endif

    inconsistency = ! ComputeWellFounded(J);

#ifdef TIMERS
    timerWellFounded.stop();
#endif

    if( inconsistency )
        cout << "Inconsistency!" << endl;
    else
        {
        cout << "{";

        bool cont=false;

        if( OptionPrintFacts )
            cont = globalEDB->printCensored(cout,false);

        cont = globalI->printCensored(cout,cont) || cont;

        // Print atoms know to be true (that are not censored).
        for(size_t i=0; i < GATOMSET::getSize(); i++)
            {
            GATOM a(i);

            if( J.isTrue(a)
                && ! a.getTheRealOne().getPredItem().isCensored() )
                {
                if(cont)
                    cout << ", ";
                else
                    cont=true;

                cout << a;
                }
            }

#ifdef NEVER
        // Print atoms know to be false (that are not censored).
        for(size_t i=0; i < GATOMSET::getSize(); i++)
            {
            GATOM a(i);

            if( J.isFalse(a)
                && ! a.getTheRealOne().getPredItem().isCensored()
                // FIXME: This is only needed, as the grounding may
                // add some atoms to the residual Herbrand Base which
                // are aready known as definitely true, and thus do
                // not occur in the head of any rule.
                && headRules_begin(a) != headRules_end(a) )
                {
                if(cont)
                    cout << ", ";
                else
                    cont=true;

                cout << "not " << a;
                }
            }
#endif

        cout << "}" << endl;
        }

    done();
    }


//////////////////////////////////////////////////////////////////////////////
bool MODEL_GENERATOR::simplifyConjunction( 
// Simplify a ground conjunction (a constraint or the body of a rule)
// by removing true positive literals and false negative literals.
//
// NOTE: In the case of a constraint, remove also MBT positive
// literals. We must not do this for rules.
//
// R is to distinguish between rules and constraints, and also used
// for debug tracing. It should be 0 for a constraint.
//
// Return true if something was removed, false otherwise.
//
    const GINTERPRET &J,
    GCONJUNCTION &C,
    const GRULE *R )
    {
    bool didRemove = false;

    // Remove true (and maybe MBT) positive literals.
    for ( GCONJUNCTION::iterator l = C.pos_begin();
          l != C.pos_end(); )
        if ( J.isTrue(*l) || ( !R  && J.isMustBeTrue(*l)) )
            {
            if( TraceLevel >= 1)
                if (R)
                    cdebug << "  Removing true literal " 
                           << *l << " from the body of "
                           << *R <<  endl;
                else
                    cdebug << "  Removing true literal " 
                           << *l << " from "
                           << C <<  endl;
            
            l = C.remove(l);
            
            didRemove = true;
            }
        else
            l++;

    // Remove false negative literals.
    for ( GCONJUNCTION::iterator l = C.neg_begin();
          l != C.neg_end(); )
        if (J.isFalse(*l))
            {
            if( TraceLevel >= 1)
                if (R)
                    cdebug << "  Removing false literal " 
                           << *l << " from the body of " 
                           << *R << endl;
                else
                    cdebug << "  Removing false literal " 
                           << *l << " from " 
                           << C << endl;
            
            l = C.remove(l);
            
            didRemove = true;
            }
        else 
            l++;
    
    return didRemove;
    } 


//////////////////////////////////////////////////////////////////////////////
bool MODEL_GENERATOR::simplifyProgram(
// Check the program w.r.t. an interpretation and simplify it if
// possible. Delete rules and constraints already satisfied, and
// remove from the heads and bodies of remaining ones those literals
// that are not "useful" anymore.
//
// Return true if something was modified, false otherwise.
//
    const GINTERPRET &J,
    bool &rulesModified )
    {
    if( TraceLevel >= 1 )
        cdebug << "Simplifying program after WF." << endl;

    bool programHasChanged = false;

    rulesModified = false;

    // For each rule...
    for( GRULES::iterator i=rules.begin(); 
         i != rules.end(); )
        {
	// If the rule is already satisfied, we can delete it.
        // NOTE: a rule is satisfied even if the body contains a
        // negative literal whose atom is MBT.
        if ( (*i).satisfied == GRULE::SATISFIED_BY_HEAD 
             || (*i).satisfied == GRULE::SATISFIED_BY_BODY
             || (*i).satisfied == GRULE::SATISFIED_BY_HEAD_AND_BODY )
            {
            // Delete the rule from the ground program.
            if( TraceLevel >= 1 )
	        {
                cdebug << "  Removing rule " << *i
		       << " It is already satisfied";

                if( (*i).satisfied == GRULE::SATISFIED_BY_HEAD )
		    cdebug << " by its head.";
		else if( (*i).satisfied == GRULE::SATISFIED_BY_BODY )
		    cdebug << " by its body.";
                else
		    cdebug << " by both its head and body.";
                cdebug << endl;
                }
            unordered_erase(rules, i);
            
            programHasChanged = true;
	    }

        // If the rule isn't satisfied yet, maybe it can be
        // simplified removing some atoms...
        else
            {
            if( i->hasBody() )
                {
                GCONJUNCTION &body = 
                    *const_cast<GCONJUNCTION*>((*i).getBody());

                programHasChanged |= simplifyConjunction(J, body, &*i);
                
                }

            if( i->hasHead() )
                { 
                GDISJUNCTION &head = i->getHeadForModification();
                
                // Remove false atoms in the head.
                for ( GDISJUNCTION::iterator l = head.begin();
                      l != head.end(); )
                    if (J.isFalse(*l))
                        {
                        if( TraceLevel >= 1)
                            cdebug << "  Removing false atom " 
                                   << *l << " from the head of " 
                                   << *i << endl;
                        
                        l = head.remove(l);
                        
                        programHasChanged = true;
                        }
                    else
                        l++;
                }

            // Increase rules iterator (only if the current
            // rule was not removed, as erasing an element
            // overwrites that element).
            i++;
            } // else
	} // for()

    // At this point constraints haven't been processed at all, so if
    // programHasChanged holds there was at least a change in rules
    // set (deletion, simplification, or both).
    if (programHasChanged)
        rulesModified = true;


    // For each constraint...
    for( GCONSTRAINTS::iterator i=constraints.begin(); 
         i != constraints.end(); )
        {
        // If the constraint is already satisfied, we can delete it.
        if ( (*i).satisfied )
            {
            // Delete the constraint from the ground program.
            if( TraceLevel >= 1 )
                {
                cdebug << "  Removing constraint " 
                       << *i << "  It is already satisfied."
                       << endl;
                }
            unordered_erase(constraints, i);
            
            programHasChanged = true;
            }

        // If the constraint isn't satisfied yet, maybe it can be
        // simplified by removing some atoms...
        else
            {
            programHasChanged |= simplifyConjunction(J, *i, 0);
            
            // Increase constraints iterator (only if the current
            // constraint was not removed, as erasing an element
            // overwrites that element).
            i++;
            } // else
        } // for()


    if( TraceLevel >= 1  &&  ! programHasChanged )
        cdebug << " No changes made to the ground program." << endl;

    return programHasChanged;
    } 


//////////////////////////////////////////////////////////////////////////////
void MODEL_GENERATOR::run()
//
    {
    turnPointLow=turnPointHigh=0;

    timerTotal.start();
    timerFirst.start();

    mode=MODE_REGULAR;

    GINTERPRET J(maxWeakConstraintLevel);

    if( wfCost )
        J.setCost(*wfCost);

    init(J);

#ifdef TIMERS
    timerWellFounded.start();
#endif

    inconsistency = ! ComputeWellFounded(J);

    assert(wfCost);
    // The well-founded cost might have been increased.
    *wfCost = J.getCost();
    if ( TraceLevel >= 3 )
        cdebug << "Well founded cost ([Weight:Level]): " 
               << *wfCost << endl;
        
#ifdef TIMERS
    timerWellFounded.stop();
#endif


    if( inconsistency )
        {
        if( TraceLevel >= 1 )
            cdebug << "W results in inconsistency." << endl;
        }
    else
        {
        if( TraceLevel >= 1 )
            cdebug << "WF results in " << J << endl;

        level=1;

        // Try to simplify the ground program after the WF.
        //
        // If the ground program was modified, free the linear ground
        // data structures and the ground dependency graph, then
        // rebuild them.
        // 
        // NOTE: The ground dependency graph has to be rebuilt only if
        // some changes occurred.

        bool rulesModified;
        
        // FIXME: Disabled for now.
        if( false && simplifyProgram(J, rulesModified) )
            {
            if( TraceLevel >= 1 )
                cdebug << "SimplifyAfterWF: rebuilding linear data structures..." 
                       << endl;

            freeLinearDataStructures();
            initialiseLinearDataStructures(J);
            
            if (rulesModified)
                {
                if( TraceLevel >= 1 )
                    cdebug << "SimplifyAfterWF: rebuilding ground dep.graph..."
                           << endl;

                assert(posGroundDG);
                delete posGroundDG;

                timerDepgraph.start();
                posGroundDG = new DEPGRAPH<GATOM,GCONSTRAINTS,GPROGRAM> 
                    (rules, GCONSTRAINTS(), 0);
                timerDepgraph.stop();
                }

            if( TraceLevel >= 3 )
                {
                cdebug << "SimplifyAfterWF: counters after rebuilding..."
                       << endl;
                PrintGatomRulesConstraints(cdebug);
                PrintGruleCounters(cdebug);
                PrintGconstraintCounters(cdebug);
                }
            }
        
        // Finally, perform the backtracking computation.
        J.enableLogging(true);
        findModelsContaining(J);
        J.enableLogging(false);
        }

    // Even if we have printed no model at all, stop the timer for the first model now.
    if( statModelsPrinted == 0 )
        timerFirst.stop();
    timerTotal.stop();

    // Restore all items still on the stack.
    //
    // Note: This should be required only when a second run of the MG will
    // follow -- which currently only happens if isOptimalCostKnown() is
    // false -- because we need to restore the values of the aggregates
    // bounds, but we decided to pop all the items on the stack in any case,
    // because this should not affect performance too much.
    ShortStack.restore();
    UnsignedStack.restore();
    IntegerStack.restore();

    done();
    }

// Local Variables:
// c-file-style: "dl"
// End:
