//////////////////////////////////////////////////////////////////////////////
// ginterpret.h 

#ifndef GINTERPRET_H
#define GINTERPRET_H
#include "gatomset.h"

//////////////////////////////////////////////////////////////////////////////
// In TruthValue the value 'true' is represented by two seperate
// truthvalues, in order to be able to differentiate between atoms
// which have become true directly from undefined and atoms which have
// become true via must-be-true.  These two truthvalues should have
// the highest numerical values, such that testing whether an atom is
// 'true', not caring about whether directly from undefined or via
// must-be-true, still amounts to one comparison: true >=
// TRUE_FROM_MUST_BE_TRUE_V.
//
// (The elements of TruthValue (apart from UNDEFINED_V) intentionally
// have exactly one bit true each to allow them to be OR-ed properly.
// UNDEFINED_V should be 0 for performance reasons, and has to be treated
// in a different way than the other OR-ed values, cf. printTruthValue.)

enum TruthValue {
    UNDEFINED_V = 0,
    FALSE_V = 1,
    MUST_BE_TRUE_V = 2, 
    TRUE_FROM_MUST_BE_TRUE_V = 4,
    TRUE_FROM_UNDEFINED_V = 8
    };

inline ostream& operator<< (ostream& out, const TruthValue v)
    {
    switch( v )
        {
        case UNDEFINED_V:
            out << "UNDEFINED";
            break;
        case TRUE_FROM_UNDEFINED_V:
            out << "TRUE (FROM UNDEFINED)";
            break;
        case TRUE_FROM_MUST_BE_TRUE_V:
            out << "TRUE (FROM MUST_BE_TRUE)";
            break;
        case FALSE_V:
            out << "FALSE";
            break;
        case MUST_BE_TRUE_V:
            out << "MUST_BE_TRUE";
            break;
        default:
            assert( 0 );
        }
    return out;
    }

//////////////////////////////////////////////////////////////////////////////
class COST
//////////////////////////////////////////////////////////////////////////////
    {
    unsigned *cost;

public:
    unsigned maxWeakConstraintLevel;

    // By default the dimension of the cost is set to maxWeakConstraintLevel,
    // otherwise a warning is shown.
    COST(unsigned numLevel = 0, unsigned value = 0)
        :maxWeakConstraintLevel(numLevel)
        {
        if(numLevel == 0)
            cost = 0;
        else
            {
            if( TraceLevel >= 2)
                {
                if(numLevel != maxWeakConstraintLevel)
                    cdebug << "Warning! In COST construction, numLevel is"
                              " set to a value which differs from"
                              " maxWeakConstraints level.";
                }
            cost = new unsigned[numLevel];
            fill_n( cost, numLevel, value);
            }
        }

    // By default the dimension of the cost is set to maxWeakConstraintLevel, 
    // otherwise a warning is shown.
    COST(const COST &c)
        : maxWeakConstraintLevel(c.maxWeakConstraintLevel)
        {
        if(c.maxWeakConstraintLevel == 0)
            cost =0;
        else
            {
            cost = new unsigned[c.maxWeakConstraintLevel];
            memcpy(cost, c.cost, c.maxWeakConstraintLevel * sizeof(unsigned));
            }
        }

    COST& operator=(const COST &c)
        {
        assert(this!=&c);

        if(cost)
            delete [] cost;

        assert(maxWeakConstraintLevel == c.maxWeakConstraintLevel);
        if(maxWeakConstraintLevel)  
            {
            cost = new unsigned[c.maxWeakConstraintLevel];
            memcpy(cost, c.cost, c.maxWeakConstraintLevel * sizeof(unsigned));
            }
        else
            cost =0;
  
        return *this;
        }

    // Get the i-th level cost.
    unsigned getCost(const unsigned i) const
        {
        if( i >= 1  &&  i <= maxWeakConstraintLevel );
            return cost[i-1];
        }

    // Set the i-th cost.
    void setCost(const unsigned level, const unsigned value)
        {
        if( level >= 1  &&  level <= maxWeakConstraintLevel )
            cost[level - 1] = value;
        }

    // Assign the last update of the cost level in the proper level.
    void updateCost(const WEIGHTS &weights)
        {
        // This method should only be called in the ground case.
        assert( weights.first.isInt() && weights.second.isInt() );

        // Increase the cost of the changing level.
        unsigned tmp = 
            getCost(weights.second.getInt()) + weights.first.getInt();
        setCost(weights.second.getInt(),tmp);
        if(TraceLevel >= 2)
            {
            cdebug << "  Updated cost of level "<< weights.second.getInt()
                   << ". New cost value: " 
                   << getCost(weights.second.getInt())/SCALEFACT;
            if( weights.first.getInt()%SCALEFACT > 0 )
                cdebug << "." << weights.first.getInt()%SCALEFACT;
            cdebug << endl;
            }
        }

    // Return true if all costs are set to zero.
    bool isZero() const
        {
        COST zero(maxWeakConstraintLevel);
        return *this == zero;
        }

    // Reset the cost.
    void reset()
        {
        assert(maxWeakConstraintLevel > 0);
        fill_n( cost,maxWeakConstraintLevel, 0);
        }


    // Compare two costs, level by level beginning from the greatest level.
    int operator!=(const COST &c) const
        {
        assert( maxWeakConstraintLevel == c.maxWeakConstraintLevel );
        for (int level = maxWeakConstraintLevel-1;level >= 0;level--)
            if ( cost[level] > c.cost[level] )
                return 1;
            else if ( cost[level] < c.cost[level] )
                return -1;

        return 0;
        }

    bool operator==(const COST &c) const
        {
        return ( *this != c ) == 0;
        }

    // Compare two costs, level by level beginning from the greatest level.
    bool operator< (const COST &c) const
        {
        return ( *this != c ) < 0;
        }

    // Compare two costs, level by level beginning from the greatest level.
    bool operator> (const COST &c) const
        {
        return ( *this != c ) > 0;
        }

    // Compare two costs, level by level beginning from the greatest level.
    bool operator>= (const COST &c) const
        {
        return ( *this != c ) >= 0;
        }
   
    COST operator-(const COST &c) const
        {
        assert( maxWeakConstraintLevel == c.maxWeakConstraintLevel );
        COST diff (maxWeakConstraintLevel,0);
        for(int level = maxWeakConstraintLevel-1; level >= 0; level--)
            diff.cost[level] = cost[level] - c.cost[level];
        return diff;
        }

    COST& operator+=(const COST &c)
        {
        assert( maxWeakConstraintLevel == c.maxWeakConstraintLevel );
        for (unsigned i = 0; i < c.maxWeakConstraintLevel; i++)
            cost[i] += c.cost[i];
        return *this;
        }

    COST operator+(const COST &c) const
        {
        assert( maxWeakConstraintLevel == c.maxWeakConstraintLevel );

        COST sum(maxWeakConstraintLevel,0);
        for (unsigned i = 0; i < maxWeakConstraintLevel; i++)
            sum.cost[i] = cost[i] + c.cost[i];
        return sum;
        }

    COST operator+(const WEIGHTS &w) const
        {           
        assert( 
            maxWeakConstraintLevel >= static_cast<unsigned>(w.second.getInt()));

        COST sum(*this);
        sum.cost[w.second.getInt()-1] += w.first.getInt();
        return sum;
        }

    COST operator+(const int scalar) const
        {
        COST sum(maxWeakConstraintLevel,0);
        for (unsigned i = 0; i < maxWeakConstraintLevel; i++)
            sum.cost[i] = cost[i] + scalar;
        return sum;
        }

    COST& operator*=(const COST &c)
        {
        assert( maxWeakConstraintLevel == c.maxWeakConstraintLevel );
        for (unsigned i = 0; i < c.maxWeakConstraintLevel; i++)
            cost[i] *= c.cost[i];
        return *this;
        }

    COST operator*(const COST &c) const
        {
        assert( maxWeakConstraintLevel == c.maxWeakConstraintLevel );

        COST prod(maxWeakConstraintLevel,0);
        for (unsigned i = 0; i < maxWeakConstraintLevel; i++)
            prod.cost[i] = cost[i] * c.cost[i];
        return prod;
        }

    COST operator*(const int scalar) const
        {
        COST prod(maxWeakConstraintLevel,0);
        for (unsigned i = 0; i < maxWeakConstraintLevel; i++)
            prod.cost[i] = cost[i] * scalar;
        return prod;
        }

    ~COST()
        {
        if(cost)
            delete[] cost;
        }
    };

inline ostream& operator<< (ostream &out, const COST &c)
    {
    out << "<";
    for(unsigned i = 1; i <= c.maxWeakConstraintLevel; i++)
        {
        out << "[" << c.getCost(i)/SCALEFACT; 
        if ( c.getCost(i)%SCALEFACT > 0)
            out << "." << c.getCost(i)%SCALEFACT;
        out << ":" << i << "]";
        if (i != c.maxWeakConstraintLevel)
            out << ",";
        }
    out << ">";

    return out;    
    }

//////////////////////////////////////////////////////////////////////////////
class GINTERPRET
//////////////////////////////////////////////////////////////////////////////
    {
    static size_t SIZE;

    TruthValue *array;

    // mbtCounter keeps track about how many atoms are must-be-true in
    // this interpretation. Needed mainly for isMustBeTrueFree().
    unsigned mbtCounter;

    // Support for logging
    bool logging;

    typedef pair<int,TruthValue> CHANGE;
    vector<CHANGE> stack;

    // Stores the updated cost of the interpretation at the current
    // non deterministic step.
    // first argument: priority level
    // second argument: cost value of the current level
    typedef pair<int,unsigned> COSTCHANGE;
    vector<COSTCHANGE> costStack;

    // Stores the cost of the interpretation.  The cost is the sum of
    // the weights of the violated weak contraints, for each level.
    COST cost;


    // Assign the index-th element of the interpretation a truth value.
    void assign(const unsigned index, const TruthValue value)
        {
        if( logging )
            stack.push_back( CHANGE(index,array[index]) );
        array[index]=value;
        }

public:
    void enableLogging(const bool logging2)
        {
        assert( logging != logging2 );

        logging=logging2;
        }

    void setLoggingMarker()
        {
        stack.push_back( CHANGE(-1,UNDEFINED_V) );
        costStack.push_back(COSTCHANGE(-1,0) );
        }

    void unrollUntilLoggingMarker()
        {
        while( stack.back().first != -1 )
            {
            if( array[stack.back().first] == MUST_BE_TRUE_V )
                {
                assert( stack.back().second != MUST_BE_TRUE_V );
                mbtCounter--;
                }
            else if( stack.back().second == MUST_BE_TRUE_V )
                {
                assert( array[stack.back().first] != MUST_BE_TRUE_V );
                mbtCounter++;
                }

            array[stack.back().first]=stack.back().second;

            stack.pop_back();
            }

        // Remove the marker.
        stack.pop_back();

       // Do the same for the value of the cost of the interpretation.
       while( costStack.back().first != -1 )
            {
            setCost(costStack.back());
            costStack.pop_back();
            }
        // Remove the marker.
        costStack.pop_back();
        }

    // Assign the last update of the cost level of the interpretation
    // in the proper level.
    void increaseCost(const WEIGHTS& weights)
        {
        assert(weights.second.isInt());

        if( logging )
            // Push in costStack the value of the changing level.
            costStack.push_back(COSTCHANGE(weights.second.getInt(), 
                                getCost(weights.second.getInt())));
        // Increase the cost of the changing level.
        cost.updateCost(weights);
        }

    // Reset the cost of the interpretation.
    void resetCost ()
        {
        cost.reset();
        }

    // Get the i-th level cost of the interpretation.
    unsigned getCost(const unsigned i) const
        {
        return cost.getCost(i);
        }

    // Print the i-th level cost of the interpretation.
    void printLevelCost(ostream& out, const unsigned i) const
        {
        if( cost.maxWeakConstraintLevel > 0
            && cost.maxWeakConstraintLevel >= i  )
            {
            out << (cost.getCost(i)/SCALEFACT); 
            if ( cost.getCost(i)%SCALEFACT > 0 )
                out << "." << (cost.getCost(i)%SCALEFACT);
            }
        else
            out << 0;
        }

    // Get the cost of the interpretation.
    const COST& getCost() const
        {
        return cost;
        }

    // Set the cost of the interpretation.
    void setCost(const COST& c)
        {
        cost = c;
        }

    // Set the cost of the interpretation.
    void setCost(const COSTCHANGE& c)
        {
        // First is the current level
        // second is the cost value of that level.
        cost.setCost(c.first, c.second);
        }
 
    // Print the cost of the interpretation.
    void printCost(ostream& out) const
        {
        if(cost.maxWeakConstraintLevel>0)
            out << cost;
        else 
            out << "<0>";
         }

    static void setSize(const size_t new_size)
        {
        assert( SIZE == 0  &&  GATOM::isfrozen() );

        SIZE=new_size;
        }

    static size_t size()
        {
        return SIZE;
        }

    GINTERPRET(
        const unsigned maxWeakConstraintLevel=0,
        const TruthValue tv=UNDEFINED_V ) 
        : mbtCounter(0), logging(false), stack(), costStack(),
          cost(maxWeakConstraintLevel)
        {
        assert( GATOM::isfrozen() );

        array = new TruthValue[SIZE];
        fill_n( array, SIZE, tv);
        }

    GINTERPRET(const GINTERPRET& I2)
        : mbtCounter(I2.mbtCounter), logging(false), stack(), costStack(),
          cost(I2.cost)
        {
        assert( GATOM::isfrozen() );

        array = new TruthValue[SIZE];
        assert( I2.array );
        memcpy( array, I2.array, SIZE * sizeof(TruthValue) );

        if( TraceLevel >= 3 )
            {
            if(cost.maxWeakConstraintLevel>0)
                {
                cdebug << "New Cost ([Weight:Level]): ";
                this->printCost(cdebug);
                }
            }
        }

    ~GINTERPRET()
        {
        delete[] array;
        }

    GINTERPRET& operator= (const GINTERPRET& I)
        {
        assert(this!=&I);
        mbtCounter = I.mbtCounter;
        logging = I.logging;
        stack = I.stack;
        costStack = I.costStack;
        cost = I.cost;
        if(array && I.array) 
            {
            delete[] array;
            array = new TruthValue[SIZE];
            memcpy( array, I.array, SIZE * sizeof(TruthValue) );
            }
        else if (array && !I.array)
            {
            delete[] array;
            array = 0;
            }
        else if (!array && I.array)
            {
            array = new TruthValue[SIZE];
            memcpy( array, I.array, SIZE * sizeof(TruthValue) );
            }
        return *this;
        }

    
    inline void addPositive(const GATOM&);
    inline void addFalse(const GATOM&);
    inline void forceFalse(const GATOM&);
    inline void addMustBeTrue(const GATOM&);
    inline void addTrueFromUndefined(const GATOM&);
    inline void addTrueFromMustBeTrue(const GATOM&);
    inline void addUndefined(const GATOM&);

    inline void unionPositive(const GINTERPRET&);

    inline bool isTrue (const GATOM&) const;
    inline bool isTrueFromUndefined (const GATOM&) const;
    inline bool isTrueFromMustBeTrue (const GATOM&) const;
    inline bool isTrue (const GLITERAL&) const;
    inline bool isTrue (const GDISJUNCTION&) const;
    inline bool isTrue (const GCONJUNCTION&) const;
    inline bool isTotallyTrue(const GATOM&) const;
    inline bool isTotallyTrue(const GLITERAL&) const;
    inline bool isTotallyTrue(const GCONJUNCTION&) const;
    inline bool isUndefined(const GATOM&) const;
    inline bool isFalse(const GATOM&) const;
    inline bool isFalse(const GLITERAL&) const;
    inline bool isFalse(const GCONJUNCTION&) const;
    inline bool isFalse(const GDISJUNCTION&) const;
    inline bool isMustBeTrue(const GATOM&) const;

    inline bool isMustBeTrueFree() const;

    // Print all the elements having TruthValue v (or, more precisely, any
    // of possibly several TruthValue's OR-ed into v).
    bool printTruthValue(
        ostream& out,
        const TruthValue v, 
        bool continuing=false ) const 
        { 
        bool printed=false;

        for( size_t i=0; i < size(); i++ )
            {
            // Unless v is 0, it can have several bits set, so we check
            // whether any of these is set in array[i]. Otherwise (v is
            // 0), we check standard equality.
            if( v ? array[i] & v : array[i] == 0 )
                {
                if( continuing )
                    out << ", ";
                else
                    continuing=true;

                printed=true;
                out << GATOM(i).getTheRealOne();
                }
            }

        return printed;
        }

    // "continuing" indicates whether this GINTERPRET should be printed
    // as a continuation of another GINTERPRET, which is not the case by
    // default.
    // returns true if anything was printed
    bool print(ostream& out, bool continuing=false) const
        {
        continuing=printTruthValue(out,TRUE_FROM_MUST_BE_TRUE_V,continuing)
                   || continuing;

        continuing=printTruthValue(out,TRUE_FROM_UNDEFINED_V,continuing)
                   || continuing;

        return continuing;
        }

/** prints a censored version of this GINTERPRET to the given output stream.
 * Uses globals PredFilter and PredPFilter that specify the wanted
 * predicates. If both are empty, no filtering will take place, and the
 * printout is verbatim.
 * @param out the destination output stream.
 * @param continuing whether the output continues preceding output.
 * @return whether something was printed.
 */
    bool printCensored(ostream& out, bool continuing=false) const
        {
        bool didPrint=false;
        for (unsigned i=0; i < size(); i++)
            if( array[i] >= TRUE_FROM_MUST_BE_TRUE_V )
                {
                const ATOM &atom=GATOM(i).getTheRealOne();

                if( ! atom.getPredItem().isCensored() )
                    {
                    if (continuing)
                        out << ", ";
                    else
                        continuing=true;

                    if( ObjectsExist )
                        atom.printWithoutObjectID(out);
                    else
                        out << atom;

                    didPrint=true;
                    }
                }
        return didPrint;
        }

    ///////////////////////////////////////////////////////////////////
    void exportMatching(
    // Copy all ATOMs which are true in the current interpretation into
    // the vector atoms and increase count accordingly.
    //
        const char *pname,
        vector<ATOM> &atoms,
        unsigned &count ) const
        {
        for (unsigned i = 0; i < size(); i++)
            {
            GATOM g(i);
            if( isTrue(g) )
                {
                const ATOM &a = g.getTheRealOne();
                if( strcmp(pname, a.getPredName()) == 0 )
                    {
                    atoms.push_back(a);
                    count++;
                    }
                }
            }
        }

    static void printSATlike(ostream &out)
        {
        for( size_t i=0; i < size(); i++ )
            out << i << '\t' << GATOM(i).getTheRealOne() << endl;
        }

    friend inline ostream& operator<< (ostream&, const GINTERPRET&);
    };


inline void GINTERPRET::addPositive( const GATOM& atom )
    {
    assert( array[atom.getIndex()] >= TRUE_FROM_MUST_BE_TRUE_V || 
            array[atom.getIndex()] == UNDEFINED_V );

    assign(atom.getIndex(),TRUE_FROM_UNDEFINED_V);
    }

inline void GINTERPRET::addFalse(const GATOM& atom)
    {
    assert( isUndefined(atom) );

    assign(atom.getIndex(),FALSE_V);
    }

inline void GINTERPRET::forceFalse(const GATOM &atom)
    {
    assign(atom.getIndex(),FALSE_V);
    }

inline void GINTERPRET::addMustBeTrue(const GATOM& atom)
    {
    assert( isUndefined(atom) );

    assign(atom.getIndex(),MUST_BE_TRUE_V);

    mbtCounter++;
    }

inline void GINTERPRET::addTrueFromUndefined(const GATOM& atom)
    {
    assert( isUndefined(atom) );

    assign(atom.getIndex(),TRUE_FROM_UNDEFINED_V);
    }

inline void GINTERPRET::addUndefined(const GATOM& atom)
    {
    if( isMustBeTrue(atom) )
        mbtCounter--;

    assign(atom.getIndex(),UNDEFINED_V);
    }


inline void GINTERPRET::addTrueFromMustBeTrue(const GATOM& atom)
    {
    assert( isMustBeTrue(atom) );

    assign(atom.getIndex(),TRUE_FROM_MUST_BE_TRUE_V);

    mbtCounter--;
    }

//////////////////////////////////////////////////////////////////////////////
inline void GINTERPRET::unionPositive(const GINTERPRET &I)
//
// Build the union of the positive parts.
    {
    for(size_t i=0; i < size(); i++)
        if( I.array[i] >= TRUE_FROM_MUST_BE_TRUE_V )
            array[i]=TRUE_FROM_UNDEFINED_V;
    }

//////////////////////////////////////////////////////////////////////////////

inline bool GINTERPRET::isTrue(const GATOM& atom) const
    {
    return ( array[atom.getIndex()] >= TRUE_FROM_MUST_BE_TRUE_V );
    }

inline bool GINTERPRET::isTrueFromUndefined(const GATOM& atom) const
    {
    return ( array[atom.getIndex()] == TRUE_FROM_UNDEFINED_V );
    }

inline bool GINTERPRET::isTrueFromMustBeTrue(const GATOM& atom) const
    {
    return ( array[atom.getIndex()] == TRUE_FROM_MUST_BE_TRUE_V );
    }

inline bool GINTERPRET::isUndefined(const GATOM& atom) const
    {
    return ( array[atom.getIndex()] == UNDEFINED_V );
    }

inline bool GINTERPRET::isFalse(const GATOM& atom) const
    {
    return ( array[atom.getIndex()] == FALSE_V );
    }

inline bool GINTERPRET::isMustBeTrue(const GATOM& atom) const
    {
    return ( array[atom.getIndex()] == MUST_BE_TRUE_V );
    }

inline bool GINTERPRET::isTrue(const GLITERAL& literal) const
    {
    if( literal.isNegative() )
        {
        if( isFalse(*static_cast<const GATOM*>(&literal)) )
            return true;
        else
            return false;
        }
    else
        return isTrue(*static_cast<const GATOM*>(&literal));
    }

inline bool GINTERPRET::isTrue(const GDISJUNCTION& disj) const
    {
    for( GDISJUNCTION::const_iterator i = disj.begin();
         i != disj.end();
         i++ )
        {
        if( isTrue(*i) )
            return true;
        }

    return false;
    }

inline bool GINTERPRET::isTrue(const GCONJUNCTION& conj) const
    {
    for( GCONJUNCTION::const_iterator i = conj.begin();
         i != conj.end();
         i++ )
        {
        if( ! isTrue(*i) )
            return false;
        }

    return true;
    }

inline bool GINTERPRET::isTotallyTrue(const GATOM& atom) const
    {
    return ( array[atom.getIndex()] >= TRUE_FROM_MUST_BE_TRUE_V 
             || array[atom.getIndex()] == MUST_BE_TRUE_V );
    }

inline bool GINTERPRET::isTotallyTrue(const GLITERAL& literal) const
    {
    // Truthtable:
    //
    // lit. negative | atom tot. true | literal tot. true
    // --------------------------------------------------
    //      true     |      true      |      false
    //      true     |      false     |      true
    //      false    |      true      |      true
    //      false    |      false     |      false
    // --------------------------------------------------
    //
    // => negated XOR => !=
    
    return( literal.isNegative() 
            !=  isTotallyTrue(*static_cast<const GATOM*>(&literal)) ); 
    }

inline bool GINTERPRET::isTotallyTrue(const GCONJUNCTION& conj) const
    {
    for( GCONJUNCTION::const_iterator j=conj.begin();
         j != conj.end();
         j++ )
        {
        if( ! isTotallyTrue(*j) )
            return false;
        }

    return true;
    }


inline bool GINTERPRET::isFalse(const GLITERAL& literal) const
    {
    if( ! literal.isNegative() )
        return isFalse(*static_cast<const GATOM*>(&literal));
    else
        return ( isTrue(*static_cast<const GATOM*>(&literal)) ||
                 isMustBeTrue(*static_cast<const GATOM*>(&literal)) );
    }

inline bool GINTERPRET::isFalse(const GCONJUNCTION& conj) const
    {
    for( GCONJUNCTION::const_iterator i = conj.begin();
         i != conj.end();
         i++ )
        {
        if( isFalse(*i) )
            return true;
        }

    return false;
    }

inline bool GINTERPRET::isFalse(const GDISJUNCTION& disj) const
    {
    for( GDISJUNCTION::const_iterator i = disj.begin();
         i != disj.end();
         i++ )
        {
        if( ! isFalse(*i) )
            return false;
        }

    return true;
    }

// Checks whether the interpretation does not contain any atoms which
// are MUST_BE_TRUE. This is equivalent to checking whether a partial
// interpretation can be totalised without generating an inconsistency.
inline bool GINTERPRET::isMustBeTrueFree() const
    {
    return ( mbtCounter == 0);
    }

//////////////////////////////////////////////////////////////////////////////
inline ostream& operator<< (
//
    ostream          &out, 
    const GINTERPRET &I )
    {
    out << "< TRUE (FROM UNDEFINED) = {";

    I.printTruthValue(out,TRUE_FROM_UNDEFINED_V);

    out << "}, TRUE (FROM MUST_BE_TRUE) = {";

    I.printTruthValue(out,TRUE_FROM_MUST_BE_TRUE_V);

    out << "}, MUST_BE_TRUE = {";

    I.printTruthValue(out,MUST_BE_TRUE_V);

    out << "}, UNDEFINED = {";

    I.printTruthValue(out,UNDEFINED_V);

    out << "}, FALSE = {";

    I.printTruthValue(out,FALSE_V);

    out << "} >";

    if( ! I.getCost().isZero() )
        {
        out << endl << "  Cost ([Weight:Level]): ";
        I.printCost(out); 
        }

    return out;
    }

#endif

// Local Variables:
// mode: c++
// c-file-style: "dl"
// End:
