//////////////////////////////////////////////////////////////////////////////
// grounding-dictionary.h

#if HAVE_GCC_VERSION(3,1)
    #include <ext/hash_set>
    using namespace __gnu_cxx;
    #define HAVE_HASHSET
#elif HAVE_GCC_VERSION(2,95)
    #include <hash_set>
    #define HAVE_HASHSET
#else
    #error "I do not know about this compiler yet!"
    #include <set>
#endif

#ifdef HAVE_HASHSET

namespace
#if HAVE_GCC_VERSION(3,1)
    __gnu_cxx
#else
    std
#endif
{

template<>
struct hash< pair <unsigned,unsigned> >
    {
    size_t operator()(const pair <unsigned,unsigned>& p) const
       {
        hash<unsigned> unsignedHasher;
        size_t h = unsignedHasher(p.first+p.second);
         
        return h;
        }
    };

}

#endif

// For each predicate in the input program, DBDICTIONARY contains the 
// selectivities of all the parameters of the predicate. 
// The selectivities are stored in a matrix of as many rows as the
// number of the predicates in the program and of n_par columns.
class DBDICTIONARY
//////////////////////////////////////////////////////////////////////////////
    {
    /// The number of predicates in the program.
    unsigned SIZE;
    
    /// An array of "SIZE" elements. 
    /// alreadyComputed[i] is true if the selectivity the 
    /// i-th predicate has been already computed.
    bool *alreadyComputed;
    
    /// Stores the selectivity of the arguments of all predicates. 
    /// We compute the selectivity only for the EDB atoms and for
    /// the atoms wich are done. In the other cases we estimate it
    double **dictionary;

    /// Stores pair of indices. A pair <idxConst,idxVar> is inserted 
    /// in the set if the constant Const is a possible value for the 
    /// variable Var in the extension of a certain atom.
    /// It has to be cleaned for each atom in the body.	
                                    
    hash_set< pair<unsigned,unsigned> > constantsSet;


 private:
    
    void computeSelectivity(const ATOM & , const bool *, const unsigned);
    void computeSelectivity(const ATOM &, ATOMSET::FIND_RESULT &);
    void initialiseDictionary();
    
 public:
    
    double getSelectivity(const ATOM &, 
                          const bool *, 
                          unsigned arg,
                          const unsigned); 
    ~DBDICTIONARY();
    DBDICTIONARY();
    DBDICTIONARY (const DBDICTIONARY &);
    void operator= (const DBDICTIONARY &);
    };

DBDICTIONARY::DBDICTIONARY()
    : SIZE(ATOM::Predicates.size())
    {
    alreadyComputed = new bool [SIZE];
    dictionary = new double *[SIZE];
    for (unsigned i = 0; i < SIZE ;i++)
        dictionary[i] = new double[n_par];
    initialiseDictionary();
    }

DBDICTIONARY::DBDICTIONARY(const DBDICTIONARY &)
    {
    assert(0);
    }

void DBDICTIONARY::operator=(const DBDICTIONARY &)
    {
    assert(0);
    }

DBDICTIONARY::~DBDICTIONARY()
    {
    for (unsigned i = 0; i < SIZE ; i++)
        delete[] dictionary[i];
    delete[] dictionary;
    delete[] alreadyComputed;
    }

/// Examines all the possible instances of an atom "atom".
/// Fill the dictionary with the selectivity (i.e. the number
/// of possible different values) of each variable of "atom".
/// Use the set in order to compute these numbers in an efficient 
/// way.
void DBDICTIONARY::computeSelectivity(
    const ATOM &atom,
    ATOMSET::FIND_RESULT &table) 
    {
    while (!(table).atEnd())
        {
        if((*table).getParams())
            { 
            const TERMS &param = *((*table).getParams());
            for( unsigned i = 0; i < param.size(); i++ )
                {
                if ( (*(atom.getParams()))[i].isVar() )	
                    {
                    pair< unsigned,unsigned > p(param[i].getIndex(),i);
                    pair< hash_set<pair<unsigned,unsigned> >::const_iterator,bool>
                        result = constantsSet.insert(p);
                    if( result.second )
                        dictionary[atom.getIndex()][i]++;
                    }
                }   
            }       
        table++;
        }
    }

/// Given an atom, if it is EDB or done, using the function
/// computeSelectivity( const ATOM &, ATOMSET::FIND_RESULT &), compute the
/// selectivity of its parameters.
/// In all other cases, it is not possible to calculate
/// the selectivity and we estimate it. 
void DBDICTIONARY::computeSelectivity(
    const ATOM &atom, 
    const bool *done,
    const unsigned card)
    { 
    if (atom.isEDB())
        { 
        // If the atom is an EDB we look it up in the EDB container.
        ATOMSET::FIND_RESULT tableEDB; 
        EDB.find(atom, tableEDB);
        // We compute its selectivity.
        computeSelectivity(atom, tableEDB); 
        alreadyComputed[atom.getIndex()] = true;
        }
    else
        // At the moment we compute the selectivity only for the predicates
        // which belong to the previous components.
        if (done && done[atom.getIndex()])
            {       
            // We look for the atom in the Undefined Part of 
            // the interpretation and in the positive part.
            ATOMSET::FIND_RESULT tableUndefined;
            I.findInUndefinedPart(atom, tableUndefined);
            computeSelectivity(atom, tableUndefined); 
            ATOMSET::FIND_RESULT tableDefined;
            I.findInPositivePart(atom, tableDefined);
            computeSelectivity(atom, tableDefined); 
            alreadyComputed[atom.getIndex()] = true;
            }
        else 
            {
            double selectivity = 
                pow(double(card),double(1)/double(atom.getArity()));
            fill_n(dictionary[atom.getIndex()], n_par, selectivity);
            // We do not cache the estimated selectivity for use in further
            // components, but will try to recompute it there using exact
            // data instead of estimates.
            }
    constantsSet.clear();
    }

void  DBDICTIONARY::initialiseDictionary()
    {
    fill_n(alreadyComputed, SIZE, false);
    for( unsigned i = 0; i < SIZE; i++ )
        fill_n(dictionary[i], n_par, 0);
    }

// Returns the selectivity of a variable of a predicate.
// Use the method computeSelectivity.
double DBDICTIONARY::getSelectivity(const ATOM &atom, 
                                    const bool *done,
                                    unsigned arg,
                                    const unsigned card)  
    {
    // If the argument is a constant its selectivity is always 1.
    if( ! (*(atom.getParams()))[arg].isVar())
        return 1;

    // If we already computed the selectivity of a variable,
    // we don't calculate it again.
    if( !alreadyComputed[atom.getIndex()] )
        computeSelectivity(atom,done,card);

    return  dictionary[atom.getIndex()][arg];
    }
// Local Variables:
// mode: c++
// c-file-style: "dl"
// End:
