//////////////////////////////////////////////////////////////////////////////
// interpret.h 

#ifndef INTERPRET_H
#define INTERPRET_H

#include "atomset.h"

//////// Friends of VATOMSET.

class INTERPRET;
class OID;
void InheritanceRewritingRules(const OID &);

//////////////////////////////////////////////////////////////////////////////
class VATOMSET
//
// This is similiar to ATOMSET in that it stores a set of ATOMs and
// implements most of the interfaces of ATOMSET. However, it is supposed to
// be more efficient for larger numbers of ATOMs, as we use one separate
// ATOMSET for every predicate.
// The allocation of these ATOMSETs is performed lazily (that is, only when
// needed by some other member function) by invoking expand().
//
    {
    vector<ATOMSET> v;

    void expand() const
        {
        size_t N=ATOM::Predicates.size();

        if( v.size() < N )
            {
            VATOMSET *i = const_cast<VATOMSET*> (this);

            i->v.insert(i->v.end(),N-v.size(),ATOMSET());
            }
        }

public:
    VATOMSET()
        : v()
        {
        }

    VATOMSET(const VATOMSET &I2) 
        : v(I2.v)
        {
        }

    void operator= (const VATOMSET&)
        {
        assert( 0 );
        }

    void add(const ATOM &atom)
        {
        expand();

        assert( atom.getIndex() < v.size() );

        v[atom.getIndex()].add(atom);
        }

    bool contains(const ATOM &atom) const 
        {
        if( atom.getIndex() < v.size() )
            return v[atom.getIndex()].contains(atom);
        else
            return false;
        }

    void erase(const ATOM &atom)
        {
        if( atom.getIndex() < v.size() )
            v[atom.getIndex()].erase(atom);
        }

    void erasePred(const ATOM &atom)
        {
        if( atom.getIndex() < v.size() )
            v[atom.getIndex()].clear(); 
        }

    void clear()
        {
        for( vector<ATOMSET>::iterator i=v.begin(); i != v.end(); i++ )
            (*i).clear();
        }

    size_t size() const 
        {
        size_t s=0;

        for( vector<ATOMSET>::const_iterator i=v.begin();
             i != v.end();
             i++ )
            s+=i->size();

        return s;
        }

    template <class F>
    void foreach(F fn) const
        {
        for( vector<ATOMSET>::const_iterator i=v.begin(); i != v.end(); i++)
            {
            for( ATOMSET::const_iterator j=(*i).begin();
                 j != (*i).end();
                 j++)
                {
                fn(*j);
                }
            }
        }

    bool printCensored(ostream &out, bool continuing) const
        {
        for( vector<ATOMSET>::const_iterator i=v.begin();
             i != v.end();
             i++)
            {
            continuing=(*i).printCensored(out,continuing) || continuing;
            }

        return continuing;
        }

    void find(const ATOM &pattern, ATOMSET::FIND_RESULT &f ) const
        {
        if( pattern.getIndex() < v.size() )
            v[pattern.getIndex()].find(pattern, f);
        else
            {
            // We do not know about this predicate yet, so we certainly
            // won't have any match.  Provide an "empty" FIND_RESULT.
            static ATOMSET empty;

            empty.find(pattern,f);
            }
        }

    ///////////////////////////////////////////////////////////////////
    void exportMatching(
    // Copy all ATOMs which are true in the current interpretation into
    // the vector atoms and increase count accordingly.
    //
        const char *pname,
        vector<ATOM> &atoms,
        unsigned &count ) const
        {
        // FIXME: Given the internal structure of VATOMSET, this could
        // be done more efficiently (or it could use find()).
        for(vector<ATOMSET>::const_iterator k = v.begin(); k != v.end(); k++)
            {
            for(set<ATOM>::const_iterator i = k->begin();
                i != k->end(); i++ )
                {
                if(strcmp(i->getPredName(), pname) == 0)
                    {
                    atoms.push_back(*i);
                    count++;
                    }
                }
            }
        }

    friend class INTERPRET;
    friend void InheritanceRewritingRules(const OID &);
    };

//////////////////////////////////////////////////////////////////////////////
class INTERPRET
//////////////////////////////////////////////////////////////////////////////
    {
    VATOMSET positive;
    VATOMSET undefined;

public:
    INTERPRET() 
        : positive(), undefined()
        {
        }

    INTERPRET(const INTERPRET &I2) 
        : positive(I2.positive), undefined(I2.undefined)
        {
        }

    void operator= (const INTERPRET&)
        {
        assert( 0 );
        }

    // Essentially, all other functions that check or set values of an
    // interpretation should do so by means of the following four functions,
    // never directly.
    inline bool isPositive(const ATOM&) const;
    inline bool isUndefined(const ATOM&) const;
    inline void addPositive(const ATOM&, const bool=true);
    inline void addUndefined(const ATOM&);
    
    inline void clearPositivePart();
    inline void clearUndefinedPart();
    inline void clear();

    inline bool isTrue (const ATOM&) const;
    inline bool isTrue (const LITERAL&) const;
    inline bool isTrue (const DISJUNCTION&) const;
    inline bool isTrue (const CONJUNCTION&) const;
    inline bool isFalse(const ATOM&) const;
    inline bool isFalse(const LITERAL&) const;
    inline bool isFalse(const CONJUNCTION&) const;


    bool isPositivePartEmpty() const
        {
        return positive.size() == 0;
        }
    
    void findInPositivePart(const ATOM &pattern, 
                            ATOMSET::FIND_RESULT &f) const
        {
        positive.find(pattern,f);
        }

    void findInUndefinedPart(const ATOM &pattern,
                             ATOMSET::FIND_RESULT &f) const
        {
        undefined.find(pattern,f);
        }

    template <class F>
    void foreachPositive(F fn) const
        {
        positive.foreach(fn);
        }

    template <class F>
    void foreachUndefined(F fn) const
        {
        undefined.foreach(fn);
        }

    bool printCensored(ostream &out, bool continuing) const
        {
        return positive.printCensored(out,continuing);
        }
    };

//////////////////////////////////////////////////////////////////////////////

inline bool INTERPRET::isPositive(
    const ATOM &atom ) const
    {
    return positive.contains(atom);
    }
   
inline bool INTERPRET::isUndefined(
    const ATOM &atom ) const
    {
    return undefined.contains(atom);
    }
   
inline void INTERPRET::addPositive(
    const ATOM& atom, 
    const bool  checkConsistency )
    {
    if( checkConsistency )
        assert( isPositive(atom) || isUndefined(atom) );

    positive.add(atom);
    undefined.erase(atom);
    }

inline void INTERPRET::addUndefined(const ATOM& atom)
    {
    assert( ! isPositive(atom) );

    undefined.add(atom);
    }


//////////////////////////////////////////////////////////////////////////////

inline void INTERPRET::clearPositivePart()
    {
    positive.clear();
    }

inline void INTERPRET::clearUndefinedPart()
    {
    undefined.clear();
    }

inline void INTERPRET::clear()
    {
    clearPositivePart();
    clearUndefinedPart();
    }

//////////////////////////////////////////////////////////////////////////////

inline bool INTERPRET::isTrue(const ATOM& atom) const
    {
    return isPositive(atom);
    }

inline bool INTERPRET::isTrue(const LITERAL& literal) const
    {
    if( literal.isNegative() )
        {
        if( isTrue((ATOM)literal)  ||  isUndefined((ATOM)literal) )
            return false;
        else
            return true;
        }
    else
        return isTrue((ATOM)literal);
    }

inline bool INTERPRET::isTrue(const DISJUNCTION& disj) const
    {
    for( DISJUNCTION::const_iterator i = disj.begin();
         i != disj.end();
         i++ )
        {
        if( isTrue(*i) )
            return true;
        }

    return false;
    }

inline bool INTERPRET::isTrue(const CONJUNCTION& conj) const
    {
    for( CONJUNCTION::const_iterator i = conj.begin();
         i != conj.end();
         i++ )
        {
        if( ! isTrue(*i) )
            return false;
        }

    return true;
    }


inline bool INTERPRET::isFalse(const ATOM& atom) const
    {
    if( isTrue(atom) )
        return false;
    else if( isUndefined(atom) )
        return false;
    else
        return true;
    }
       
inline bool INTERPRET::isFalse(const LITERAL& literal) const
    {
    if( ! literal.isNegative() )
        return isFalse((ATOM)literal);
    else
        return isTrue((ATOM)literal);
    }

inline bool INTERPRET::isFalse(const CONJUNCTION& conj) const
    {
    for( CONJUNCTION::const_iterator i = conj.begin();
         i != conj.end();
         i++ )
        {
        if( isFalse(*i) )
            return true;
        }

    return false;
    }

//////////////////////////////////////////////////////////////////////////////

class AtomPrinter : public unary_function<ATOM&,void>
    {
    ostream &out;
public:
    AtomPrinter(ostream &out2)
        : out(out2)
        {
        }

    void operator() (const ATOM &a) const
        {
        out << a << ' ';
        }
    };

//////////////////////////////////////////////////////////////////////////////
inline ostream& operator<< (
//
    ostream        &out,
    const VATOMSET &atomset )
    {
    out << "{";
    atomset.foreach( AtomPrinter(out) );
    out << "}";

    return out;
    }

//////////////////////////////////////////////////////////////////////////////
inline ostream& operator<< (
//
    ostream         &out,
    const INTERPRET &interpret )
    {
    out << "< {";
    interpret.foreachPositive( AtomPrinter(out) );
    out << "} , {";
    interpret.foreachUndefined( AtomPrinter(out) );
    out << "} >";

    return out;
    }

#endif

// Local Variables:
// c-file-style: "dl"
// End:
