//////////////////////////////////////////////////////////////////////////////
// names.h 

#ifndef NAMES_H
#define NAMES_H

template <class T>
class NAMESTABLE
    {
public:
    typedef size_t index_t;

private:
    typedef map<T,index_t> lookup_t;
    typedef vector<typename lookup_t::const_iterator> indexed_t;

    lookup_t lookup;
    indexed_t indexed;
    size_t indexcount;

public:

    class const_iterator
        {
        typename lookup_t::const_iterator it;

    public:
        const_iterator()
            {
            assert( 0 );
            }

        const_iterator(const typename lookup_t::const_iterator &it1)
            : it(it1)
            {
            }

        const_iterator(const const_iterator& ni1)
            : it(ni1.it)
            {
            }

        const_iterator& operator=(const const_iterator& ni1)
            {
            if( this != &ni1 )
                {
                it = ni1.it;
                }
            return *this;
            }

        ~const_iterator()
            {
            }

        const T& operator*() const
            {
            return (*it).first;
            }

        void operator++()
            {
            it++;
            }

        void operator++(int)
            {
            ++it;
            }

        bool operator== (const const_iterator& i2) const
            {
            return it == i2.it;
            }

        bool operator!= (const const_iterator& i2) const
            {
            return it != i2.it;
            }

        };

    NAMESTABLE(index_t startindex=0) 
        : lookup(), indexed(), indexcount(startindex)
        {
        }

    NAMESTABLE(const NAMESTABLE<T>& n2)
        : lookup(n2.lookup), indexed(n2.indexed), indexcount(n2.indexcount)
        {
        // These tables should not be copied.
        assert(0);
        }

    NAMESTABLE& operator=(const NAMESTABLE<T>& n2)
        {
        // These tables should not be copied.
        assert(0);

        if( this != &n2 )
            {
            lookup = n2.lookup;
            indexed = n2.indexed;
            indexcount = n2.indexcount;
            }

        return *this;
        }

    const_iterator begin() const
        {
        return const_iterator(lookup.begin());
        }

    const_iterator end() const
        {
        return const_iterator(lookup.end());
        }

    size_t size() const
        {
        return indexed.size();
        }

    // @return a pair: The second item indicates whether an entry has been 
    // found; the first item contains the index of the entry if one has been
    // found, 0 otherwise.
    pair<index_t,bool> find(const T &item) const
        {
        typename lookup_t::const_iterator i = lookup.find(item);

        if( i == lookup.end() )
            return pair<index_t,bool>(0,false);
        else
            return pair<index_t,bool>((*i).second,true);
        }

    pair<index_t,bool> add(const T &item)
        {
        index_t itemindex;

        // Try to insert the item with the next available index (if a
        // copy of item already exists, the index is not overwritten!)
        pair<typename lookup_t::const_iterator,bool> result = lookup.insert(pair<T,index_t>(item,indexcount));

        // See whether this item already existed in the datastructures.
        if(result.second)
            {
            // This is a new item.
            itemindex = indexcount++;
            indexed.push_back(result.first);
            }
        else
            {
            // The item already existed.
            itemindex = (*(result.first)).second;
            }

        return pair<index_t,bool>(itemindex,result.second);
        }

    const T& getItem(index_t index) const
        {
	assert(index < indexed.size());
	return (*(indexed[index])).first;
        }

    pair<index_t,bool> getIndex(const T &item) const
	{
	typename lookup_t::iterator i = lookup.find(item);
	if( i == lookup.end() )
	    return pair<index_t,bool>(0,false);
	else
	    return pair<index_t,bool>((*i).second,true);
	}
    };


//////////////////////////////////////////////////////////////////////////////
class NAMES_ITEM
//
    {
    char *name;

public:
    NAMES_ITEM()
        : name(0)
        {
        assert( 0 );
        }

    NAMES_ITEM(const NAMES_ITEM& item2)
        {
        size_t length=strlen(item2.name);

        name=new char[length+1];
        memcpy(name,item2.name,length+1);
        }

    NAMES_ITEM(const char *name2)
        {
        size_t length=strlen(name2);

        name=new char[length+1];
        memcpy(name,name2,length+1);
        }

    ~NAMES_ITEM()
        {
        delete[] name;
        }

    void operator=(const NAMES_ITEM&)
        {
        assert( 0 );
        }

    bool operator==(const NAMES_ITEM &n) const
        {
        return strcmp(name,n.getName()) == 0;
        }

    bool operator!=(const NAMES_ITEM& n) const
        {
        return strcmp(name,n.getName()) != 0;
        }

    bool operator==(const char* s) const
        {
        assert(s);
        return (strcmp(name,s) == 0);
        }

    const char *getName() const
        {
        return name;
        }

    /** 
     * @return whether this name designates a positive predicate.
     */
    bool isPositive() const
	{
	return ! isNegative();
	}

    /** checks for true negation.
     * @return whether this name designates a (truly) negative predicate.
     */
    bool isNegative() const
	{
	return name[0]=='-';
	}

    /** provides the positive form of this name.
     * @return the name without any leading negation symbols.
     */
    const char *getPositiveName() const
	{
	if (isNegative())
	    return name+1;
	else
	    return name;
	}

    /** provides the negative form of this name.  This is relatively
     *  inefficient compared to getPositiveName() and should be avoided
     *  where possible.
     * @return the name with a leading negation symbol. The name is allocated
     *         in any case and should be deallocated by the caller.
     */
    const char *getNegativeName() const
	{
        char *temp;

	if (isPositive())
            {
            temp = new char[strlen(name)+2];
            temp[0] = '-';
            strcpy(temp+1,name);
            }
	else
            {
            temp = new char[strlen(name)+1];
            strcpy(temp,name);
            }
	return temp;
	}

    /** check if predicates of the given name should be censored.
     * Uses globals PredFilter and PredPFilter that specify the wanted
     * predicates. Censorship takes place if the name is not in PredFilter,
     * and PredPFilter also does not apply (because name is not mentioned,
     * or is truly negative).
     *
     * PREDICATE NAMESPACE:
     * ===================
     * Names starting with "\"_" are generated by the rewriting pass, those
     * starting with "aux#" and "#" represent auxiliary predicates generated
     * by the parser to support aggregates.
     * The inheritance frontend uses names starting with "^", "@", and "__".
     * The diagnosis frontend uses names starting with "_".
     * The planning frontend uses names starting with "%".
     *
     * None of the above should be printed as part of the regular output.
     *
     * @param name the name to check.
     * @return whether the name should be censored.
     */
    bool isCensored() const
	{
        if( getName()[0] == '\"'  &&  getName()[1] == '_' )
            return true;

        if( strncmp(getName(),AUXNAME,4) == 0  || getName()[0] == '#' )
            return true;

        if( getName()[0] == '^' || getName()[0] == '@' || getName()[0] == '_' )
            return true;

        if( PredFilter.empty() && PredPFilter.empty() )
            return false;

	return PredFilter.find(getPositiveName())==PredFilter.end()
	    && (isNegative() || PredPFilter.find(name)==PredPFilter.end()); 
	}

    void printDebug(ostream&) const;
    };

inline bool operator<(const NAMES_ITEM& n1, const NAMES_ITEM& n2)
    {
    return (strcmp(n1.getName(),n2.getName()) < 0);
    }

inline ostream& operator<< (ostream& out, const NAMES_ITEM& item)
    {
    return out << item.getName();
    }

inline void NAMES_ITEM::printDebug(ostream &out) const
    {
    out << '\"' << name << '\"'
        << " (" << (const void*)this << ":" << (const void*)name << ") ";
    }

//////////////////////////////////////////////////////////////////////////////

#ifdef HAVE_HASHMAP

namespace
#if HAVE_GCC_VERSION(3,1)
    __gnu_cxx
#else
    std
#endif
{

template<>
struct hash<NAMES_ITEM>
    {
    size_t operator()(const NAMES_ITEM& x) const
        {
        hash<const char*> H;
        return H(x.getName());
        }
    };

}

#endif

//////////////////////////////////////////////////////////////////////////////

typedef NAMESTABLE<NAMES_ITEM> STRING_CONSTANTS;
typedef NAMESTABLE<int> INTEGER_CONSTANTS;

//////////////////////////////////////////////////////////////////////////////
class PREDICATE_NAMES 
//
    {
public:
    enum TYPE {
        typeUndef=0,
        typeEDB, typeIDB, typeBuiltin, typeAggregate,
        typeQuery
    };

    typedef NAMESTABLE<NAMES_ITEM>::const_iterator const_iterator;
    typedef NAMESTABLE<NAMES_ITEM>::index_t index_t;

private:
    NAMESTABLE<NAMES_ITEM> names;
    vector<TYPE> types;
    vector<unsigned> arities;

public:
    PREDICATE_NAMES(unsigned i=0) 
        : names(i), types(), arities()
        {
        // Standard constructor.
        }

    PREDICATE_NAMES(const PREDICATE_NAMES& pn2)
        : names(pn2.names), types(pn2.types), arities(pn2.arities)
        {
        // There is no good reason why this structure should be copied.
        assert(0);
        }

    PREDICATE_NAMES& operator=(const PREDICATE_NAMES& pn2)
        {
        // There is no good reason why this structure should be copied.
        assert(0);

        if( this != &pn2 )
            {
            names = pn2.names;
            types = pn2.types;
            arities = pn2.arities;
            }

        return *this;
        }

    ~PREDICATE_NAMES()
        {
        // Standard destructor.
        }

    pair<index_t,bool> add(const char *name, unsigned arity, TYPE type )
        {
        pair<index_t,bool> addresult = names.add(name);

        index_t &idx = addresult.first;

        if(addresult.second)
            {
            // A new name.
            assert( idx == arities.size() );
            assert( idx == types.size() );

            arities.push_back(arity);
            types.push_back(type);
            }
        else
            {
            // An existing name.
            // Check whether arity matches.
            if( arities[idx] != arity )
                // If not, indicate an error.
                return pair<index_t,bool>(idx,false);

            // Check whether types match.
            // typeUndef can be replaced by any other type.
            if( type != typeUndef && types[idx] != type )
                {
                // Builtins should never be redefined.
                assert( types[idx] != typeBuiltin );
                assert( types[idx] != typeAggregate );

                if( types[idx] == typeUndef )
                    // typeUndef can be replaced by any other type.
                    types[idx] = type;
                else
                    // everything else is currently an error.
                    return pair<index_t,bool>(idx,false);
                }
            }

        return pair<index_t,bool>(idx,true);
        }

    unsigned getArity(index_t index) const
        {
        assert(index < arities.size());
        return arities[index];
        }

    bool isEDB(index_t index) const
        {
        assert(index < types.size());
        return (types[index] == typeEDB);
        }

    bool isIDB(index_t index) const
        {
        assert(index < types.size());
        return (types[index] == typeIDB);
        }

    bool isUndef(index_t index) const
        {
        assert(index < types.size());
        return (types[index] == typeUndef);
        }

    bool isBuiltin(index_t index) const
        {
        assert(index < types.size());
        return (types[index] == typeBuiltin);
        }
    
    bool isAggregate(index_t index) const
        {
        assert(index < types.size());
        return (types[index] == typeAggregate);
        }

    TYPE getType(index_t index) const
        {
        assert(index < types.size());
        return types[index];
        }

    void print(index_t index, ostream& o) const
        {
        o << names.getItem(index);

        if( PTraceLevel >= 1 )
            switch( types[index] )
                {
                case typeEDB:
                    o << "/E";
                    break;
                case typeIDB:
                    o << "/I";
                    break;
                case typeBuiltin:
                    o << "/B";
                    break;
                case typeAggregate:
                    o << "/A";
                    break;
                default:
                    o << "/*";
                }

        }

    const NAMES_ITEM& getItem(index_t index) const
        {
        assert(index < names.size());

        return names.getItem(index);
        }

    // Change the type of this predicate. Return true if this definitely
    // does not require some changes to other structures, false else.
    bool refineType(const index_t index, const TYPE newtype)
        {
        assert( index < names.size() );

        TYPE &type = types[index];

        assert( type != typeBuiltin );
        assert( type != typeAggregate );

        if( type == newtype )
            return true;
        else if( type == typeUndef)
            {
            type=newtype;
            return true;
            }
        else
            {
            type=newtype;
            return false;
            }
        }

    void refineArity(const index_t index, const unsigned newArity)
        {
        arities[index] = newArity;
        }

    pair<index_t,bool> find(const NAMES_ITEM &n) const
        {
        return names.find(n);
        }

    pair<index_t,bool> find(const char *ns) const
        {
        return find(NAMES_ITEM(ns));
        }

    size_t size() const
        {
        return names.size();
        }

    
    const_iterator begin() const
        {
        return names.begin();
        }

    const_iterator end() const
        {
        return names.end();
        }
    };

#endif

// Local Variables:
// mode: c++
// c-file-style: "dl"
// End:
