//////////////////////////////////////////////////////////////////////////////
// plan.C

#define WITH_GLOBAL_VARIABLES
#define WITH_DEPGRAPH

#include <stdio.h>
#include <sstream>
#include "dl.h"
#include "stats.h"
#include "plan.h"
#include "generate.h"

//////////////////////////////////////////////////////////////////////////////

PREDICATE_NAMES fluents(0);
PREDICATE_NAMES actions(0);
unsigned maxActionArity = 0;
unsigned minActions = 0;
unsigned maxActions = 0;
int maxPlanLength = -1;
int maxPlanLengthParsed = -1;
unsigned planCacheSize = PLANCACHE;
unsigned planCacheHit = 0;
unsigned planCheckCounter = 0;

unsigned FPcheck = 0;
int FPsoundcheck = -1;
int FPcompletecheck = -1;

KLITERALS *Goal = 0;
KRULES krules;
KRULES declarations;

// Does this program use action costs?
bool costsUsed = false;
// Does this program levels for action costs?
bool costLevelsUsed = false;


static GATOMSET *fluentGatoms;
static GATOMSET *actionGatoms;
static GATOMSET *costGatoms;

// This is used to do the mux-strat-check:
static vector< GRULES >  causationRules;
static vector< vector< vector< unsigned long > > > mutexLists;

// Does this program have positive cyclic executability conditions?
static bool execCyclic = false;

static PlanCache collectedPlans[2];
static set<GATOMSET> collectedPrintedPlans;

//////////////////////////////////////////////////////////////////////////////
// Helper functions as expected by LEX/FLEX

extern "C" int planyywrap()    // End-of-file handler for LEX
    {
    return 1;
    }

int planyyerror(const char *s)
    {
    return yyerror(s);
    }
 
#include "plan_parser2.c"

//////////////////////////////////////////////////////////////////////////////
static int getLastParamAsInt( const ATOM &atom )
    {
    const TERMS *params=atom.getParams();

    assert( params );
    assert( params->size() >= 1 );
 
    return ((*params)[params->size()-1]).getInt();
    } 

//////////////////////////////////////////////////////////////////////////////
static void extractPlanFromGInterpret( const GINTERPRET &I, GATOMSET &thePlan)
    {
    for(unsigned i=0; i < GATOMSET::getSize(); i++)
        {
        
        GATOM a(i);
        
        if( I.isTrue(a) && actionGatoms->contains(a) )
            {
            thePlan.add(a);
            }
        }
    }

//////////////////////////////////////////////////////////////////////////////
static bool executeCheck( 
//
    GRULES &rules, 
    GCONSTRAINTS &constraints  )
    {
    if( FTraceLevel >= 1 )
        {
        cdebug << "Checking the plan by means of " << "\n  ";
        print_list(cdebug, rules, "\n  ");
        if( rules.size() > 0  &&  constraints.size() > 0 )
            cdebug << "\nand\n  ";
        print_list(cdebug, constraints, "\n  ");
        cdebug << endl;
        }
    
    GWEAKCONSTRAINTS dummy_weakconstraints;
    MODEL_GENERATOR mg1(rules,constraints,
                       dummy_weakconstraints,
                       0,0,0,
                       1,                // One model only.
                       0,                // Minimum cost bound.
                       static_cast<MODEL_GENERATOR::FLAGS>(
                           MODEL_GENERATOR::PERFORM_GUS
                           | MODEL_GENERATOR::PERFORM_CHECK),
                       0,0,false,0,0,0,0,0,0,
                       AGGREGATEATOM::getAggregateFunctionsCount(),
                       HeuristicSequence, HeuristicCombinationSequence
                       );
    mg1.run();

    if( mg1.getNumberOfModelsComputed() == 0 )
        {
        return true;
        }
        
    return false;
    }

//////////////////////////////////////////////////////////////////////////////
static bool checkPlan(
//
    const GINTERPRET   &I,
    const GRULES       &origRules,
    const GCONSTRAINTS &origConstraints,
    unsigned checkMethod = FPcheck
    )
    {

    planCheckCounter++;  

    const GATOM pred_goal(ATOM(PRED_GOAL,0));
    const GATOM pred_abs(ATOM(PRED_ABS,0));
    
    GATOMSET currentPlan;

    extractPlanFromGInterpret( I, currentPlan );
 
    const pair<bool,bool> lookup = 
        collectedPlans[checkMethod].lookup(currentPlan);

    if ( lookup.first )
        {
        // Plan already checked and found to be secure.
        planCacheHit++;
        return lookup.second;
        }
    
    GRULES       rules;
    GCONSTRAINTS constraints; 

    // Add rules and constraints for the plan found
    // which are necessary.
    for( unsigned i=0; i < GATOMSET::getSize(); i++ )
        {
        if( currentPlan[i] )
            {
            GATOM a(i);
            ////////
            // Add rule $abs :- not a(t).
            // to guarantee that the actions in the plan are always executable.
            
            GDISJUNCTION head;
            head.add( pred_abs );

            GCONJUNCTION body;
            body.add( GLITERAL(true,a) );

            GRULE rule(&head,&body);           
            rules.push_back(rule);

            if( FTraceLevel >= 1 )
                cdebug << "Adding " << rule << endl;
            }
        }
    
	
    // Copy origRules to rules, but ignore those rules that have an
    // action in the head or positive body, which is false in the current 
    // interpretation.
    for( GRULES::const_iterator i=origRules.begin();
         i != origRules.end();
         i++ )
        {
        bool copy=true;

        // Do not copy rules with an action as the 
        // first element of the head (this is an action guess!).
        if( actionGatoms->contains(*(i->getHead().begin())) )
            {
            copy=false;
            
            // If this action is in the current plan,
            // add rule 
            //   a :- body.
            // instead of the guessing rule
            //   a v -a :- body.
            if( I.isTrue(*(i->getHead().begin())) )
                {
                GDISJUNCTION head1;
                head1.add( *(i->getHead().begin()) );

                if( FTraceLevel >= 1 )
                    cdebug << "Drop negative part of disjunction in rule: " 
                           << *i << endl;
                
                rules.push_back( GRULE(&head1, i->getBody()) );                
                }
            }

        // Do not copy rules with an action 
        // 1) in the pos. body which is not in the current plan or 
        // 2) with an action in the neg. body, which IS in the 
        //    current plan.
        else if( i->getBody() )
            {
            for( GCONJUNCTION::const_iterator j = i->getBody()->begin();
                 j != i->getBody()->end();
                 j++ )
                {
                if( I.isFalse(*j) && actionGatoms->contains(*j) ) 
                    {
                    copy = false;
                    break;
                    }
                }
            }
        
        if( copy )
            rules.push_back(*i);
        else
            if( FTraceLevel >= 1 )
                cdebug << "Removing " << *i << endl;
        }
    
    bool goalTriviallyTrue = true;

    // Copy origConstraints to constraints, except
    // those constraints which have an action literal body, 
    // which is false in the current interpretation!
    // Also ignore constraints with aggregates, which could only
    // arise from constrainNumberOfActions().
    for( GCONSTRAINTS::const_iterator i = origConstraints.begin();
         i != origConstraints.end();
         i++ )
        {
        if( i->begin()->getAtom() == pred_goal )
            {
            // Remove goal constraint ":- not $goal."
            if( FTraceLevel >= 1 )
                cdebug << "Removing old goal constraint:  " 
                       << *i << endl;

            // If the goal is trivially true, then the goal
            // constraint will be removed before model generation.
            goalTriviallyTrue = false;
            }
        else if( i->begin()->getAtom() == pred_abs ) 
            {
            // Secure Check 2: Remove constraint: ":- $abs."
            if( FTraceLevel >= 1 )
                cdebug << "Removing constraint:  " 
                       << *i << endl;
            }
        else if( i->begin()->getAtom().isAggregate() )
            {
            // Also ignore constraints with an aggregate
            // which could only arise from constrainNumberOfActions().
            // as aggregates are forbidden elsewhere.
            // FIXME: If ever we allow aggregates in the background 
            //        knowledge, this has to be modified!
            }
        else
            {
            // CHECK 1:
            // In general: change all other constraints 
            //   :- l1, ..., ln
            // to rule:
            // %abs :- l1, ..., ln.
            // Exceptions: 
            // 1) initial state constraints should 
            //    not be affected and will be copied as is.
            //    This is indicated by the variable isInit 
            // 2) Constraints with an action in the positive
            //    body which is not part of the current plan
            //    can be skipped. On the other hand, also 
            //    constraints with an action in the negative
            //    part of the body which IS part of the current
            //    plan can be skipped. 
            //    This is indicated by the variable copy 
            bool isInit = true;
            bool copy = true;

            for (GCONSTRAINT::const_iterator j = i->begin(); 
                 j != i->end(); 
                 j++)
                if( // Look for all occurences of positive ...
                    // ... and negative fluent predicates ...
                    fluentGatoms->contains(*j) &&
                    // ... with timestamp > 0 (timestamp is  the last param)...
                    (j->getTheRealOne().getParams()->end()-1)->getInt() )
                    {
                    isInit = false;
                    break;
                    }
                // ... whenever an action predicate occurs, 
                // this is no initial state constr.  
                else if ( actionGatoms->contains(*j) )
                    {   
                    isInit = false;
                    // Skip constraints with false action literals
                    // in the current Interpretation:
                    if( I.isFalse(*j) )
                        copy = false;
                    break;
                    }

            if( copy)
                {
                if( isInit )
                    {
                    // Copy constraint as is
                    // :- l1, ..., ln.
                    constraints.push_back(*i);
                    if( FTraceLevel >= 1 )
                        cdebug << "Constraint " << *i << " copied" << endl;
                    }
                else if( checkMethod == 0 )
                    {
                    // For CHECK1:
                    // Generate Rule
                    // %abs :- l1, ..., ln.
                    GDISJUNCTION head;
                    
                    head.add( pred_abs );
                    
                    GRULE rule( &head, &*i );           
                    rules.push_back( rule );
                    if( FTraceLevel >= 1 )
                        cdebug << "Constraint " << *i << endl
                               << "changed to rule " << rule << endl;
                    }
                else if( checkMethod == 1 )
                    {
                    // For CHECK2:
                    // Generate Constraint:
                    // :- l1, ..., ln, not %abs.
                    GCONSTRAINT newConstr(*i);
                    newConstr.add( GLITERAL(true, pred_abs ) );
                    constraints.push_back( newConstr );
                    if( FTraceLevel >= 1 )
                        cdebug << "Constraint " << *i << endl
                               << "changed to constraint " << newConstr 
                               << endl;
                    }
                else
                    // NO further checks implemented so far.
                    assert(0);
                }
            else if( FTraceLevel >= 1 )
                        cdebug << "Removing constraint " << *i << endl;
                
            }
        } 


    // Adding the new goal constraint: ":- $goal, not $abs."
    // or ":- not $abs." whenever the goal is trivially true.

    GCONSTRAINT newGoal;
    
    if ( ! goalTriviallyTrue)
        // add $goal to constraint
        newGoal.add( GATOM(ATOM(PRED_GOAL,0)) );
    
    // add  not $abs to constraint
    newGoal.add( GLITERAL( true, GATOM(ATOM(PRED_ABS,0)) ) );
    constraints.push_back(newGoal);
    
    if( FTraceLevel >= 1 )
        cdebug << "Adding new goal constraint:"
               << newGoal << endl;

    bool secure = false;
    if ( executeCheck( rules, constraints ) )
        {
        secure =  true;
        }

    collectedPlans[checkMethod].insert( currentPlan, secure );
    return secure;
    }

static void printStateOrActions( CONJUNCTION *stateOrActions, 
                                 CONJUNCTION *planCosts )
    {
    bool first = true;
    for( CONJUNCTION::const_iterator i = stateOrActions->begin();
         i != stateOrActions->end(); i++ )
        {
        TERMS newParams( *(i->getAtom().getParams()) );
        assert ( newParams.size() );

        int costs = 0;
        int level = 0;
        // Remark:(FIXME? - low priority!)
        //        This is probably not the most efficient way to 
        //        display the costs, but for printing
        //        we do not need to be efficient.
        if(planCosts)
            {
            for( CONJUNCTION::const_iterator j = planCosts->begin();
                 j != planCosts->end(); j++)
                {
                if( strcmp( j->getPredName()+strlen(COST_PREFIX),
                            i->getPredName() ) == 0 )            
                    {
                    TERMS costParams( *(j->getAtom().getParams()) );
                    costParams.pop_back();
                    if( costLevelsUsed )
                        {
                        level = costParams[costParams.size()-1].getInt();
                        costParams.pop_back();
                        }
                    if( costParams == newParams )
                        {
                        costs = getLastParamAsInt(*j);
                        }
                    }
                }
            }
            newParams.pop_back();
        if ( !first) 
            {
            cout << ", ";
            }
        else 
            {
            first = false;
            }
        cout << ( ATOM(i->getAtom(), &newParams) );
        if( costs )
            {
            cout << ":";
            if ( costLevelsUsed )
                cout << "<";                
            cout << costs;
            if ( costLevelsUsed )
                cout << ":" << level << ">";
            }
        }    
    }
//////////////////////////////////////////////////////////////////////////////
static bool printPlan(
//
    const INTERPRET  &A,
    const GINTERPRET *I,
    const bool        plansPrinted, 
    const bool        printCosts )
    {
    
    // For secure planning, print each plan only once. 
    if ( FPlan & FRONTEND_PLANNING_SECURE )
        {
        GATOMSET currentPlan;

        if (I) 
            extractPlanFromGInterpret( *I, currentPlan );
        
        if( collectedPrintedPlans.insert( currentPlan ).second == false )
            {
            // Plan already printed.
            return false;
            }
        }
    
    
    // Collect actions and fluents and store them into vectors, which consist
    // of one CONJUNCTION per time step.  First take those in the global
    // interpretation A.
  
    vector<CONJUNCTION> planStates(maxPlanLength+1);

    vector<CONJUNCTION> planActions(maxPlanLength);
    CONJUNCTION planCosts;

#ifndef NDEBUG
    // Actions should not occur in the global interpretation as
    // they can only be derived from guessing rules:
    // #ifndef NDEBUG added to search for actions only in debugging mode...
    for( PREDICATE_NAMES::const_iterator i=actions.begin();
         i != actions.end(); 
         i++ )
        {
        const pair<PREDICATE_NAMES::index_t,bool> j=ATOM::Predicates.find(*i);

        if( j.second )
            {
            const ATOM pattern(j.first,0,false);

            ATOMSET::FIND_RESULT f;
            A.findInPositivePart(pattern, f);

            if( ! f.atEnd() )
                {
                // See above: actions should not be found here!
                assert( 0 );
                }
            }
        }
#endif

    

    if(! (FPlan & FRONTEND_PLANNING_SECURE) )                     
        {
        ATOMSET::FIND_RESULT f;

        for( PREDICATE_NAMES::const_iterator i=fluents.begin();
             i != fluents.end(); 
             i++ )
            {
            // look for positive...
            pair<PREDICATE_NAMES::index_t,bool> j=ATOM::Predicates.find(*i);
            if( j.second )
                {
                const ATOM pattern(j.first,0,false);
                
                A.findInPositivePart(pattern, f);
                
                while( ! f.atEnd() )
                    {
                    int i = getLastParamAsInt(*f);
                    planStates[i].add(*f);
                    f++;
                    }
                }
                        
            // ... and negative occurrence of the fluent
            const char *temp = (*i).getNegativeName();
            j=ATOM::Predicates.find(temp);
            delete[] temp;
            if( j.second )
                {
                const ATOM pattern(j.first,0,false);

                A.findInPositivePart(pattern, f);
                
                while( ! f.atEnd() )
                    {
                    int i = getLastParamAsInt(*f);
                    planStates[i].add(*f);
                    f++;
                    }
                
                }
            }
        }

    // Then consult the interpretation generated by the Model Generator,
    // which is usually much more interesting.

    if( I )
        {
        for(unsigned i=0; i < GATOMSET::getSize(); i++)
            {
            GATOM a(i);

            // Find the fluents and positive actions in the model and add
            // them to the plan.

            if( I->isTrue(a) )
                {

                if( actionGatoms->contains(a) )
                    {	       
                    const ATOM atom(a.getTheRealOne());
                    
                    int i = getLastParamAsInt(atom);

                    if ( FTraceLevel >= 1 )                        
                        cdebug << "Found action " 
                               << atom << " in model at state: " 
                               << i << endl;

                    planActions[i].add(atom);
                    }    
                else if( costsUsed && 
                         costGatoms->contains(a) )
                    {
                    const ATOM atom(a.getTheRealOne());
                    if ( FTraceLevel >= 1 )
                        cdebug << "Found (action) costs " << atom 
                               << " in model.";
                    
                    planCosts.add(atom);
                    }
                else if(!(FPlan & FRONTEND_PLANNING_SECURE) &&
                        fluentGatoms->contains(a) )
                    {
                    const ATOM atom(a.getTheRealOne());
                    
                    int i = getLastParamAsInt(atom);

                    if ( FTraceLevel >= 1 )
                        cdebug << "Found fluent " << atom 
                               << " in model at state: " 
                               << i << endl;

                    planStates[i].add(atom);
                    }
                }
            }
        }

    // Finally, print the plan.

    if( ! OptionSilent )
        if( plansPrinted )
            cout << endl;
    if( ! (FPlan & FRONTEND_PLANNING_SECURE) ) 
        {
        for( int i = 0; i <= maxPlanLength; i++)
            {
            cout << "STATE " << i << ": ";
            printStateOrActions(&planStates[i],0);
            cout << endl;
            if ( i < maxPlanLength)
                {    cout << "ACTIONS: ";
                if(planActions[i].size())
                    printStateOrActions(&planActions[i], 
                                        costsUsed? &planCosts: 0);
                else
                    cout << "(no action)";
                cout << endl;
                }
            }
        }
    
    // If I=0 (Problem solved deterministically by the grounding)
    // then don't bother about secureChecks.
    if ( I && FPsoundcheck > -1 && checkPlan( *I,GroundIDB,
                                              GroundConstraints, 
                                              FPsoundcheck) )
        cout << "PLAN (secure): ";
    else if ( I && FPcompletecheck > -1 && checkPlan( *I,GroundIDB,
                                                      GroundConstraints, 
                                                      FPcompletecheck) )
        cout << "PLAN (security unknown): ";
    else 
        cout << "PLAN: ";
    
    bool firstItem=true;
    for( int i = 0; i < maxPlanLength; i++)
        {
        if( firstItem )
            firstItem=false;
        else
            cout << "; ";

        if( planActions[i].size() )
            printStateOrActions(&planActions[i],
                                costsUsed? &planCosts: 0);
        else
            cout << "(no action)";
        }
    
    if ( printCosts )
        {
        cout << "  COST: "; 
        if ( costLevelsUsed )
            cout << I->getCost();
        else
            I->printLevelCost(cout, 1);
        }
    cout << endl;

    return true;
    }

//////////////////////////////////////////////////////////////////////////////
static void processNoConc()
    // This procedure adds all necessary constraints to guarantee that
    // concurrent actions are disallowed.
    {
    // For all actions:
    for( PREDICATE_NAMES::index_t i = 0;
         i < actions.size();
         i++)
        {
        ostringstream actstr1,actstr2, sstr;
        actstr1 << actions.getItem(i).getName() << "(";
        actstr2 << actions.getItem(i).getName() << "(";

        for(int k = actions.getArity(i)-1;
            k > 0;
            k--)
            {
            actstr1 << "X" << 2*k     << ",";
            actstr2 << "X" << 2*k - 1 << ",";
            }
        
        actstr1 << "T)";
        actstr2 << "T)";
        

        // Don't allow the same action with two pairwise different parameters.
        for( int k = actions.getArity(i)-1; k > 0; k-- )
            {
            sstr << ":- " << actstr1.str() << ", " << actstr2.str() 
                 << ", X" << 2*k << "!= X" << 2*k - 1 << "." << endl;
            }
						      
	 // Don't allow two different actions at the same time.
        for( PREDICATE_NAMES::index_t j = i+1;
             j < actions.size(); 
             j++)
            {
            actstr2.str("");
            actstr2 << actions.getItem(j).getName() << "(";
            for( unsigned k = 
                     2*(actions.getArity(i)-1) + actions.getArity(j) - 1;
                 k > 2*(actions.getArity(i)-1);
                 k--)
                {
                actstr2 << "X" << k << ", ";
                }
            actstr2 << "T)";
            
            sstr << ":- " << actstr1.str() << ", " 
                 << actstr2.str() << "." << endl;
            }

        if( FTraceLevel >= 1 )
            cdebug << sstr.str();

        if ( ! parseStringStream(sstr) )
            InternalError("processNoConc");
        }
    }

//////////////////////////////////////////////////////////////////////////////
static void constrainNumberOfActions()
    // This procedure adds all necessary rules and constraints
    // to constrain the number of actions per time.
    {
    // For all actions:
    ostringstream sstr;
    for( PREDICATE_NAMES::index_t i = 0;
         i < actions.size();
         i++)
        {
        sstr << PRED_ACT << "(" << actions.getItem(i).getName() << ",";
        for(unsigned j=1; j <= maxActionArity; j++)
            {
            if( j < actions.getArity(i) )
                sstr << "X" << j << ",";
            else
                sstr << "0,";
            }
        sstr << "T) :- " << actions.getItem(i).getName() << "(";
        for(unsigned j=1; j < actions.getArity(i); j++)
            sstr << "X" << j << ",";
        sstr << "T)." << endl;
        }

    sstr << ":-"<< PRED_ACTIONTIME <<"(T), not "
         << minActions << " <= #count{A";
    for(unsigned j=1; j <= maxActionArity; j++)
        sstr << ",X" << j;
    sstr << ":" <<  PRED_ACT << "(A";
    for(unsigned j=1; j <= maxActionArity; j++)
        sstr << ",X" << j;
    sstr << ",T)}"; 
    
    if(maxActions) 
        sstr << "<= "<< maxActions;
    
    sstr << "." << endl;

    if ( FTraceLevel >= 1 )
        cdebug << sstr.str();
    if ( ! parseStringStream(sstr) )
        InternalError("constrainNumberOfActions");
    }

//////////////////////////////////////////////////////////////////////////////
static bool muxStrat( unsigned long i, 
                      unsigned int t,
                      GRULES good, 
                      set< long unsigned > bad)
    {
    // FIXME: Check this comment! Might be obsolete!
    // In the current version this function only checks
    // whether causationRules are mux-stratified. This is a sufficient 
    // condition to guarantee that domains are secure.
    // Still, a domain could be proper, even if this check fails!

    // Simple proper check:
    // This check works as follows:
    // 1) build a dep graph for all causation rules 
    //    in the always part:
    //    caused a if b,c, ... after ... 
    // where 
    // a) we do not consider the after-part at all
    // b) we do only consider fluents (no type predicates or builtins).
    // c) in this simple version we check only the nonground program.
    // d) Mutual exclusive inertia rules are recognized and 
    //    treated separatedly, i.e. two graphs are built for any pair
    //    of mutual exclusive inertia rules. This function steps through
    //    all pairs of mutually exclusive rules recursively.
    //    FIXME: This is done in function addIfMutex() above and could be
    //           extended to arbitrary pairs of rules with mutually
    //           exclusive after-parts.
    // 2) check whether this graph is stratified (doesn't
    //    contain circles with "not")

    // Create dep.graph for proper-check for
    // causation rules (including total) with a 
    // non-empty head and if-part:
    if( i == causationRules[t].size() )
        {
        DEPGRAPH<GATOM,GCONSTRAINTS,GPROGRAM> 
            dg( good, GCONSTRAINTS(),
                DEPGRAPH<GATOM,GCONSTRAINTS,GPROGRAM>::WITHSPECIALEDGES );
        
        for(unsigned k = 0; k < dg.getComponents().size(); k++)
            {
            if( !(dg.isStratified(k)) )
                return false;
            }

        return true;
        }
    else
        {
        // If this rule is already in the bad list...
        if( bad.find(i) != bad.end() )
            {
            // ... do not take this rule proceed to the next:
            return muxStrat(i+1, t, good, bad);
            }

        if( mutexLists[t][i].size() )
            {
            // Is domain proper when not taking this rule
            // and taking the mutex rules instead?
            GRULES newGood(good);
            set< unsigned long > newBad(bad);
            if( ! muxStrat(i+1, t, newGood, newBad) )
                {
                return false;
                }
            // If yes, add them to mutexList and try with this rule.
            for ( vector<unsigned long>::const_iterator k = 
                      mutexLists[t][i].begin();
                  k != mutexLists[t][i].end(); k++ )
                {
                bad.insert(*k);
                }
            }
        // add only causation/total rule which are not on the 
        // bad list
        good.push_back( causationRules[t][i] );
        return muxStrat(i+1, t, good, bad);
        }
    }


//////////////////////////////////////////////////////////////////////////////
void PlanCallbackAfterParsing()
    {
    // This function does the following:
    // 1) Check whether background knowledge
    //    a) does not contain weak constraints or 
    //    b) aggregates
    //    c) is stratified
    // 2) Check whether a 
    //    a) fluent/action names already have occured in the 
    //       background knowledge positively or negatively.
    //    b) there are positive dependencies among executability 
    //       conditions.
    // 3) Add cost rules and weak constraints for action 
    //    declarations with a costs-clause.
    // 4) Save fluents and actions in causation rules wrt. declarations.
    // 5) Translate all krules and add them.
    // 6) If the number of actions per time has been constrained
    //    via options -planmin/maxactions or keyword noConcurrency,
    //    add necessary rules and constraints as well.
    // 7) Add the auxiliary predicates and the planning goal.

    // 1)a) Check whether background knowledge does not contain weak
    //      constraints.
    if( WConstraints.size() )
        {
        cerr << "Background knowledge must not contain weak constraints."
             << endl;
        SafeExit(1);
        }
    
    // 1)b) Check whether background knowledge does not contain aggregates.
    bool bkHasAggregates=false;
    
    // Search for aggregates in constraints...
    for( CONSTRAINTS::const_iterator i = Constraints.begin(); 
         i != Constraints.end(); i++)
        {
        for( CONJUNCTION::const_iterator j = i->begin(); 
             j != i->end(); j++)
            if(j->isAggregate())
                bkHasAggregates = true;
        }
    // ... and rule bodies
    for( RULES::const_iterator i = IDB.begin(); 
         i != IDB.end(); i++)
        {
        if( i->getBody() )
            for( CONJUNCTION::const_iterator j = i->getBody()->begin(); 
                 j != i->getBody()->end(); j++)
                if(j->isAggregate())
                    bkHasAggregates = true;
        }
    
    if( bkHasAggregates )
        {
        cerr << "Background knowledge must not contain aggregates."
             << endl;
        SafeExit(1);
        }
    
    // 1)c) Check whether background knowledge is stratified.
    DEPGRAPH<ATOM,CONSTRAINTS,PROGRAM> bkdg(
        IDB, CONSTRAINTS(),
        DEPGRAPH<ATOM,CONSTRAINTS,PROGRAM>::NONGROUND |
        DEPGRAPH<ATOM,CONSTRAINTS,PROGRAM>::WITHSPECIALEDGES );
    
    for(unsigned i = 0; i < bkdg.getComponents().size(); i++)
        {
        if( !bkdg.isStratified(i) || bkdg.isDisjunctive(i) )
            {
            cerr << "Warning: Background knowledge is either unstratified "
                    "or includes disjunction." << endl;
            break;
            }
        }
    
    // 2) a) Check whether a fluent/action names already have
    //       occured in the background knowledge.
    const char *temp = 0;
    
    for( PREDICATE_NAMES::const_iterator i=fluents.begin();
         i != fluents.end(); 
         i++ )
        {
        if( (ATOM::Predicates.find(*i)).second ||
            (ATOM::Predicates.find(temp = (*i).getNegativeName())).second )
            {
            cerr << "Predicate names used in the background knowledge "
                    "can not be used as fluent names." << endl;
            SafeExit(1);
            }
        delete[] temp;
        }
    
    for( PREDICATE_NAMES::const_iterator i=actions.begin();
         i != actions.end(); 
         i++ )
        {
        if( (ATOM::Predicates.find(*i)).second ||
            (ATOM::Predicates.find(temp = (*i).getNegativeName())).second )
            {
            cerr << "Predicate names used in the background knowledge "
                    "can not be used as action names." << endl;
            SafeExit(1);
            }
        delete[] temp;        
        }


    // 2) b) Check whether there are positive cyclic dependencies 
    //       among executability conditions.

    // Variable execRules stores some possibly cyclic executability 
    // conditions: 
    RULES execRules;

    for( KRULES::const_iterator i = krules.begin();
         i != krules.end();
         i++ )
        {
        // Is this an executability condition?
        if( i->getCaused() && actions.find(i->getCaused()->getName()).second )
            {
            if( i->getIf() )
                {
                for( KLITERALS::const_iterator j = i->getIf()->begin();
                     j != i->getIf()->end(); j++) 
                    {
                    if( (actions.find(j->getName())).second &&
                        !j->isNegative())
                        {
                        // Positive action dependency detected!
                        char *headstr = 
                            new char[strlen(i->getCaused()->getName())+2];
                        char *bodystr = 
                            new char[strlen(j->getName())+2];
                        *headstr = *bodystr = '$';
                        strcpy( headstr+1,i->getCaused()->getName() );
                        strcpy( bodystr+1,j->getName() );
                        DISJUNCTION head;
                        CONJUNCTION body;
                        head.add( ATOM(headstr,0) );
                        body.add( ATOM(bodystr,0) );
                        execRules.push_back( RULE( &head, &body ) );
                        }
                    }
                }
            }
        }

    CONSTRAINTS dummy;
    DEPGRAPH<ATOM,CONSTRAINTS,PROGRAM> 
        dg( execRules, dummy,
            DEPGRAPH<ATOM,CONSTRAINTS,PROGRAM>::NONGROUND );
    if( (dg.isCyclic()) )
        execCyclic = true;


    for( KRULES::const_iterator i = declarations.begin();
         i != declarations.end();
         i++ )
        {
        assert( i->getCaused() );

        // 3) Add weak constraints for costs:
        if ( i->getCost() )
            {
            // Only Actions might have costs:
            assert ( (actions.find(i->getCaused()->getName())).second );
           
            KATOM cost_katom( *(i->getCaused()) ); 
            
            // Flag indicating whether cost depend on "time":
            bool timeUsed( i->getCost()->find( VAR_T1 ) < i->getCost()->npos );

            // Split cost and level:
            string cost = i->getCost()->substr( 0,i->getCost()->find(':') );
            string level = i->getCost()->substr( i->getCost()->find(':')+1 );

            cost_katom.push_back_param( VAR_T );
            ostringstream wcsstr, sstr;
            sstr << COST_PREFIX << cost_katom.getName() << "("
                 << *(cost_katom.getParams());

            // FIXME: Adding cost term only if it is not an integer
            //        constant would be an optimization, but then costs
            //        can no longer be read by printPlan().
            if( costLevelsUsed )
                {
                // Add cost level only if necessary:
                sstr << "," << level;
                }
            // Add cost:
            sstr << "," << cost << ")";
            
            wcsstr << ":~ " << sstr.str();
            
            // If the cost is a variable then add "0 < Cost" 
            // to  ensure that it will only bind to a positive integer:
            if ( isupper( *(cost.begin()) ) ) 
                wcsstr << ", 0 < " << cost;
            // If the cost level is a variable then add "0 < Level" 
            // to  ensure that it will only bind to a positive integer:
            if ( isupper( *(level.begin()) ) )
                 wcsstr << ", 0 < " << level;
            
            wcsstr << ". [" << *(i->getCost()) << "]";

            // Add cost constriant:
            if ( FTraceLevel >= 1 )
                cdebug << wcsstr.str() << endl;
            if( ! parseStringStream(wcsstr) )
                {
                cerr << "Error while translating declaration:" << endl
                     << "   " << *i << endl;
                SafeExit(1);
                }

            // Add cost rule:
            sstr << " :- " << cost_katom; 

            //Add requires part ...
            if( i->getIf() )
                sstr << ", " << *(i->getIf());

            //...and where-clause:
            if ( i->getAfter() )
                {
                sstr << ", " << *(i->getAfter());
            
                //Check whether 'time' was used in the where-clause:
                for(KLITERALS::const_iterator j = i->getAfter()->begin();
                    !timeUsed && j != i->getAfter()->end(); j++) 
                    {
                    if( j->getParams() )
                        {   
                        for( KPARAMS::const_iterator k = 
                                 j->getParams()->begin();
                             !timeUsed && k != j->getParams()->end();
                             k++ )
                            {
                            if ( *k == VAR_T1) 
                                timeUsed = true;
                            }
                        }
                    }
                }
            
            
            // Add $nexttime(T,T1) only if needed:  
            if( timeUsed )
                sstr << ", " PRED_NEXTTIME "(" VAR_T "," VAR_T1 ")";
            
            sstr << "." << endl;
            
            if ( FTraceLevel >= 1 )
                cdebug << sstr.str();
            if( ! parseStringStream(sstr) )
                {
                cerr << "Error while translating declaration:" << endl
                     << "   " << *i << endl;
                SafeExit(1);
                }
            }
        }


    // In case of cyclic executability condiditions
    // rewrite all executability conditions:
    //    executable a if B ==>
    //       executable a.
    //       nonexecutable a if not exec_a.
    //       caused exec_a if B.
    // FIXME: It would be sufficient to do this rewriting only for those
    //        actions, which contain positive cyclic action dependencies.
    if( execCyclic )
        {
        KRULES newKrules;
        for( KRULES::iterator i = krules.begin();
             i != krules.end();
             i++ )
            // For all executability conditions with a non-empty if-part
            // of the form "executable a if B":
            if( i->getCaused() && i->getIf() &&
                actions.find(i->getCaused()->getName()).second
                )
                {
                // Add new Krule: "executable a."
                newKrules.push_back( KRULE(i->getCaused(),0,0,false,true) );

                char *exec_a =
                    new char[strlen(i->getCaused()->getName()) + 
                            strlen(EXEC_PREFIX) + 1];
                strcpy( exec_a, EXEC_PREFIX );
                strcat( exec_a, i->getCaused()->getName() );

                KATOM exec_act( false, exec_a, i->getCaused()->getParams());

                delete[] exec_a;

                // Add new nonexeutability cond: 
                // "nonexecutable a if not $exec_a."
                KLITERALS hkliterals;
                hkliterals.push_back( KLITERAL( false, *(i->getCaused()) ) );
                hkliterals.push_back( KLITERAL( true, exec_act ) );
                newKrules.push_back( KRULE(0,0,&hkliterals,false,false) );

                // Change head of original executability condition to
                // from "executable a if B." to "caused $exec_a if B."
                i->setCaused( exec_act  );
                i->setGuess( false );
                }

        // Finally add the newly generated Rules:
        for( KRULES::iterator i = newKrules.begin();
             i != newKrules.end();
             i++ )
            krules.push_back(*i);
        
        }

    for( KRULES::iterator i = krules.begin();
         i != krules.end();
         i++ )
        {
        // 4) Save fluents and actions in causation rules wrt. 
        //    declarations.
        i->save();
        // 5) Translate all krules:
        i->translate();
        }

    // 6) Constrain the number of actions per time
    if ( minActions || maxActions)
        {
        if( maxActions && (minActions > maxActions) )
            {
            cerr << "-planminactions: Minimum number of actions must be less "
                    "than or equal to maximum number of actions per time." 
                 << endl;
            SafeExit(1);
            }
        if(minActions == 0 && maxActions == 1)
            // The constraints version is more efficient than the aggregates 
            // version for noConcurrency, i.e. minActions == 0 and 
            // maxActions == 1
            // FIXME: If the performance of aggregates improves, 
            //        processNoConc() could be removed as it only
            //        handles one special case of
            //        constrainNumberofActions() coul.
            processNoConc();
        else
            constrainNumberOfActions();
        }


    // sstr stores the auxiliary time predicates and the goal:
    ostringstream sstr;

    
    // 7) Now create the constraint which corresponds to the given
    //    goal and add the facts: 
    //    $time(i). $actiontime(i). $nexttime(i, i+1) 
    //    for all i < maxPlanLength.
    
    if ( maxPlanLengthParsed >= 0 )
        {
        maxPlanLength = maxPlanLengthParsed;
        }
    else if ( maxPlanLength < 0 )
        {
        maxPlanLength = 0;
        cerr << "Warning: No plan length given. Plan length "
                "defaults to zero."
             << endl;
        }

    // Set global variables MaxIntegerSeen.
    MaxIntegerSeen = max( MaxIntegerSeen, maxPlanLength );
    
    // Add facts: $time(i). $actiontime(i). $nexttime(i, i+1) 
    // for all i < maxtime 
    for(int i=0; i < maxPlanLength; i++)
	{
        sstr << PRED_TIME "(" << i << "). ";
        sstr << PRED_ACTIONTIME "(" << i << "). ";
        sstr << PRED_NEXTTIME "(" << i << ", " << i+1 <<  ")." << endl;
        }
    
    {
    // Add fact: $time(maxPlanLength). 
    sstr << PRED_TIME "(" << maxPlanLength << ")." << endl;
    if ( FTraceLevel >= 1 )
        cdebug << sstr.str();
    // FIXME: Don't send this to the dlv core parser, cause
    //        it will complain if #maxint has not been set:
    // if( ! parseStringStream(sstr) ) 
    //    InternalError("PlanCallbackAfterParsing");
    // Instead, add it by hand:
    // ************** Remove this when possible: *****************
    // Throw a warning if MaxInteger is smaller than planlength:
    if( MaxInteger > -1  && MaxInteger < maxPlanLength )
        {
        cerr << "Warning: #maxint is smaller than plan length." << endl;
        }

    TERMS *params;    
    // Add facts: $time(i). $actiontime(i). $nexttime(i, i+1) 
    // for all i < maxtime. For maxtime only add: $time(i). 
    for(int i=0; i <= maxPlanLength; i++)
	{
	params = new TERMS();

        params->push_back( TERM(i,0) );
        const ATOM atom_time(PRED_TIME, params, 
                             PREDICATE_NAMES::typeEDB);
        EDB.add( atom_time );

        if( i < maxPlanLength)
            {
            const ATOM atom_actiontime(PRED_ACTIONTIME, params, 
                                       PREDICATE_NAMES::typeEDB);
            params->push_back( TERM(i+1,0) );
            const ATOM atom_nexttime(PRED_NEXTTIME, params, 
                                     PREDICATE_NAMES::typeEDB);
            EDB.add( atom_actiontime );
            EDB.add( atom_nexttime );
            }
	delete params;
	}
    // *********************************************************

    // Reset sstr:
    sstr.str("");
    }
    
    // unless query is empty (e.g. "goal: true?"
    if ( Goal )
	{
        char idMaxTime[NUM_LENGTH+1];
        sprintf( idMaxTime, "%d", maxPlanLength );

        // Copy the original goal query before time-parameters are added:
        const KLITERALS origGoal(*Goal);

	for(KLITERALS::iterator i = Goal->begin();
	    i != Goal->end(); i++)
	    {
	    if( ! (fluents.find(i->getName())).second ) 
		{
		cerr << "Only fluent literals allowed in "
                        "planning goal queries." << endl;
		SafeExit(1);
		}
	    
	    i->push_back_param( idMaxTime );
	    }
	
        sstr << PRED_GOAL " :- " << *Goal << "." << endl;
        sstr << ":- not " PRED_GOAL ". " << endl;
	if( FTraceLevel >= 1 )
            cdebug << sstr.str();

        // FIXME: Don't send this to the dlv core parser, cause
        //        it will complain if #maxint has not been set:
        //if( ! parseStringStream(sstr) )
        //    {
        //        cerr << "Error while translating goal query:" << endl
        //             << "   " << origGoal << "? (" 
        //             << maxPlanLength << ")"<<endl;
        //        SafeExit(1);
        //    }
        // if( ! parseStringStream(sstr) ) 
        //    InternalError("PlanCallbackAfterParsing");
        // Instead, add it by hand:
        // ************** Remove this when possible: *****************
	CONSTRAINT goalquery;
        goalquery.add( LITERAL( true, ATOM(PRED_GOAL,0) ) );
        Constraints.push_back(goalquery);
                                
        // Create a conjunction from the given KATOMS.
        CONJUNCTION body;
	for(KLITERALS::const_iterator i = Goal->begin();
	    i != Goal->end(); i++)
	    {
            TERMS *hterms = 0;
            if( i->getParams() )
                {
                hterms = new TERMS;
                for( KPARAMS::const_iterator j = i->getParams()->begin();
                     j != i->getParams()->end();
                     j++)
                    {
                    TERM *hterm;
                    if( isdigit(*(j->begin())) )
                        {
                        char *err;
                        hterm = new TERM(strtoul(j->c_str(),&err,10),0);
                        assert( *err == '\0' );
                        }
                    else if( *j == "_" )
                        hterm = new TERM(LocalVariables.anonymous());
                    else
                        hterm = newTERM(j->c_str());

                    hterms->push_back(*hterm);
                    delete hterm;
                    }
                }
            ATOM *hatom = i->isTrueNegative() ? 
                newNegativeATOM( i->getName(), hterms ) :
                new ATOM( i->getName(), hterms );
            
            body.add( LITERAL(i->isNegative(), *hatom) );
            delete hatom;
            delete hterms;
            }

        if( !body.isSafe() )
            {
            cerr << "Goal query is not safe:" << endl
                 << "   " << origGoal << "? (" 
                 << maxPlanLength << ")"<<endl;
            SafeExit(1);
            }

	DISJUNCTION head;
	head.add( ATOM(PRED_GOAL,0) );
     	IDB.push_back( RULE( &head, &body ) );
        
        LocalVariables.clear();
        // **********************************************************

        // FIXME: How to deal with multiple goal queries?
        //        We could remove the last constraint generated
        //        and add it as rule... dirty.
	}
    else
        {
        cerr << "No goal query has been specified." << endl;
        SafeExit(1);
        }


    // This is just to declare the predicate Names of 
    // PRED_ABS and PRED_GOAL
    // which is used later on in the secure check.
    GATOM a(ATOM(PRED_ABS,0));
    GATOM b(ATOM(PRED_GOAL,0));
    }

//////////////////////////////////////////////////////////////////////////////
void PlanCallbackAfterGrounding()
    {
    // Collect all the grounded positive actions and fluents
    // and the resp. costs_atoms.
    fluentGatoms = new GATOMSET();
    actionGatoms = new GATOMSET();
    costGatoms = new GATOMSET();

    for(unsigned i=0; i < GATOMSET::getSize(); i++)
        {
        const GATOM gatom(i);

        const ATOM hatom(gatom.getTheRealOne());

        if( (fluents.find(hatom.getPredItem().getPositiveName())).second )
            {
            fluentGatoms->add(gatom);
            }        
        else if( (actions.find(hatom.getPredName())).second )
            {
            actionGatoms->add(gatom);
            }
        else if ( costsUsed && 
                  strncmp(hatom.getPredName(), COST_PREFIX, 
                          strlen(COST_PREFIX)) == 0 )
            {
            costGatoms->add(gatom);
            }
        }
    
    // add constraint :- $abs.
    GCONSTRAINT absconstraint;
    
    absconstraint.add( GATOM(ATOM(PRED_ABS,0)) );
    
    if( FTraceLevel >= 1 )
        {
        cdebug << "Adding constraint: " << absconstraint << endl;
        }
    
    GroundConstraints.push_back(absconstraint);


    // Check Mux-Stratification
    // (only if optimistic planning is not explicitely enforced):
    // 
    // 0) If a domain is stratified, it is also mux-stratified.

    DEPGRAPH<GATOM,GCONSTRAINTS,GPROGRAM>
        dg( GroundIDB, GCONSTRAINTS(),
            DEPGRAPH<GATOM,GCONSTRAINTS,GPROGRAM>::WITHSPECIALEDGES );

    bool unstratified = false;
    for(unsigned k = 0; k < dg.getComponents().size(); k++)
        {
        if( !(dg.isStratified(k)) )
            unstratified = true;
        }
    if (! unstratified)
        return;

    // Otherwise: 
    // 1) Produce List of causation Rules
    // 2) Add mux-Lists, i.e. mux-pairs for each pair of caus.rules.
    // 3) Check muxStrat wrt. to these mux-lists
    
    // ad 1) Produce Lists of causation Rules and MutexLists
    for(int t = maxPlanLength; t > 0; t--)
        {
        causationRules.push_back( GRULES() );
        mutexLists.push_back( vector< vector< unsigned long > >() );
        }
    if ( !(FPlan & FRONTEND_PLANNING_OPTIMISTIC) ) 
        {
        for( GRULES::const_iterator i=GroundIDB.begin();
             i != GroundIDB.end();
             i++ )
            {
            if( i->hasHead()
                && fluentGatoms->contains( *(i->getHead().begin()) ) )
                    {
                // Only causation rules with a non-empty body are interesting.
                if ( i->getBody() )
                    {
                    unsigned t = 
                        getLastParamAsInt( i->getHead().begin()->getTheRealOne() );
                    // Do not consider initial state constraints:
                    if( t > 0 )
                        {
                        causationRules[t-1].push_back( *i );
                        mutexLists[t-1].push_back( vector< unsigned long >() );
                        }
                    }
                }
            }

        // For each time step t > 0 ...
        for(int t = maxPlanLength-1; t >= 0; t--)
            {
            // ... 
            // ad 2) create mux-pairs (opposite fluent literals in after-parts) 
            //       for each pair of causation rules:
            for( unsigned long i = 0; i < causationRules[t].size(); i++ )
                {
                for( unsigned long j=i+1; j < causationRules[t].size(); j++ )
                    {
                    // find mux literals for each fluent in the after-part of i, 
                    // i.e. where timestamp = t, ...
                    for( GCONJUNCTION::const_iterator body_i = 
                             causationRules[t][i].getBody()->begin();
                         body_i != causationRules[t][i].getBody()->end();
                         body_i++)
                        {
                        if ( fluentGatoms->contains( body_i->getAtom() ) &&
                             getLastParamAsInt(body_i->getAtom().getTheRealOne()) == t )
                            {
                            // ... and any literal in the body  part of j:
                            for( GCONJUNCTION::const_iterator body_j = 
                                     causationRules[t][j].getBody()->begin();
                                 body_j != causationRules[t][j].getBody()->end();
                                 body_j++)
                                {
                                if ( body_i->isNegative() && 
                                     body_j->isNegative() )
                                    {
                                    // default negated literals are not comparable:
                                    // so break, if we are in the negative body of
                                    // body_i an d body_j.
                                    break;
                                    }
                                else if ( (!body_i->isNegative() && !body_j->isNegative() &&
                                           // both literals are positive:
                                           // check for case 1) a <-> -a, -a <-> a
                                           body_i->getAtom().isComplementary(body_j->getAtom()))
                                          ||
                                          (body_i->isNegative() != body_j->isNegative() &&
                                           // one literal is positive:
                                           // check for case 2) not a <-> a, a <-> not a
                                           body_i->getAtom() == body_j->getAtom()) )
                                    {
                                    if( FTraceLevel >= 1 )
                                        {
                                        cdebug << "Added " << endl
                                               << "  " << causationRules[t][j] 
                                               << endl
                                               << "to mutex-list of " << endl 
                                               << "  " << causationRules[t][i] 
                                               << endl;
                                        }
                                    mutexLists[t][i].push_back( j );
                                    break;
                                    }
                                }
                            }
                        }
                    }
                }

            // ad 3) Check whether the rules are mux-strat wrt. to time step t
            GRULES good;
            set<unsigned long> bad;
            if ( ( (FPlan & FRONTEND_PLANNING_SECURE) || 
                   (FPlan & FRONTEND_PLANNING_ASK) ) &&
                 ( ! muxStrat(0, t, good, bad) ) )
                {
                cerr << "Warning: The domain is probably not mux-stratified, "
                    "secure check might not work correctly."
                     << endl;
                break;
                }
            }
        }
    }

//////////////////////////////////////////////////////////////////////////////
bool PlanCallbackCheck(
    MODEL_GENERATOR *,
    const INTERPRET *, 
    const GINTERPRET *I)
    {
    // For secure planning only consider
    // secure plans for the cost computation.
    if( ! (FPlan & FRONTEND_PLANNING_SECURE) )
        return true;        

    assert(I);

    // At this point do not decide, whether 
    // security is unknown. Return true for secure plans
    // AND for possibly secure plans.
    if (FPsoundcheck > -1 && 
        checkPlan(*I, GroundIDB, GroundConstraints, FPsoundcheck) )
            {
            return true;
            }
    else if (FPcompletecheck > -1 && 
             checkPlan(*I, GroundIDB, GroundConstraints, FPcompletecheck) )
            {
            return true;
            }
    else if ( checkPlan(*I, GroundIDB, GroundConstraints) )
        {
        return true;
        }
    
    return false;
    }

//////////////////////////////////////////////////////////////////////////////
bool PlanCallback(
    MODEL_GENERATOR *mg,
    const INTERPRET  *A, 
    const GINTERPRET *I)
    
    {
    assert( A );
    
    // If I is 0, we assume that the problem has been deterministically
    // solved by the Grounding.
    if( ! I )
        {
        printPlan(*A,0,false, false);
        return false;
        }
    
    if( FTraceLevel >= 1 )
        cdebug << "FPlan = " << static_cast<unsigned>(FPlan) << endl;
    

    // Print costs only if the optimal costs are known:
    if ( printPlan(*A,I,mg->getNumberOfModelsPrinted(), costsUsed) )
        mg->incrNumberOfModelsPrinted();
    
    if( FPlan & FRONTEND_PLANNING_ASK)
        {
        if( ! ( FPlan & FRONTEND_PLANNING_SECURE ) ) 
            {
            cout << endl << "Check whether that plan is secure (y/n)? ";
            char c[100];
            fgets(c,sizeof c,stdin);
        
            if (c[0] == 'y')
                {
                if ( checkPlan( *I,GroundIDB, GroundConstraints) )
                    {
                    cout << "The plan is secure." << endl;
                    }
                else
                    {
                    cout << "The plan is NOT secure." << endl;
                    }
                }
            }
        
        cout << endl << "Search for other plans (y/n)? ";
        char c[100];
        fgets(c,sizeof c,stdin);
        
        return ( c[0] == 'y' );
        }

    return true;
    }

void KRULE::save()
    {
    // Declarations can not be saved.
    assert(!decl);

    // Collect all the fluents and actions which might cause unsafe rules.
    KATOMS atoms_to_save;

    if( caused ) 
        {
        // The caused-part has to be saved, even if it has no parameters
        // as  the requires part of the resp. declaration might still
        // be non-empty!
        atoms_to_save.push_back( *caused );
        }

    if( ifpart )
        {
        for( KLITERALS::const_iterator i = ifpart->begin();
             i != ifpart->end(); i++) 
            {            
            // Add default negated fluents to atoms_to_save
            // For FRONTEND_PLANNING_C add all fluents if there is
            // a caused-part, due to double negation in if-part, see below.
            if( ( (fluents.find(i->getName())).second ) &&
                ( i->isNegative() ||
                  (caused && (OptionFrontend == FRONTEND_PLANNING_C)) ) )
                {
                atoms_to_save.push_back(*(i->getKatom()));        
                }
            }
        }

    if( afterpart )
        {
        for( KLITERALS::const_iterator i = afterpart->begin();
             i != afterpart->end(); i++) 
            {            
            // Add default negated fluents and actions to atoms_to_save
            if( ( (actions.find(i->getName())).second ||
                  (fluents.find(i->getName())).second ) &&
                i->isNegative() )
                {
                atoms_to_save.push_back(*(i->getKatom()));        
                }            
            }
        }

    // Save head and negative fluents/actions in body wrt. declarations:
    for(KATOMS::const_iterator i = atoms_to_save.begin();
        i != atoms_to_save.end();
        i++)
        {
        KRULES::const_iterator d = declarations.begin();

        for( ; d != declarations.end(); d++ )
            {
            // If the atom to save begins with EXEC_PREFIX
            // search for declaration, disregarding the prefix.
            short offset = 
                strncmp(EXEC_PREFIX,i->getName(),
                        strlen(EXEC_PREFIX)) ? 0 : strlen(EXEC_PREFIX);
            if( !strcmp(d->getCaused()->getName(), i->getName() + offset) )
                break;
            }

        assert( d != declarations.end() );
        
        // If the resp. declaration has a non-empty requires part...
        if( d->getIf() )
            {
            // ... add requires part ...
            for( KLITERALS::const_iterator reqIt = d->getIf()->begin();
                 reqIt != d->getIf()->end(); reqIt++)
                {
                // ... to the ifpart ...
                if ( ! ifpart )
                    ifpart = new KLITERALS;                            

                if( reqIt->getParams() )
                    {
                    // ... and change all matching parameters:
                    KPARAMS hparams;
                    
                    bool parameter_unchanged;

                    for(KPARAMS::const_iterator 
                            j = reqIt->getParams()->begin(); 
                        j != reqIt->getParams()->end();
                        j++)
                        {
                        parameter_unchanged = true;
                        // For all parameters of the fluent/action...
                        if ( i->getParams() )
                            {
                            for(unsigned k = 0; k < i->getParams()->size();
                                k++)
                                {
                                // ... check whether they match.
                                if(*j==(*(d->getCaused()->getParams()))[k])
                                    {
                                    hparams.push_back((*(i->getParams()))[k]);
                                    parameter_unchanged = false;
                                    break;
                                    }
                                }
                            }
                        // Unchanged variables are marked with suffix 
                        // '_<name of saved atom><Params of saved atom>"'
                        // to distinguish them from other variables:
                        if ( parameter_unchanged ) 
                            {
                            if( isupper(*(j->begin())) )
                                {
                                string newParam(*j + '_' + i->getName());
                                if( i->getParams() )
                                    for( KPARAMS::const_iterator k =
                                             i->getParams()->begin(); 
                                         k != i->getParams()->end();
                                         k++ )
                                        newParam += *k;
                                hparams.push_back(newParam);
                                }
                            else
                                {
                                hparams.push_back(*j);
                                }
                            }
                        }                    
                    ifpart->push_back( KLITERAL(*reqIt, &hparams) );
                    }
                else
                    {
                    ifpart->push_back(*reqIt);
                    }
                }
            }
        }
    }


void KRULE::translate()
    {
    // Copy the original rule before time-parameters are added:
    const KRULE origRule(*this);

    // This function translates the given krule to a dlv-rule or
    // and passes it to the dlv core parser via parseStringStream().

    // T1 stands for 0 in initial rules:
    const char *T1(init ? "0" : VAR_T1 );
    
    // These stringstreams collect the head and body of the generated rule.
    ostringstream headsstr, bodysstr;
    
    // Causation Rules must not have costs!
    assert( !cost );
    assert(!decl);
    // Guessing rules must have a head!
    assert ( !guess || caused );
    // Guessing rules must not have a negative head!
    assert ( !(guess && caused->isTrueNegative()) );

    bool isExecutabilityCond = 
        caused && 
        (actions.find(caused->getName()).second ||
         // Special rules from cyclic executability conditions 
         // starting with EXEC_PREFIX are treated
         // like normal executability conditions:
         !strncmp(EXEC_PREFIX,caused->getName(), strlen(EXEC_PREFIX)) );

    // Initial rules must not have an after-part!
    assert( !(init && afterpart) );
    // Executability conditions must not occur in the initial section!
    assert( !(init && isExecutabilityCond) );

    // caused-part
    if( caused ) 
        {
        caused->push_back_param(T1);
   
        // Add pos. atom for pos. caused-part, neg. atom for
        // neg. caused-part, and both for guessing rule.
        headsstr << *caused;
        if (guess)
            headsstr << " v -" << *caused;
        }

    // if/requires-part
    if( ifpart )
        {
        bool actionFound;
        for( KLITERALS::iterator i = ifpart->begin();
             i != ifpart->end(); i++) 
            {
            if( (actionFound = (actions.find(i->getName())).second) ||
                (fluents.find(i->getName())).second ) 
                {
                if ( !isExecutabilityCond && actionFound )
                    {
                    cerr << "Actions must not be used in the "
                         << "if-part of causation rules." << endl;
                    SafeExit(1);
                    }
                                
                // For FRONTEND_PLANNING_C:
                // All fluents in if-part must be double negated.
                // No double Neg in constraints (due to prop 2 in 
                // "Repr. Transition Systems by Logic Programs" 
                // (Lifschitz, Turner 1999) !
                if ( (OptionFrontend == FRONTEND_PLANNING_C) && caused )
                    {
                    i->setTrueNegative( !( i->isTrueNegative()) );
                    i->setNegative( !( i->isNegative()) );
                    }

                // For fluents add time parameter.
                i->push_back_param(T1);
                }
            }
        bodysstr << *ifpart;
        }

    // after/where-part
    if( afterpart ) 
        {
        for(KLITERALS::iterator i = afterpart->begin();
            i != afterpart->end(); i++) 
            {
            if( (actions.find(i->getName())).second ||
                !strncmp(EXEC_PREFIX,i->getName(), strlen(EXEC_PREFIX)) ||
                (fluents.find(i->getName())).second ) 
                {
                // For fluents or actions add time parameter.
                i->push_back_param( VAR_T);
                }
            }
        if( ifpart )
            bodysstr << ", ";          
        bodysstr << *afterpart;
        }

    // Add auxiliary predicates for timestamps.
    if ( ifpart || afterpart ) 
        bodysstr << ", ";
    
    if( afterpart ) 
        {
        // Add "$nexttime(T$,T1$)" for dynamic rules
        // and for cost constraints.
        // For cost constraints the $nexttime might be superflous
        // but should be removed by the grounding, anyway.
        
        bodysstr << PRED_NEXTTIME << "(" VAR_T "," << T1 << ")";
        }
    else if( isExecutabilityCond )
        {
        // Add "$actiontime(T1)" for executable statements as actions must 
        // not occur at maxtime.
        bodysstr << PRED_ACTIONTIME << "(" << T1 << ")";
        }
    else if( caused || ifpart )
        {
        // Constraint found, add "$time(T1$)."
        // FIXME:(optional): eventually add time-predicate only 
        //       if ifpart contains fluents (flag)?
        //       add testcases?
        //       caused false if fluent.
        //       caused false if constant.
        //       caused false after fluent.
        //       caused false after action.
        //       caused false after constant.
        //       etc.
        bodysstr << PRED_TIME << "(" << T1 << ")";
        }
    else
        assert(0);

    // Translate constraints with an empty if-part in a different way
    // (only if optimistic planning is not explicitely enforced).
    if ( !caused && !ifpart && !(FPlan & FRONTEND_PLANNING_OPTIMISTIC) )
        {
        // Generate rule:           $abs :- body.
        // instead of constraint:   :- body.
        headsstr << PRED_ABS;
        }
    
    headsstr << " :- " << bodysstr.str() << ".";
    
    if( FTraceLevel >= 1 )
        {
        cdebug << headsstr.str() << endl;
        }

    if( ! parseStringStream(headsstr) )
        {
        cerr << "Error while translating rule:" << endl
             << "   " << origRule << endl;
        SafeExit(1);
        }

    }

// Local Variables:
// c-file-style: "dl"
// End:
