/////////////////////////////////////////////////////////////////////////////
// plan.h

#ifndef PLAN_H
#define PLAN_H

#include <queue>

//////////////////////////////////////////////////////////////////////////////
// Global variables and functions

extern PREDICATE_NAMES fluents;
extern PREDICATE_NAMES actions;
// Maximum arity of declared actions:
extern unsigned maxActionArity;
// Minimum/maximum number of actions allowed per time:
extern unsigned minActions;
extern unsigned maxActions;

// Variable maxPlanLength contains the planlength
// optionally given as a commanline arument.
extern int maxPlanLength;
// Variable maxPlanLengthParsed contains the planlength
// optionally given in a planning goal query.
extern int maxPlanLengthParsed;
// Variable FPcheck indicatess, which sec.check. method should be used. 
extern unsigned FPcheck;
extern int FPsoundcheck;
extern int FPcompletecheck;

class KRULE;
typedef vector<KRULE> KRULES;
class KLITERAL;
typedef vector<KLITERAL> KLITERALS;
extern KLITERALS *Goal;
// Variable krules contains the K-rules collected during parsing
extern KRULES krules;
extern KRULES declarations;

// Flags indicating whether action costs (resp. cost levels) are used
extern bool costsUsed;
extern bool costLevelsUsed;

// Variable planCacheHit contains number of plan cache hits
// in case of secure planning
extern unsigned planCacheHit;
extern unsigned planCacheSize;
extern unsigned planCheckCounter;  

void PlanCallbackAfterGrounding();
void PlanCallbackAfterParsing();

class MODEL_GENERATOR;

bool PlanCallbackCheck(MODEL_GENERATOR*, const INTERPRET*, const GINTERPRET*);
bool PlanCallback(MODEL_GENERATOR*, const INTERPRET*, const GINTERPRET*);

/////////////////////////////////////////////////////////////////////////////
// Prefixes for aux. predicates inserted during translation
#define PRED_ACTIONTIME "$actiontime"
#define PRED_TIME "$time"
#define PRED_GOAL "$goal"
#define PRED_ABS  "$abs"
#define PRED_NEXTTIME "$nexttime"
#define PRED_ACT "$proj_action"
#define COST_PREFIX "$costs_"
#define LEGAL_PREFIX "$legal_"
#define EXEC_PREFIX "$exec_"
/////////////////////////////////////////////////////////////////////////////
// Special variables for T and T1
#define VAR_T "T$"
#define VAR_T1 "T$1"
#define TIME_PREFIX "$t"
// Default PlanCacheSize:
#define PLANCACHE 10000


/////////////////////////////////////////////////////////////////////////////
// KPARAMS is a class for Parameters of KATOM and KLITERAL:
// Class KPARAMS is a vector of strings
class KPARAMS : public vector<string> 
    {
public:
    KPARAMS(const char *s1, const char *s2=0, const char *s3=0)
        : vector<string>()
        {
        push_back(s1);
        if(s2) push_back(s2);
        if(s3) push_back(s3);
        }
    
    KPARAMS() 
        : vector<string>()
        {}
    };

inline ostream& operator<< (ostream &out, const KPARAMS &terms)
    {
    print_list(out,terms,",");
    return out;
    }

/////////////////////////////////////////////////////////////////////////////
// An own class for K atoms, which doesn't use the global stuff
//
class KATOM 
    {
    bool isTrueNeg;
    char *name;
    KPARAMS *params;
    
public:
    KATOM(const bool trueNeg, const char *name2, const KPARAMS *params2)
        : isTrueNeg(trueNeg)
        {
        assert( name2);
        name = new char[strlen(name2)+1];
        strcpy(name,name2);
        params = params2 ? new KPARAMS(*params2) : 0;
        }
   
    KATOM(const KATOM &c)
        : isTrueNeg(c.isTrueNeg) 
	{
        name = new char[strlen(c.name)+1];
        strcpy(name,c.name);
        params = c.params ? new KPARAMS(*c.params) : 0;
        }

    KATOM(const KATOM &c, const KPARAMS *params2)
        : isTrueNeg(c.isTrueNeg)
        {
        name = new char[strlen(c.name)+1];
        strcpy(name,c.name);
        params = params2 ? new KPARAMS(*params2) : 0;
        }
        
    ~KATOM()
	{
        if (name) delete [] name;
        if (params) delete params;
        }

    void operator= (const KATOM&)
            {
            // Forbid assignment.
            assert(0);
            }
    
    void push_back_param(const char *id)
        {
        if(!params)
            params = new KPARAMS(id);
        else
            params->push_back(id);
        }

    void setTrueNegative(bool neg)
	{
	isTrueNeg = neg;
	}
   
    bool isTrueNegative() const
	{
	return isTrueNeg;
	}
   
    const char* getName() const
	{
        return name; 
	}

    const KPARAMS* getParams() const
	{
        return params;
	}

    unsigned getArity() const
        {
	return params ? params->size() : 0;
        }
    };

inline bool operator== (const KATOM &a, const KATOM &b)
    {
    return strcmp(a.getName(), b.getName()) == 0
           && a.isTrueNegative() == b.isTrueNegative()
           && a.getParams() == b.getParams()
           ;
    }

inline ostream& operator<< (ostream &out, const KATOM &katom)
    {
    if( katom.isTrueNegative() )
        out << "-";

    out << katom.getName();

    if( katom.getParams()
        && katom.getParams()->begin() != katom.getParams()->end() )
        out << "(" << *(katom.getParams()) << ")";

    return out;
    }

typedef vector<KATOM> KATOMS;

class KLITERAL 
    {
    bool isNeg;
    KATOM katom;

public:
    KLITERAL(const bool neg, const KATOM &at) 
        : isNeg(neg), katom(at) {}

    KLITERAL(const KLITERAL &lit) 
        : isNeg(lit.isNeg), katom(lit.katom) {}

    KLITERAL(const KLITERAL &lit, const KPARAMS *params2)
        : isNeg(lit.isNeg), katom(lit.katom, params2) {}

    bool isNegative() const
        {
        return isNeg;
	}

    bool isTrueNegative() const
        {
        return katom.isTrueNegative();
        }
   
    const char* getName() const
        {
        return katom.getName();
        }

    const KPARAMS* getParams() const
        {
        return katom.getParams();
	}

    unsigned getArity() const
        {
        return katom.getArity();
        }
   
    // FIXME: Is it a good idea to return a pointer?
    //        Discuss with Gerald!
    const KATOM *getKatom() const
        {
        return &katom;
        }

    void setNegative(bool neg)
        {
        isNeg = neg;
        }

    void setTrueNegative(bool neg)
        {
        katom.setTrueNegative(neg);
        }

    void push_back_param(const char *id) 
        {
        katom.push_back_param(id);
        }
    };

inline bool operator== (const KLITERAL &a, const KLITERAL &b)
    {
    return *(a.getKatom()) == *(b.getKatom())
           && a.isNegative() == b.isNegative();
    }

inline ostream& operator<< (ostream &out, const KLITERAL &kliteral)
    {
    if( kliteral.isNegative() )
        out << "not ";

    assert( kliteral.getKatom() );
    out << *(kliteral.getKatom());

    return out;
    }

inline ostream& operator<< (ostream &out, const KLITERALS &kliterals)
    {
    print_list(out,kliterals,",");
    return out;
    }

// Prototype:
inline ostream& operator<< (ostream &, const KRULE &);

class KRULE
    {
    bool init;
    bool guess;
    bool decl;
    string *cost;
    KATOM *caused;
    KLITERALS *ifpart;
    KLITERALS *afterpart;

    bool isInertiaRule() const
            {
            bool isInertia = false;
            if ( caused && ifpart && afterpart )
                {
                for( KLITERALS::const_iterator i = afterpart->begin();
                     i != afterpart->end();
                     i++ )
                    {
                    if(*caused == *(i->getKatom()) )
                        // after part found:
                        // This is a "preserving" rule
                        // of the form: caused f if X after f, Y.
                        {
                        // check, whether this is an inertia rule:
                        // of the form: caused f if not -f, X after f, Y.
                        for( KLITERALS::const_iterator j = ifpart->begin();
                             j != ifpart->end();
                             j++ )
                            {
                            if( j->isNegative() 
                                && caused->isTrueNegative() != j->isTrueNegative() 
                                && strcmp(j->getName(), 
                                          caused->getName() ) == 0 )
                                {
                                if( j->getParams() && caused->getParams() )
                                    {
                                    if( *(j->getParams()) == 
                                        *(caused->getParams()) )
                                        {
                                        isInertia = true;
                                        break;
                                        }
                                    }
                                else if( j->getParams() || caused->getParams() )
                                    {
                                    cerr << "Fluent name used with different arities." << endl;
                                    SafeExit(1);
                                    }
                                else
                                    {
                                    isInertia = true;
                                    break;
                                    }
                                }
                            }
                        break;
                        }
                    }
                }
            return isInertia;
            }


public:
    KRULE()
            {
            // empty constructor not allowed.
            assert(0);
            }

    KRULE( const KATOM *c,
           const KLITERALS *i,  
           const KLITERALS *a,
           bool ini,
           bool g,
           bool d = false,
           const char *co = 0) 
        : init(ini), guess(g), decl(d)
            {
            caused    = c ? new KATOM(*c) : 0;
            ifpart    = i ? new KLITERALS(*i) : 0;
            afterpart = a ? new KLITERALS(*a) : 0;
            cost      = co? new string(co) : 0;
            }

    ~KRULE()
            {
            if (caused) delete caused;
            if (ifpart) delete ifpart; 
            if (afterpart) delete afterpart;
            if (cost) delete cost;
            }
    
    KRULE(const KRULE &krule) : 
        init(krule.init), 
        guess(krule.guess),
        decl(krule.decl)
            {
            caused    = krule.caused ? new KATOM(*(krule.caused)):0;
            ifpart    = krule.ifpart ? new KLITERALS(*(krule.ifpart)):0;
            afterpart = krule.afterpart ? new KLITERALS(*(krule.afterpart)):0;
            cost      = krule.cost? new string(*(krule.cost)):0;
            }

    void operator= (const KRULE&)
            {
            // Forbid assignment.
            assert(0);
            }

    void translate();

    void save();

    void setCaused(const KATOM &c)
            {
            if (caused ) delete caused;
            caused = new KATOM(c);
            }

    void setGuess(bool g)
            {
            guess = g;
            }

    const KATOM* getCaused() const 
            { return caused; }

    const KLITERALS* getIf() const 
            { return ifpart; }

    const KLITERALS* getAfter() const 
            { return afterpart; }

    bool isGuess() const
            { return guess; }

    bool isInit() const
            { return init; }

    bool isDecl() const
            { return decl; }

    const string* getCost() const
            { return cost; }
    };

inline ostream& operator<< (ostream &out, const KRULE &krule)
    {
    if( krule.getCaused() )
        {    
        if( krule.isDecl() )
            {
            if( ! (fluents.find(krule.getCaused()->getName())).second ) 
                out << "actions: ";
            else
                out << "fluents: ";
            }
        else if ( krule.isGuess() ) 
            {
            if( (fluents.find(krule.getCaused()->getName())).second ) 
                out << "total ";
            else
                out << "executable ";
            }
        else
            {
            out << "caused ";
            }
        out << *(krule.getCaused());
        }
    else
        out << "caused false";

    if( krule.getIf() )
        {
        if( krule.isDecl() )
            out << " requires ";
        else 
            out << " if ";
        out << *(krule.getIf());
        }

    if( krule.isDecl() && krule.getCost() )
        {
        out << " costs ";
        // Print cost levels only if levels are used anywhere in the
        // planning domain:
        if( costLevelsUsed )
           out << *(krule.getCost());
        else
           out << (krule.getCost()->substr(0,krule.getCost()->find(':')));
        }
    if( krule.getAfter() ) 
        {
        if( krule.isDecl() )
            out << " where ";
        else
            out << " after ";
        out << *(krule.getAfter());
        }

    out << ".";

    return out;
    }

class PlanCache
    {
    typedef map<GATOMSET, bool> PlanCacheMap;
    typedef deque<GATOMSET> PlanCacheList;

    PlanCacheMap map;
    PlanCacheList list;

public:
    const pair<bool, bool> lookup(GATOMSET set) const
        {
        PlanCacheMap::const_iterator p = map.find(set);
        if ( p != map.end() ) 
            return pair<bool, bool>(true,p->second);
        else
            return pair<bool, bool>(false,false);
        }
    
    void insert(GATOMSET plan, bool sec)        
        {
        map[plan] = sec;
        list.push_front(plan);
        
        // remove the last 10% of the cache, whenever the limit 
        // is reached, but remove at least one.
        if( list.size() > planCacheSize )
            {   
            for (unsigned i = 0; i <= planCacheSize/10; i++)
                { 
                map.erase(list.back());
                list.pop_back();
                }
            }
        }

    };

//////////////////////////////////////////////////////////////////////////////
// Global Functions

extern "C" FILE* planyyin; // Where LEX reads its input from 

// BUG-FIX: bison[1.24] fails to prototype its interfaces yylex() and yyerror().
extern "C" int planyylex();
extern "C" int planyyerror(const char*);

#endif

// Local Variables:
// mode: c++
// c-file-style: "dl"
// End:
