%{
//////////////////////////////////////////////////////////////////////////////
// plan_parser.y

#include <string>
#include <stdio.h>
#include "plan.h"

#define STRINGIFY(X) STRINGIFY1(X)
#define STRINGIFY1(X) #X

static enum { ACT, FLU } declType = ACT;

// Variable init states whether we are in an "initially" block 
// or in an "always" block.
static bool init = false;

// Is the keyword "time" is allowed at the current state of the parser?
// So far "time" is only allowed for idents in the where-clause of
// an action declaration cost statement.
static bool timeAllowed = false;

//////////////////////////////////////////////////////////////////////////////
static void processRule(
//
// This procedure handles translation of all static and dynamic rules.
//
// When the guess-flag is set it generates a guessing rule instead of a normal
// rule is used to generate the guessing rules for "executable A if B".

    const KATOM *caused_Part,
    const KLITERALS *if_Part,  
    const KLITERALS *after_Part,
    bool guess = false ) 
    {

    if( !caused_Part && !if_Part && !after_Part)
        {
        yyerror("Empty rules not allowed");
        parser_errors++;
        }

    // Dynamic rules are not allowed under initial scope!
    if ( init && after_Part)
        {
        yyerror("No dynamic rules allowed in initial section");
        parser_errors++;
        }

    if( caused_Part )
        if ( guess  )
	     {
#ifndef NDEBUG
	     // This is checked when parsing the EXEC and TOTAL statement.
	     // To be cautious check again here.
	     if( ! ( (actions.find(caused_Part->getName())).second ||
		     (fluents.find(caused_Part->getName())).second ) )
                 {
                 parser_errors++;
                 }
#endif
	     }
        // normal caused - rule: Check whether the head is a
        // declared fluent:
        else if( ! (fluents.find(caused_Part->getName())).second ) 
            {
            yyerror("Only fluents can be caused");
            parser_errors++;
            }

    krules.push_back( KRULE(caused_Part, if_Part, after_Part, 
                                  init, 
                                  guess) );
    
    LocalVariables.clear();
    }
 
static void processDecl(KATOM *declared, KLITERALS *requires, 
                        char* costs = 0, KLITERALS *whereclause = 0)
    {
    
    // No costs allowed for fluents!
    if ( costs && declType == FLU)
        {
        yyerror("Costs not allowed for fluents");
        parser_errors++;
        }
    
    // Check whether only variables are used in fluent/action!!!    
    if(declared->getParams())
        {
        for(KPARAMS::const_iterator i = declared->getParams()->begin();
            i != declared->getParams()->end(); i++)
            {
            if( !isupper(*(i->begin())) )
                {
                yyerror("Only variables allowed in action/fluent declaration");
                parser_errors++;
                }
            }
        }


    // Check, that not already defined.
    // and add fluent/action-name to global variable actions/fluents
    if( (fluents.find(declared->getName())).second ||
        (actions.find(declared->getName())).second )
        {
        yyerror("Action/Fluent name already in use");
        parser_errors++;
        }

    pair<PREDICATE_NAMES::index_t,bool> r;

    switch(declType)
        {
        case ACT:
            r = actions.add( declared->getName(),
                             // Inc arity to take timestamp into account:
                             declared->getArity() + 1, 
                             PREDICATE_NAMES::typeIDB );

            if (declared->getArity() > maxActionArity)
                maxActionArity = declared->getArity();
            break;
        case FLU: 
            r = fluents.add( declared->getName(), 
                             // Inc arity to take timestamp into account:
                             declared->getArity() + 1, 
                             PREDICATE_NAMES::typeIDB );
            break; 
        default:
            assert(0);
        }
    if( ! r.second )
        {
        assert(0);
        }

    declarations.push_back( KRULE(declared, 
                                  requires, 
                                  whereclause,
                                  // This is mainly for the C Frontend,
                                  // where we have to create a initial guessing
                                  // rule for fluents and a guess for all 
                                  // states for actions, 
                                  // as no executable statements are used
                                  // in C.
                                  (OptionFrontend == FRONTEND_PLANNING_C && 
                                   declType == ACT)? false : true,
                                  true,
                                  true,
                                  costs) );
    // FIXME: do we still need this
    LocalVariables.clear();
    }
//////////////////////////////////////////////////////////////////////////////

%}


%union
    {
    char*         string;
    KPARAMS*      params;
    KATOM*        katom;
    ATOM*         atom;
    KLITERAL*     kliteral;
    KLITERALS*    kliterals;
    DISJUNCTION*  head;
    LITERAL*      literal;
    unsigned	  integer;
    pair<KLITERALS*, KLITERALS*>* ifafter;
    }

%token <string>      ID NUM
%type  <string>      ident binop tertop weight weightpart term
%type  <params>      terms terms1 terms2 terms3
%type  <katom>       user_pred propositional_atom head builtin_pred
%type  <kliteral>    kliteral
%type  <kliterals>   kliterals whereclause requiresclause
%type  <integer>     number integer
%type  <ifafter>     ifafter

%token ERROR

%token NEWLINE
%token COMMA DOT COLON SEMICOLON PLUS ASTERISK EQUALS VERTICAL_BAR

%token BOOL_QUERY

%token CONS WEAK_CONS ANON_VAR
%token PARAM_OPEN PARAM_CLOSE
%token PRED_EQUAL PRED_UNEQUAL PRED_LESS PRED_GREATER
%token PRED_LESS_OR_EQ PRED_GREATER_OR_EQ
%token PRED_INT PRED_SUCC
%token TRUE_NOT NOT
%token MAXINTEGER

// Tokens for planning frontend
%token CAUSED IF AFTER ACTION FLUENT NONEX EXEC FORBIDDEN
%token INERTIAL DEFAULT ALWAYS REQUIRES COSTS WHERE NOCONC 
%token SECPLAN INITIALLY TOTAL GOAL CONST_FALSE
%token TIMESTAMP

%start input
%%
input     : kprogram {}
          | error { parser_errors++; }
          ;

kprogram : /* empty */ 
         | kprogram decls
         | kprogram rules
         ;

decls : ACTION { declType = ACT; } COLON decllist {}
      | FLUENT { declType = FLU; } COLON decllist {}
      ;

decllist : declaration {}
         | decllist declaration {}
         ;

declaration : user_pred requiresclause DOT         
                                    { 
                                    processDecl( $1, $2  );
                                    if( OptionFrontend == FRONTEND_PLANNING_C )
                                        {
                                        init = (declType==FLU)?true:false;
                                        processRule($1, $2, 0, true);
                                        }
                                    delete $1;
                                    delete $2;
                                    }
            | user_pred requiresclause COSTS weight 
              { timeAllowed = true; } whereclause { timeAllowed = false; }DOT 
                                    { 
                                    if( OptionFrontend == FRONTEND_PLANNING_C )
                                        {
                                        yyerror("Costs not allowed in"
                                                "Planning Language C");
                                        parser_errors++;
                                        }
                                    processDecl( $1, $2, $4, $6 ); 
                                    delete $1;
                                    delete $2;
                                    delete[] $4;
                                    delete $6;
                                    }
            ;

requiresclause : /* empty */ { $$ = 0; }
               | REQUIRES kliterals { $$ = $2; }
               ;

whereclause :   /* empty */ { $$ = 0; }
            |   WHERE kliterals { $$ = $2; }
            ;

weight      : weightpart 
                 {
                 costsUsed = true;
                 $$ = new char[NUM_LENGTH+3];
                 sprintf($$,"%s:1",$1);
                 }
            | weightpart COLON weightpart
                 { 
                 costsUsed = true;
                 costLevelsUsed = true;
                 $$ = new char[2*NUM_LENGTH+2];
                 sprintf($$,"%s:%s",$1,$3);
                 }
            ;
            
weightpart  : TIMESTAMP 
                 { 
                 $$ = new char[strlen(VAR_T1)+1];
                 strcpy( $$, VAR_T1 ); 
                 }
            | integer   
                 {
                 $$ = new char[NUM_LENGTH+1];
                 sprintf( $$, "%d", $1 ); 
                 }
            | ident     
                 {
                 // Non-integer constants are forbidden
                 if( !isupper($1[0]) )
                     {
                     yyerror( "Action costs or cost levels should be 'time', "
                              "a number, or a variable" );
                     parser_errors++;
                     }
                 $$ = $1; 
                 }
            ;

rules : INITIALLY { init = true;  } COLON ruleslist {}
      | ALWAYS    { init = false; } COLON ruleslist {}
      | GOAL COLON query {}
      ;

ruleslist : rule DOT {}
          | ruleslist rule DOT {} 
          ;

head : user_pred
     | CONST_FALSE { $$ = 0; }
     ;


ifafter : /* empty */ { $$ = new pair<KLITERALS*, KLITERALS*>(0,0); }
        | IF kliterals 
                          {
                          $$ = new pair<KLITERALS*, KLITERALS*>($2,0);
                          }
        | AFTER kliterals 
                          { 
                          $$ = new pair<KLITERALS*, KLITERALS*>(0,$2);
                          }
        | IF kliterals AFTER kliterals 
                          { 
                          $$ = new pair<KLITERALS*, KLITERALS*>($2,$4); 
                          }
        ;

rule : CAUSED head ifafter 
              {
              processRule( $2, $3->first, $3->second );
              delete $2;
              delete $3;
              }
     | user_pred 
              {
              processRule( $1, 0, 0 );
              delete $1;
              }
     | NONEX kliterals ifafter          
              { // Concatenate kliterals and if-part,
                // after part not allowed!

              if($3->second)
                  {
                  yyerror("No after part allowed in "
                          "nonexecutable statements");
                  parser_errors++;
                  }

              // Check whether first argument are only actions
              for( KLITERALS::const_iterator i = $2->begin();
                   i != $2->end(); i++)
                  if (i->isNegative())
                      {
                      yyerror("Negation as failure not allowed for A "
                              "in rule: nonexecutable A if B");
                      parser_errors++;
                      }
                  else if( ( ! (actions.find((*i).getName())).second) )
                      {
                      yyerror("Only actions allowed for A in rule: "
                              "nonexecutable A if B");
                      parser_errors++;
                      }

              KLITERALS *hkliterals = $3->first ? $3->first 
                                               : new KLITERALS();     

              for( KLITERALS::const_iterator i = $2->begin(); 
                   i != $2->end(); i++) 
                  {
                  hkliterals->push_back(*i);
                  }

              processRule( 0, 0, hkliterals );

              delete hkliterals;
              delete $2;
              delete $3;
              }
     | FORBIDDEN kliterals
              {
              processRule( 0, $2, 0 );
              delete $2;      
              }
     | FORBIDDEN kliterals AFTER kliterals
              {
              processRule( 0, $2, $4 );
              delete $2;      
              delete $4;      
              }
     | EXEC user_pred ifafter
              {
              if( OptionFrontend == FRONTEND_PLANNING_C )
                  {
                  yyerror("Executable not allowed in Frontend -FPc");
                  parser_errors++;
                  }
              else if( ( ! (actions.find($2->getName())).second) )
                  {
                  yyerror("Only actions allowed for A in rule: "
                          "executable A if B");
                  parser_errors++;
                  }
              
              if( $3->second )
                  {
                  yyerror("No after part allowed in "
                          "executable statements");
                  parser_errors++;
                  }

              bool hinit = init;
              init = false;
              processRule( $2, $3->first, 0, true);
              init = hinit; 
              delete $2;
              delete $3;
              }
     | INERTIAL kliterals ifafter
              {
              if(( $3->first || $3->second) 
                 && OptionFrontend == FRONTEND_PLANNING_C )
                  {
                  yyerror("inertial ... if... after ... not allowed "
                          "in Frontend -FPc");
                  parser_errors++;
                  }

              KLITERALS *if_part = $3->first?$3->first:new KLITERALS(), 
                        *after_part = $3->second?$3->second:new KLITERALS();

              for( KLITERALS::const_iterator i = $2->begin();
                   i != $2->end();
                   i++)
                  {
                  if (i->isNegative() )
                      {
                      yyerror("Negation as failure not allowed for A "
                              "in rule: inertial A if B after C");
                      parser_errors++;
                      }

                  
                  KLITERAL dneg_i(*i);
                  dneg_i.setNegative(true);
                  dneg_i.setTrueNegative(!dneg_i.isTrueNegative());

                  // For each kliteral i create a rule:
                  // caused i if not -.i after i.
                  // For FRONTEND_PLANNING_C:
                  // caused i if i after i.
                  
                  if ( OptionFrontend == FRONTEND_PLANNING_C )
                      {
                      if_part->push_back(*i);
                      }
                  else
                      {
                      if_part->push_back(dneg_i);
                      }
                  
                  after_part->push_back(*i);

                  processRule( i->getKatom(), if_part, after_part );
                  }
              delete if_part; 
              delete after_part;
              delete $2;
              delete $3;
              }
     | TOTAL user_pred ifafter
              {
              if( $2->isTrueNegative() )
                  {
                  yyerror("True negation not allowed for A in rule: "
                          "total A if B after C");
                  parser_errors++;
                  }
              if( OptionFrontend == FRONTEND_PLANNING_C )
                  {
                  yyerror("Total not allowed in Frontend -FPc");
                  parser_errors++;
                  }
              else if( ( ! (fluents.find($2->getName())).second) )
                  {
                  yyerror("Only fluents allowed for A in rule: "
                          "total A if B");
                  parser_errors++;
                  }
              processRule( $2, $3->first, $3->second, true);
              delete $2;
              delete $3;
              }
     | DEFAULT kliterals
              // Without IF or AFTER part a list of default fluents is allowed.
              {            
              for( KLITERALS::const_iterator i = $2->begin();
                   i != $2->end();
                   i++)
                  {
                  if (i->isNegative() )
                      {
                      yyerror("Negation as failure not allowed for A "
                              "in rule: default A");
                      parser_errors++;
                      }

                  // For each kliteral i create a rule:
                  // caused i if not -.i.
                  // For FRONTEND_PLANNING_C:
                  // caused i if i.
                  KLITERAL dneg_i(*i);
                  dneg_i.setTrueNegative(!dneg_i.isTrueNegative());
                  dneg_i.setNegative(!dneg_i.isNegative());
                  KLITERALS if_part;
                  if ( OptionFrontend == FRONTEND_PLANNING_C )
                      {
                      if_part.push_back(*i);
                      }
                  else
                      {
                      if_part.push_back(dneg_i);
                      }
                  processRule( i->getKatom(), &if_part, 0 );			  
                  } 
              delete $2;
              }            				   
     | NOCONC   
              { 
              // Set maxActions to 1, if minActions/maxActions has not been
              // already set to a value > 0 by commandline.
              if(!minActions && !maxActions)
                  maxActions = 1;
              }
     | SECPLAN 
              {
              // Disallow in frontend C
              if( OptionFrontend == FRONTEND_PLANNING_C )
                  {
                  yyerror("Secure Planning not allowed in Frontend -FPc");
                  parser_errors++;
                  }
               
              // Only switch to secure mode if optimistic mode has not
              // been requested explicitly.
              if( ! ( FPlan & FRONTEND_PLANNING_OPTIMISTIC ) )
                  {    
                  FPlan = static_cast<FPLAN>
                      (FPlan|FRONTEND_PLANNING_SECURE);
                  }
              }
     ;    
 
propositional_atom : ident
             {
             $$ = new KATOM(false, $1, 0);
             delete[] $1;
             }
     | TRUE_NOT ident
             {
             $$ = new KATOM(true, $2, 0);
             delete[] $2;
             }
     ;

integer	: number
             {
             $$ = $1;
             if( MaxInteger != -1  &&  (int)$$ > MaxInteger )
                 {
                 yyerror("number outside of given "
                         "integer range");
                 parser_errors++;
                 }
             }
        | MAXINTEGER	      
             {
             if( MaxInteger == -1 )
                 {
                 yyerror("#maxint used but no upper "
                         "integer bound (option -N) given");
                 parser_errors++;
                 }
             $$ = MaxInteger; 
             }
	;

number : NUM		      
             {
             char *err;
             $$ = strtoul(($1),&err,10);
             assert(*err==0);
             delete[] $1;

             if( static_cast<int>($$) > MaxIntegerSeen )
                 MaxIntegerSeen=$$;
             }
       ;

kliterals : /*empty*/ { $$=0; }
          | kliteral
             {                                  
             // "True" in the body is equal to an empty body
             if($1) 
                 {
                 $$ = new KLITERALS;
                 $$->push_back(*$1);
		    
                 delete $1;
                 }
             else
                 $$ = 0;
             }
          | kliterals COMMA kliteral 
             {
             // "True" in the body is equal to an empty body
             // theoretically the user could write something
             // caused X if true,true,true,Y.
             if( $1 )
                 $$ = $1;
             else if( $3 )
                 $$ = new KLITERALS;
             else 
                 $$ = 0;

             if($3) 
                 {
                 $$->push_back(*$3);
		    
                 delete $3;
                 }  
             }
          ;

kliteral: user_pred     
             {
             $$ = new KLITERAL( false, *$1 ); delete $1; 
             }
        | NOT user_pred 
             { 
             if ( OptionFrontend == FRONTEND_PLANNING_C ) 
                 { 
                 yyerror("Negation as failure not allowed in "
                         "Planning Language C");
                 parser_errors++;
                 }
             $$ = new KLITERAL( true,  *$2 ); delete $2; 
             }
        | builtin_pred  { $$ = new KLITERAL(false, *$1); delete $1; }
        ;

query : kliterals BOOL_QUERY PARAM_OPEN number PARAM_CLOSE  
             {
             if( Goal )
                 {
                 cerr << "Warning: goal: " << *($1) 
                      << "? ("
                      << ( maxPlanLength < 0 ? $4 
                           : maxPlanLength )
                      << ") replaces goal: " 
                      << *Goal << "? (" 
                      << ( maxPlanLength < 0 ? 
                           ( maxPlanLengthParsed < 0 ? 
                             0 : maxPlanLengthParsed )
                           : maxPlanLength )
                      << ")" << endl;
                 delete Goal; 
                 }
             
             // Commandline option overrides query-argument,
             // so defaultValue 0 of maxPlanLengthParsed is
             // only changed if no maxPlanLength has been
             // given as a commandline option:
             if(maxPlanLength < 0)
                 maxPlanLengthParsed = $4;


             Goal =  $1;
             }
      | kliterals BOOL_QUERY
             {
             if( Goal )
                 {
                 cerr << "Warning: goal: " << *($1) 
                      << "? ("
                      << ( maxPlanLength < 0 ? 0 : maxPlanLength ) 
                      << ") replaces goal:" 
                      << *Goal << "? (" 
                      << ( maxPlanLength < 0 ? 
                           ( maxPlanLengthParsed < 0 ? 
                             0 : maxPlanLengthParsed )
                           : maxPlanLength ) 
                      << ")" << endl;
                 delete Goal; 
                 }
             
             // Reset maxPlanLengthParsed:
             maxPlanLengthParsed = -1;
             
             Goal = $1;
             }

      ;

user_pred : propositional_atom           
          | ident PARAM_OPEN terms PARAM_CLOSE
             {
             $$ = new KATOM(false,$1,$3);
             delete[] $1;
             delete $3;
             }
          | TRUE_NOT ident PARAM_OPEN terms PARAM_CLOSE
             {
             $$ = new KATOM(true,$2,$4);
             delete[] $2;
             delete $4;
             }
          ;

builtin_pred : PRED_INT terms1
                 {
                 $$ = new KATOM(false, "#int", $2);
                 delete $2;
                 }
             | PRED_SUCC terms2
                 {
                 $$ = new KATOM(false, "#succ",$2);
                 delete $2;
                 }
             | binop terms2
                 {
                 $$ = new KATOM(false,$1,$2); 
                 delete[] $1;
                 delete $2;
                 }
             | term binop term
                 {
                 KPARAMS terms($1,$3);
                 
                 $$ = new KATOM(false,$2,&terms);
                 delete[] $1;
                 delete[] $2;                 
                 delete[] $3;
                 }
             | tertop terms3
                 {
                 $$ = new KATOM(false,$1,$2);
                 delete[] $1;
                 delete   $2;
                 }
             | term EQUALS term tertop term
                 {
                 KPARAMS terms($3,$5,$1);
                 
                 $$ = new KATOM(false,$4,&terms);
                 delete[] $1;
                 delete[] $3;
                 delete[] $4;
                 delete[] $5;
                 }
             ;             

terms    : term             
             {
             $$ = new KPARAMS;

             $$->push_back($1);
             delete[] $1;
             }
          | terms COMMA term
             {
             $$ = $1;

             $$->push_back($3);
             delete[] $3;
             }
          ;

terms1   : PARAM_OPEN term PARAM_CLOSE
             {
             $$ = new KPARAMS($2);
             delete[] $2;
             }
          ;

binop     : PRED_EQUAL         { $$ = new char[2]; strcpy($$, "=="); }
          | PRED_UNEQUAL       { $$ = new char[2]; strcpy($$, "!="); }
          | PRED_LESS          { $$ = new char[2]; strcpy($$, "<" ); }
          | PRED_GREATER       { $$ = new char[2]; strcpy($$, ">" ); }
	  | PRED_LESS_OR_EQ    { $$ = new char[2]; strcpy($$, "<="); }
	  | PRED_GREATER_OR_EQ { $$ = new char[2]; strcpy($$, ">="); }
          ;

terms2   : PARAM_OPEN term COMMA term PARAM_CLOSE
             {
             $$ = new KPARAMS($2,$4);
             delete[] $2;
             delete[] $4;
             }
          ;

tertop    : PLUS     { $$ = new char[2]; strcpy($$,"+"); }	      
          | ASTERISK { $$ = new char[2]; strcpy($$,"*"); }
	  ;

terms3	  : PARAM_OPEN term COMMA term COMMA term PARAM_CLOSE
             {
             $$ = new KPARAMS($2,$4,$6);
             delete[] $2;
             delete[] $4;
             delete[] $6;
             }
          ;

ident	  : ID	      
             {
             $$ = $1;
             if (strlen($$) > IDENT_LENGTH)
                 {
                 $$[IDENT_LENGTH]=0;
                 yyerror("identifier too long, truncated "
                         "to " STRINGIFY(IDENT_LENGTH)
                         " characters");
                 parser_errors++;
                 }
             }
          ;

term     : ident
             {
             $$ = $1;
             }
          | integer
             {
             $$ = new char[NUM_LENGTH+1];
             sprintf( $$, "%d", $1 );
             }
	  | ANON_VAR
             {
             $$ = new char[2];
             strcpy( $$, "_" );
             }
          | TIMESTAMP
             { 
             if(!timeAllowed)
                 {
                 yyerror("Keyword 'time' not allowed here");
                 parser_errors++;
                 }
             $$ = new char[strlen(VAR_T1)+1];
             strcpy( $$, VAR_T1 ); 
             }
          ;

%%

// Local Variables:
// mode: c++
// c-file-style: "dl"
// End:
