//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: DegreeOfFreedomSetBuilder.cpp,v 1.7 2006/03/04 18:36:50 delpinux Exp $

#include <DegreeOfFreedomSetBuilder.hpp>
#include <Mesh.hpp>

#include <DegreeOfFreedomSetManager.hpp>

#include <Structured3DMesh.hpp>
#include <MeshOfTetrahedra.hpp>

#include <ErrorHandler.hpp>

template <typename MeshType>
void DegreeOfFreedomSetBuilder::
__buildFictitious(const size_t& numberOfVariables,
		  const MeshType& mesh,
		  const Domain& domain)
{
  Vector<bool> dofVertices;

  dofVertices.resize(mesh.numberOfVertices());
  dofVertices = false;

  for (size_t i=0; i<mesh.numberOfVertices(); ++i) {
    dofVertices[i] = domain.inside(mesh.vertex(i));
  }

  Vector<bool> usedVertex(mesh.numberOfVertices());
  usedVertex = false;

  typedef typename MeshType::CellType CellType;

  for (size_t i=0; i<mesh.numberOfCells();++i) {
    unsigned numberIn = 0;
    const CellType& k = mesh.cell(i);

    // Computation of the caracteristic function
    for (unsigned j=0; j<CellType::NumberOfVertices; ++j) {
      numberIn += (dofVertices(mesh.vertexNumber(k(j))))?1:0;
    }

    // reduction of degrees of freedom
    if (numberIn != 0) {
      for (unsigned j=0; j<CellType::NumberOfVertices; ++j) {
	const size_t n = mesh.vertexNumber(k(j));
	usedVertex[n] = true;
      }
    }
  }

  // numbering correspondant vertices remapping
  std::map<size_t, size_t> newVerticesCorrespondance;
  for (size_t i=0; i<mesh.numberOfVertices(); ++i) {
    if (usedVertex[i]) {
      newVerticesCorrespondance[mesh.correspondance(i)] = 0;
    }
  }

  size_t numberOfDLVertices = 0;
  for (std::map<size_t, size_t>::iterator i=newVerticesCorrespondance.begin();
       i != newVerticesCorrespondance.end(); ++i) {
    i->second = numberOfDLVertices++;
  }

  ReferenceCounting<DegreeOfFreedomSet::Correspondance> pCorrespondance
    = new DegreeOfFreedomSet::Correspondance(mesh.numberOfVertices());

  DegreeOfFreedomSet::Correspondance& correspondance = *pCorrespondance;
  correspondance = -1;

  for (size_t i = 0; i<mesh.numberOfVertices(); ++i) {
    if (usedVertex[i]) {
      correspondance[i] = newVerticesCorrespondance[mesh.correspondance(i)];
    }
  }

  __degreeOfFreedomSet
    = new DegreeOfFreedomSet(numberOfVariables,
			     numberOfDLVertices,
			     __dofPositionSet,
			     pCorrespondance);
}

DegreeOfFreedomSetBuilder::
DegreeOfFreedomSetBuilder(const size_t& numberOfVariables,
			  const DiscretizationType& discretization,
			  const Mesh& mesh)
  : __dofPositionSet(DegreeOfFreedomSetManager::instance().getDOFPositionsSet(mesh,
									      discretization))
{
  typedef DegreeOfFreedomSet::Correspondance Correspondance;

  ReferenceCounting<Correspondance> pCorrespondance
    = new Correspondance(__dofPositionSet.number());
  Correspondance& correspondance = *pCorrespondance;

  const VerticesCorrespondance& vc
    = *(mesh.verticesCorrespondance());

  fferr(1) << __FILE__ << ':' << __LINE__ << ": NOT FINISHED (PERIODIC BC)\n";
  for (size_t i=0; i<__dofPositionSet.number(); ++i) {
    correspondance[i] = i;// = vc[i];
  }

  __degreeOfFreedomSet
    = new DegreeOfFreedomSet(numberOfVariables,
			     vc.numberOfCorrespondances(),
			     __dofPositionSet,
			     pCorrespondance);
}

DegreeOfFreedomSetBuilder::
DegreeOfFreedomSetBuilder(const size_t& numberOfVariables,
			  const DiscretizationType& discretizationType,
			  const Mesh& mesh,
			  const Domain& domain)
  : __dofPositionSet(DegreeOfFreedomSetManager::instance().getDOFPositionsSet(mesh,
									      discretizationType))
{
  this->__buildFictitious(numberOfVariables,
			  static_cast<const Structured3DMesh&>(mesh),
			  domain);
}


DegreeOfFreedomSetBuilder::
~DegreeOfFreedomSetBuilder()
{
  DegreeOfFreedomSetManager::instance().unsubscribe(__dofPositionSet);
}
