/*
 * Copyright (C) 2014-2024 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#include <QCoreApplication>
#include <QDateTime>
#include <QDir>
#include <QtTest/QtTest>

#include "src/datovka_shared/isds/error.h"
#include "src/datovka_shared/isds/types.h"
#include "src/datovka_shared/log/log.h"
#include "src/datovka_shared/settings/prefs.h"
#include "src/global.h"
#include "src/identifiers/account_id_db.h"
#include "src/identifiers/message_id.h"
#include "src/io/account_db.h"
#include "src/io/isds_sessions.h"
#include "src/io/message_db.h"
#include "src/isds/services_login.h"
#include "src/settings/accounts.h"
#include "src/settings/preferences.h"
#include "src/settings/prefs_defaults.h"
#include "src/settings/prefs_specific.h"
#include "src/worker/message_emitter.h"
#include "src/worker/task_download_message.h"
#include "src/worker/task_download_message_list.h"
#include "tests/helper_qt.h"
#include "tests/test_task_downloads.h"

class TestTaskDownloads : public QObject {
	Q_OBJECT

public:
	explicit TestTaskDownloads(const qint64 &receivedMsgId);

	~TestTaskDownloads(void);

private slots:
	void initTestCase(void);

	void cleanupTestCase(void);

	void downloadMessageList(void);

	void getDeliveryTime(void);

	void downloadMessage(void);

private:
	/*
	 * An application object must exist before any QObject.
	 * See https://doc.qt.io/qt-6/threads-qobject.html ,
	 * also see QTEST_GUILESS_MAIN().
	 */
	int m_argc;
	char **m_argv;
	QCoreApplication m_app;

	const bool m_testing; /*!< Testing account. */
	const enum MessageDbSet::Organisation m_organisation; /*!< Database organisation. */

	const QString m_connectionPrefix; /*!< SQL connection prefix. */

	const QString m_testPath; /*!< Test path. */
	QDir m_testDir;  /*!< Directory containing testing data. */

	const QString m_credFName; /*!< Credentials file name. */

	/* Log-in using user name and password. */
	LoginCredentials m_sender; /*!< Sender credentials. */
	LoginCredentials m_recipient; /*!< Recipient credentials. */

	MessageDbSet *m_recipientDbSet; /*!< Databases. */

	const qint64 &m_receivedMsgId; /*!< Identifier ow newly received message. */
	QDateTime m_deliveryTime; /*!< Message delivery time. */
};

TestTaskDownloads::TestTaskDownloads(const qint64 &receivedMsgId)
    : m_argc(0), m_argv(NULL), m_app(m_argc, m_argv),
    m_testing(true),
    m_organisation(MessageDbSet::DO_YEARLY),
    m_connectionPrefix(QLatin1String("GLOBALDBS")),
    m_testPath(QDir::currentPath() + QDir::separator() + QLatin1String("_test_dir")),
    m_testDir(m_testPath),
    m_credFName(QLatin1String(CREDENTIALS_FNAME)),
    m_sender(),
    m_recipient(),
    m_recipientDbSet(Q_NULLPTR),
    m_receivedMsgId(receivedMsgId),
    m_deliveryTime()
{
}

TestTaskDownloads::~TestTaskDownloads(void)
{
	/* Just in case. */
	delete m_recipientDbSet; m_recipientDbSet = Q_NULLPTR;
}

void TestTaskDownloads::initTestCase(void)
{
	bool ret;
	bool onDisk = true;

	if (!LoginCredentials::isReadableFile(m_credFName)) {
		QSKIP("Missing login credentials file. Skipping remaining tests.");
	}

	QVERIFY(GlobInstcs::logPtr == Q_NULLPTR);
	GlobInstcs::logPtr = new (::std::nothrow) LogDevice;
	QVERIFY(GlobInstcs::logPtr != Q_NULLPTR);

	QVERIFY(GlobInstcs::msgProcEmitterPtr == Q_NULLPTR);
	GlobInstcs::msgProcEmitterPtr =
	    new (::std::nothrow) MessageProcessingEmitter;
	QVERIFY(GlobInstcs::msgProcEmitterPtr != Q_NULLPTR);

	QVERIFY(GlobInstcs::iniPrefsPtr == Q_NULLPTR);
	GlobInstcs::iniPrefsPtr = new (::std::nothrow) INIPreferences;
	QVERIFY(GlobInstcs::iniPrefsPtr != Q_NULLPTR);

	/* Set configuration subdirectory to some value. */
	GlobInstcs::iniPrefsPtr->confSubdir = QLatin1String(".datovka_test");

	QVERIFY(GlobInstcs::prefsPtr == Q_NULLPTR);
	GlobInstcs::prefsPtr = new (::std::nothrow) Prefs;
	QVERIFY(GlobInstcs::prefsPtr != Q_NULLPTR);
	PrefsDefaults::setDefaults(*GlobInstcs::prefsPtr);

	/* Create empty working directory. */
	m_testDir.removeRecursively();
	QVERIFY(!m_testDir.exists());
	m_testDir.mkpath(".");
	QVERIFY(m_testDir.exists());

	/* Load credentials. */
	ret = m_sender.loadLoginCredentials(m_credFName, 1);
	if (!ret) {
		QSKIP("Failed to load login credentials. Skipping remaining tests.");
	}
	QVERIFY(ret);
	QVERIFY(!m_sender.userName.isEmpty());
	QVERIFY(!m_sender.pwd.isEmpty());

	ret = m_recipient.loadLoginCredentials(m_credFName, 2);
	if (!ret) {
		QSKIP("Failed to load login credentials. Skipping remaining tests.");
	}
	QVERIFY(ret);
	QVERIFY(!m_recipient.userName.isEmpty());
	QVERIFY(!m_recipient.pwd.isEmpty());

	/* Access message database. */
	m_recipientDbSet = MessageDbSet::createNew(m_testPath, m_recipient.userName,
	    m_testing, m_organisation, onDisk, m_connectionPrefix,
	    MessageDbSet::CM_CREATE_EMPTY_CURRENT);
	if (m_recipientDbSet == Q_NULLPTR) {
		QSKIP("Failed to open message database.");
	}
	QVERIFY(m_recipientDbSet != Q_NULLPTR);

	/*
	 * Create accounts database and open it. It is required by the task.
	 */
	QVERIFY(GlobInstcs::accntDbPtr == Q_NULLPTR);
	GlobInstcs::accntDbPtr = new (::std::nothrow) AccountDb("accountDb", false);
	if (GlobInstcs::accntDbPtr == Q_NULLPTR) {
		QSKIP("Cannot create accounts database.");
	}
	QVERIFY(GlobInstcs::accntDbPtr != Q_NULLPTR);
	ret = GlobInstcs::accntDbPtr->openDb(
	    m_testPath + QDir::separator() + "messages.shelf.db",
	    SQLiteDb::CREATE_MISSING);
	if (!ret) {
		QSKIP("Cannot open account database.");
	}
	QVERIFY(ret);

	/* Create ISDS session container. */
	QVERIFY(GlobInstcs::isdsSessionsPtr == Q_NULLPTR);
	GlobInstcs::isdsSessionsPtr = new (::std::nothrow) IsdsSessions;
	if (GlobInstcs::isdsSessionsPtr == Q_NULLPTR) {
		QSKIP("Cannot create session container.");
	}
	QVERIFY(GlobInstcs::isdsSessionsPtr != Q_NULLPTR);

	/* Log into ISDS. */
	Isds::Session *ctx = GlobInstcs::isdsSessionsPtr->session(
	    m_recipient.userName);
	if (!GlobInstcs::isdsSessionsPtr->holdsSession(m_recipient.userName)) {
		QVERIFY(ctx == Q_NULLPTR);
		ctx = GlobInstcs::isdsSessionsPtr->createCleanSession(
		    m_recipient.userName,
		    PrefsSpecific::isdsDownloadTimeoutMs(*GlobInstcs::prefsPtr));
	}
	if (ctx == Q_NULLPTR) {
		QSKIP("Cannot obtain communication context.");
	}
	QVERIFY(ctx != Q_NULLPTR);
	Isds::Error err = Isds::Login::loginUserName(ctx, m_recipient.userName,
	    m_recipient.pwd, m_testing);
	if (err.code() != Isds::Type::ERR_SUCCESS) {
		QSKIP("Error connection into ISDS.");
	}
	QVERIFY(err.code() == Isds::Type::ERR_SUCCESS);

	QVERIFY(GlobInstcs::acntMapPtr == Q_NULLPTR);
	GlobInstcs::acntMapPtr = new (::std::nothrow) AccountsMap(
	    AcntContainer::CT_REGULAR);
	QVERIFY(GlobInstcs::acntMapPtr != Q_NULLPTR);
}

void TestTaskDownloads::cleanupTestCase(void)
{
	delete m_recipientDbSet; m_recipientDbSet = Q_NULLPTR;

	delete GlobInstcs::acntMapPtr; GlobInstcs::acntMapPtr = Q_NULLPTR;

	/* Destroy ISDS session container. */
	delete GlobInstcs::isdsSessionsPtr; GlobInstcs::isdsSessionsPtr = Q_NULLPTR;

	/* Delete account database. */
	delete GlobInstcs::accntDbPtr; GlobInstcs::accntDbPtr = Q_NULLPTR;

	/* The configuration directory should be non-existent. */
	if (GlobInstcs::iniPrefsPtr != Q_NULLPTR) {
		QVERIFY(!QDir(GlobInstcs::iniPrefsPtr->confDir()).exists());
	}

	/* Delete testing directory. */
	m_testDir.removeRecursively();
	QVERIFY(!m_testDir.exists());

	delete GlobInstcs::prefsPtr; GlobInstcs::prefsPtr = Q_NULLPTR;

	delete GlobInstcs::iniPrefsPtr; GlobInstcs::iniPrefsPtr = Q_NULLPTR;

	delete GlobInstcs::msgProcEmitterPtr; GlobInstcs::msgProcEmitterPtr = Q_NULLPTR;

	delete GlobInstcs::logPtr; GlobInstcs::logPtr = Q_NULLPTR;
}

void TestTaskDownloads::downloadMessageList(void)
{
	TaskDownloadMessageList *task;
	const bool downloadWhole = false;
	const bool checkDownloadedList = false;

	QVERIFY(!m_recipient.userName.isEmpty());

	QVERIFY(m_recipientDbSet != Q_NULLPTR);

	QVERIFY(GlobInstcs::isdsSessionsPtr->isConnectedToIsds(m_recipient.userName));
	Isds::Session *ctx = GlobInstcs::isdsSessionsPtr->session(
	    m_recipient.userName);
	QVERIFY(ctx != Q_NULLPTR);
	QVERIFY(GlobInstcs::isdsSessionsPtr->isConnectedToIsds(m_recipient.userName));

	task = new (::std::nothrow) TaskDownloadMessageList(
	    AcntIdDb(m_recipient.userName, m_testing, m_recipientDbSet),
	    AcntId(), MSG_RECEIVED, downloadWhole, checkDownloadedList,
	    MESSAGE_LIST_LIMIT, Isds::Type::MFS_ANY);

	QVERIFY(task != Q_NULLPTR);
	task->setAutoDelete(false);

	task->run();

	QVERIFY(task->m_result == TaskDownloadMessageList::DL_SUCCESS);

	delete task; task = Q_NULLPTR;
}

void TestTaskDownloads::getDeliveryTime(void)
{
	if (m_receivedMsgId == 0) {
		QSKIP("No specific message to download.");
	}

	QVERIFY(!m_recipient.userName.isEmpty());

	QVERIFY(m_recipientDbSet != Q_NULLPTR);

	MsgId msgId(m_recipientDbSet->msgsMsgId(m_receivedMsgId));
	QVERIFY(msgId.isValid());

	m_deliveryTime = msgId.deliveryTime();
	QVERIFY(m_deliveryTime.isValid());
}

void TestTaskDownloads::downloadMessage(void)
{
	TaskDownloadMessage *task;

	if (m_receivedMsgId == 0 || !m_deliveryTime.isValid()) {
		QSKIP("No specific message to download or delivery time invalid.");
	}
	QVERIFY(m_receivedMsgId != 0);

	QVERIFY(!m_recipient.userName.isEmpty());

	QVERIFY(m_recipientDbSet != Q_NULLPTR);

	QVERIFY(GlobInstcs::isdsSessionsPtr->isConnectedToIsds(m_recipient.userName));
	Isds::Session *ctx = GlobInstcs::isdsSessionsPtr->session(
	    m_recipient.userName);
	QVERIFY(ctx != Q_NULLPTR);
	QVERIFY(GlobInstcs::isdsSessionsPtr->isConnectedToIsds(m_recipient.userName));

	/* Should fail, is a received message. */
	task = new (::std::nothrow) TaskDownloadMessage(
	    AcntIdDb(m_recipient.userName, m_testing, m_recipientDbSet),
	    MSG_SENT, MsgId(m_receivedMsgId, m_deliveryTime), false);

	QVERIFY(task != Q_NULLPTR);
	task->setAutoDelete(false);

	task->run();

	QVERIFY(task->m_result == TaskDownloadMessage::DM_ISDS_ERROR);

	delete task; task = Q_NULLPTR;

	/* Must succeed. */
	task = new (::std::nothrow) TaskDownloadMessage(
	    AcntIdDb(m_recipient.userName, m_testing, m_recipientDbSet),
	    MSG_RECEIVED, MsgId(m_receivedMsgId, m_deliveryTime), false);

	QVERIFY(task != Q_NULLPTR);
	task->setAutoDelete(false);

	task->run();

	QVERIFY(task->m_result == TaskDownloadMessage::DM_SUCCESS);

	delete task; task = Q_NULLPTR;
}

QObject *newTestTaskDownloads(const qint64 &receivedMsgId)
{
	return new (::std::nothrow) TestTaskDownloads(receivedMsgId);
}

//QTEST_MAIN(TestTaskDownloads)
#include "test_task_downloads.moc"
