/*
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.facebook.mojo;

import java.io.File;
import java.io.IOException;
import org.apache.commons.io.FileUtils;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;

/**
 * Deletes files generated by {@link FBThriftMojo}.
 *
 * @author jsailor
 */
@Mojo(name = "clean", defaultPhase = LifecyclePhase.CLEAN)
public class CleanMojo extends AbstractMojo {

  @Parameter(required = true, defaultValue = "${project.build.directory}/generated-sources")
  @SuppressWarnings("unused")
  private File generatedSourcesDirectory;

  /** Ignored; present for compatibility. */
  @Parameter(defaultValue = "false")
  @SuppressWarnings("unused")
  private boolean debug;

  private void deleteFile(File f) throws IOException {
    if (f.exists()) {
      boolean ok = f.delete();
      if (!ok) throw new IOException("Deletion failed: " + f);
    }

    // Files.delete(genChecksums.toPath());
  }

  private void deleteEmptyDirectory(File f) throws IOException {
    if (f.exists()) {
      String[] contents = f.list();
      if (contents == null) {
        throw new IOException("Deletion failed: could not read directory: " + f);
      }
      if (contents.length != 0) {
        return;
      }
      boolean ok = f.delete();
      if (!ok) {
        throw new IOException("Deletion failed: " + f);
      }
    }
  }

  @Override
  public void execute() throws MojoExecutionException, MojoFailureException {
    try {
      // Delete the checksums file first, so that we maintain the invariant
      // that a valid checksums file indicates valid generated content.
      File genChecksums = new File(generatedSourcesDirectory, "gen-java-checksums");
      if (genChecksums.exists()) {
        deleteFile(genChecksums);
      }

      // Then delete the code
      File genSource = new File(generatedSourcesDirectory, "gen-java");
      if (genSource.exists()) {
        FileUtils.deleteDirectory(genSource);
      }

      // If the generatedSourcesDirectory is empty, delete it. This lets
      // us handle the case where the directory is just for us reasonably
      // cleanly while not stomping on other plugins if we're sharing a
      // directory.
      deleteEmptyDirectory(generatedSourcesDirectory);
    } catch (IOException e) {
      throw new MojoExecutionException("thrift clean failed", e);
    }
  }
}
